/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.property.css2;

import org.trix.cuery.property.CascadableProperty;
import org.trix.cuery.property.PropertyTestCase;
import org.trix.cuery.property.SimpleProperty;
import org.trix.cuery.value.CSSPercentage;
import org.trix.cuery.value.CSSURI;
import org.trix.cuery.value.CSSValue;
import org.trix.cuery.value.Identifier;
import org.trix.cuery.value.PointLength;
import org.trix.cuery.value.ValueUtil;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: BackgroundDefinitionTest.java,v 1.0 2005/09/08 22:54:56 Teletha Exp $
 */
public class BackgroundDefinitionTest extends PropertyTestCase {

    /*
     * Test property for 'background'
     */
    public void testParse1() {
        CSSValue value = new Identifier("top", null);
        new Identifier("center", value);

        SimpleProperty property = new SimpleProperty();
        property.setProperty("background", value, false);

        CascadableProperty cascadable = new CascadableProperty();
        cascadable.addProperty(createDescription(property));

        assertEquals("50", cascadable.getValueAsString("background-horizontal-position"));
        assertEquals("0", cascadable.getValueAsString("background-vertical-position"));

        assertEquals("top center", ValueUtil.toString(cascadable.getValue("background")));
    }

    /*
     * Test property for 'background'
     */
    public void testParse2() {
        CSSValue value = new Identifier("top", null);
        new Identifier("center", value);

        SimpleProperty property = new SimpleProperty();
        property.setProperty("background", value, false);

        CascadableProperty cascadable = new CascadableProperty();
        cascadable.addProperty(createDescription(property));

        assertEquals("50", cascadable.getValueAsString("background-horizontal-position"));
        assertEquals("0", cascadable.getValueAsString("background-vertical-position"));

        assertEquals("top center", ValueUtil.toString(cascadable.getValue("background")));
    }

    /*
     * Test property for 'background'
     */
    public void testParse3() {
        CSSValue value = new Identifier("center", null);

        SimpleProperty property = new SimpleProperty();
        property.setProperty("background", value, false);

        CascadableProperty cascadable = new CascadableProperty();
        cascadable.addProperty(createDescription(property));

        assertEquals("50", cascadable.getValueAsString("background-horizontal-position"));
        assertEquals("50", cascadable.getValueAsString("background-vertical-position"));

        assertEquals("center", ValueUtil.toString(cascadable.getValue("background")));
    }

    /*
     * Test property for 'background-position'
     */
    public void testParse4() {
        CSSValue value = new CSSPercentage(0, null);

        SimpleProperty property = new SimpleProperty();
        property.setProperty("background", value, false);

        CascadableProperty cascadable = new CascadableProperty();
        cascadable.addProperty(createDescription(property));

        assertEquals("0", cascadable.getValueAsString("background-horizontal-position"));
        assertEquals("50", cascadable.getValueAsString("background-vertical-position"));

        // apply initial value
        assertEquals("scroll", cascadable.getValueAsString("background-attachment"));
        assertEquals("none", cascadable.getValueAsString("background-image"));
        assertEquals("repeat", cascadable.getValueAsString("background-repeat"));

        assertEquals("0%", ValueUtil.toString(cascadable.getValue("background")));
    }

    /*
     * Test property for 'background-position' with invalid value.
     */
    public void testParse5() {
        CSSValue value = new Identifier("center", null);
        new PointLength(2, value);

        SimpleProperty property = new SimpleProperty();
        property.setProperty("background", value, false);

        CascadableProperty cascadable = new CascadableProperty();
        cascadable.addProperty(createDescription(property));

        // apply initial value
        assertEquals("0", cascadable.getValueAsString("background-horizontal-position"));
        assertEquals("0", cascadable.getValueAsString("background-vertical-position"));
        assertEquals("scroll", cascadable.getValueAsString("background-attachment"));
        assertEquals("none", cascadable.getValueAsString("background-image"));
        assertEquals("repeat", cascadable.getValueAsString("background-repeat"));
    }

    /*
     * Test property for 'background-position'
     */
    public void testParse6() {
        CSSValue value = new Identifier("no-repeat", null);
        CSSValue value2 = new PointLength(2, value);
        new CSSURI("http://test.org/", value2);

        SimpleProperty property = new SimpleProperty();
        property.setProperty("background", value, false);

        CascadableProperty cascadable = new CascadableProperty();
        cascadable.addProperty(createDescription(property));

        assertEquals("2", cascadable.getValueAsString("background-horizontal-position"));
        assertEquals("50", cascadable.getValueAsString("background-vertical-position"));
        assertEquals("http://test.org/", cascadable.getValueAsString("background-image"));
        assertEquals("no-repeat", cascadable.getValueAsString("background-repeat"));

        assertEquals("no-repeat 2pt url(\"http://test.org/\")", ValueUtil.toString(cascadable.getValue("background")));
    }

}
