require 'umigame/xunit'
require 'umigame/creator'
require 'umigame/adder'
require 'umigame/sourcecode'
require 'umigame/method'

module Umigame

  class RubyUnit < XUnit
    def RubyUnit.version
      "Test::Unit-0.1.4(Ruby)"
    end
    
    def RubyUnit.usage
      print <<"EOB"

#{RubyUnit.version}(--lang ruby)
  support option: -a, -i, -m, -N, -P, -r, -s
EOB
    end
    
    def RubyUnit.reference
      print <<"EOB"
Assertion Quick Reference
  for #{RubyUnit.version}

flunk(message)
assert(boolean, message)
assert_block(message)
assert_equal(expected, actual, message)
assert_not_equal(expected, actual, message)
assert_raises(expected_exception_klass, message)
assert_nothing_raised(*args)
assert_throws(expected_symbol, message)
assert_nothing_throws(message)
assert_instance_of(klass, object, message)
assert_nil(object, message)
assert_not_nil(object, message)
assert_kind_of(klass, object, message)
assert_respond_to(object, method, message)
assert_match(regexp, string, message)
assert_does_not_match(regexp, string, message)
assert_same(expected, actual, message)
assert_not_same(expected, actual, message)
assert_operator(object1, operator, object2, message)
assert_in_delta(expected_float, actual_float, delta, message)
assert_send(send_array, message)
EOB
    end
  end
  
  class RubyUnitMethod < Method
    def initialize(class_name, method_name, statement)
      super(class_name, method_name, "# ", statement)
    end
    
    def name
      if @prefix
        "test" + (@name[0].chr == "_" ? "" : "_") + @name
      else
        @name
      end
    end
    
    def decl
     ret = []
      ret.push "def #{name}"
      ret += @statement.split("\n")
      ret.push "end"
      return prefix + ret.join("\n" + prefix) + "\n"
    end
    
    def suite
      return "#{prefix}  suite.add #{@classname}.new(\"#{name}\")\n"
    end
  end
  
  class RubyUnitSourceCode < SourceCode
    def add_method(methodname, prefix=false, statement=nil)
      if statement
        statement =~ /^(.*)end/m
        statement = $1
      else
        statement = "  flunk('not implement yet')"
      end
      method = RubyUnitMethod.new(@class_name, methodname, statement)
      super(method, prefix)
    end
    
    def print
      include = ""
      decl = ""
      suite = <<"EOB"
  def #{@class_name}.suite
    suite = Test::Unit::TestSuite.new
# CUPPA:suite=+
EOB
      @include.each do |file|
        include +="\nrequire '#{file}'"
      end
      @method.each do |method|
        decl += "\n" + method.decl
        suite += method.suite
      end
      suite += "# CUPPA:suite=-\n    return suite\n  end\n"
      class_end = @class_end.join
      return <<"EOB"
require 'test/unit'
# CUPPA:include=+#{include}
# CUPPA:include=-
#{@class_begin.join}# CUPPA:decl=+#{decl}
# CUPPA:decl=-

#{suite if has_suite?}#{class_end}
EOB
    end
  end
  
  class RubyUnitAdder < Adder
    def initialize
      super("# ", RubyUnitSourceCode.new)
    end
    
    protected
    
    def parse_include
      do_parse_include(/require '(.+)'/)
    end
    
    def parse_class_begin
      do_parse_class_begin(/class (.+) < Test::Unit::TestCase/)
    end
    
    def parse_method
      do_parse_method(/^(..)def (.+)/)
    end
    
    def parse_suite
      do_parse_suite(/def #{@code.class_name}.suite/, 
        /(#\s|\s\s)\s*suite.add #{@code.class_name}\.new\("(.+)"\)/)
    end
  end
  
  class RubyUnitCreator < Creator
    
    attr_reader :code
    
    def filename(name)
      name.downcase + ".rb"
    end
    
    def eval_argv
      # nothing to do
    end
    
    def write_skeleton
      code = RubyUnitSourceCode.new
      code.class_name = @skeleton
      @include.each do |includefile|
        code.add_include includefile
      end
      code.class_begin.push <<"EOB"

class #{@skeleton} < Test::Unit::TestCase
  private
  # your stuff...
  
  public
  def set_up
    # initialize
  end
  
  def tear_down
    # terminate
  end

EOB
      code.has_suite = !@prefix

      @testmethod.each do | testcase |
        code.add_new_method testcase
      end

      unless @noregist
        code.class_end.push <<"EOB"
  if $global_testsuite_registry
    $global_testsuite_registry.add #{@skeleton}.suite
  end
EOB
      end
      code.class_end.push "end"
      return code.print
    end
    
    def write_main
     return <<"EOF"
require 'getoptlong'
require 'test/unit/testsuite'

parser = GetoptLong.new
parser.set_options(
  ['--help', '-h', GetoptLong::NO_ARGUMENT],
  ['--console', '-c', GetoptLong::NO_ARGUMENT],
  ['--gtk', '-g', GetoptLong::NO_ARGUMENT],
  ['--fox', '-f', GetoptLong::NO_ARGUMENT]
  )
begin
  parser.each_option do | name, arg |
    eval "$OPT_" + name.sub(/^--/,'').gsub(/-/,'_').upcase + "= '" + arg + "'"
  end
rescue
  exit(1)
end

if $OPT_HELP then
  print <<"EOB"
--help,    -h           : display help
--console, -c           : Test TestRunner(default)
--fox,     -f           : FOX TestRunner
--gtk,     -g           : GTK TestRunner
EOB
  exit(0)
end

$global_testsuite_registry = Test::Unit::TestSuite.new

unless ARGV.empty?
  ARGV.each do |testcase|
    require testcase
  end
else
  Dir.glob("test*.rb").each do |testcase|
    require testcase
  end
end

if $OPT_GTK
  require 'test/unit/ui/gtk/testrunner'
  Test::Unit::UI::GTK::TestRunner.run $global_testsuite_registry
elsif $OPT_FOX
  require 'test/unit/ui/fox/testrunner'
  Test::Unit::UI::Fox::TestRunner.run $global_testsuite_registry
else
  require 'test/unit/ui/console/testrunner'
  Test::Unit::UI::Console::TestRunner.run $global_testsuite_registry
end
EOF
    end
  end
end
