/*
 * AbstractExecuter.java
 *
 * Copyright 2014 the curewallet.org.
 * http://www.curewallet.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.curewallet.nagesen.ececuter;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang3.StringUtils;
import org.curewallet.lib.api.CoinAPIException;
import org.curewallet.lib.api.SendFrom;
import org.curewallet.lib.api.SendToAddress;
import org.curewallet.lib.api.ValidateAddress;
import org.curewallet.lib.api.WalletLock;
import org.curewallet.lib.api.WalletPassphrase;
import org.curewallet.lib.jsonrpc.JsonRPCClient;
import org.curewallet.lib.jsonrpc.JsonRPCException;
import org.curewallet.nagesen.AddressData;
import org.curewallet.nagesen.NagesenException;
import org.curewallet.nagesen.amount.Amount;
import org.curewallet.nagesen.log.SendLog;
import org.curewallet.nagesen.validator.message.MessageValidator;
import org.curewallet.nagesen.validator.unique.UniqueValidator;
import org.curewallet.nagesen.wallet.WalletInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * 
 * @author Nezumi Kozo
 * @since 1.0
 */
public abstract class AbstractExecuter implements Executer {

//	private static org.apache.log4j.Logger log = org.apache.log4j.Logger.getLogger(AbstractExecuter.class);	
	private static Logger log = LoggerFactory.getLogger(AbstractExecuter.class);
	
	// 検証処理
	private UniqueValidator uniqueValidator = null;
	private MessageValidator messageValidator = null;

	private Amount amount = null;

	private WalletPassphrase walletPassphrase = null;
	private WalletLock walletLock = null;
	private SendFrom sendFrom = null;
	private SendToAddress sendToAddress = null;
	private ValidateAddress validateAddress = null;
	
	private WalletInfo walletInfo = null;
	
	private JsonRPCClient jsonRPCClient = null;
	
	/**
	 * 
	 * @since 1.0
	 */
	public AbstractExecuter() {
	}

	public int execute(int maxCount, long timeIntervalMillis) throws NagesenException {

		try {

			// 送金済みのリスト
			List<AddressData> dataList2 = new ArrayList<AddressData>();
			
			boolean owari = false;

			for (;;) {
				// データのリスト
				List<AddressData> dataList = getData();
				if (dataList == null) {
					break;
				}
				for (AddressData data : dataList) {
					
					AddressData data2 = new AddressData();
					BeanUtils.copyProperties(data2, data);

					JsonRPCClient jsonRPCClient = getJsonRPCClient();

					// 投げ銭対象であるかの検証をする
					// ウォレットアドレスの検証
					if ( validateAddress(jsonRPCClient, data2.getAddress()) == false) {
						continue;
					}
					// メッセージの検証
					if ( getMessageValidator().validate(data2.getMessage()) == false) {
						continue;
					}
					// ユニークの検証
					if ( getUniqueValidator().validate(dataList2, data2) == false) {
						continue;
					}
					log.debug("Send candidate: " + data2.getCoin() + " " + data2.getAddress());

					// 金額計算
					data2.setAmount(getAmount().calc(data2));

					
					// 送金処理
					try {
						sendFrom(jsonRPCClient, data2);
						SendLog.log(data2);
					} catch (CoinAPIException e) {
						log.error("Send error " + data2.getAddress(), e);
					}
//					log.info(data2.getCoin() + " " + data2.getAddress() + " " + data2.getAmount());

					// 送金済みデータ登録
					dataList2.add(data2);
					if (dataList2.size() >= maxCount) {
						owari = true;
						break;
					}
					
					if (isContinue(data2) == false) {
						owari = true;
						break;
					}
				}

				if (owari) {
					break;
				}
				
				try {
					Thread.sleep(timeIntervalMillis);
				} catch (InterruptedException e) {
				}

			}
			
		} catch (Exception e) {
			if (e instanceof NagesenException) {
				throw (NagesenException) e;
			}
			throw new NagesenException(e);
		} finally {
			if (getJsonRPCClient() != null) {
				try {
					getJsonRPCClient().close();
				} catch (JsonRPCException e) {
					log.error("JsonRPCClient close failed.", e);
				}
			}
		}

		return 0;
	}

	protected boolean validateAddress(JsonRPCClient jsonRPCClient, String address) {
		try {
			return getValidateAddress().call(jsonRPCClient, address).getIsvalid();
		} catch (Exception e) {
			log.error("ValidateAddress error " + address, e);
			return false;
		}
	}
	
	protected void sendFrom(JsonRPCClient jsonRPCClient, AddressData data) throws CoinAPIException {

		try {
			if (StringUtils.isEmpty(getWalletInfo().getPassphrase()) == false) {
				getWalletPassphrase().call(jsonRPCClient, getWalletInfo().getPassphrase(), 20);
			}
			if (StringUtils.isEmpty(getWalletInfo().getFromAccount())) {
				getSendToAddress().call(jsonRPCClient, data.getAddress(), data.getAmount());
			} else {
				getSendFrom().call(jsonRPCClient, getWalletInfo().getFromAccount(), data.getAddress(), data.getAmount());
			}
			
			data.setTimestamp(Calendar.getInstance());

		} finally {
			if (StringUtils.isEmpty(getWalletInfo().getPassphrase()) == false) {
				try {
					getWalletLock().call(jsonRPCClient);
				} catch (CoinAPIException e) {
					log.error("WalletLock error " + data.getAddress(), e);
				}
			}
		}
	}
	
	protected abstract List<AddressData> getData() throws NagesenException ;

	protected abstract boolean isContinue(AddressData data) throws NagesenException ;

	
	/**
	 * @return uniqueValidator
	 * @since 1.0
	 */
	public UniqueValidator getUniqueValidator() {
		return uniqueValidator;
	}

	/**
	 * @param uniqueValidator セットする uniqueValidator
	 * @since 1.0
	 */
	public void setUniqueValidator(UniqueValidator uniqueValidator) {
		this.uniqueValidator = uniqueValidator;
	}

	/**
	 * @return messageValidator
	 * @since 1.0
	 */
	public MessageValidator getMessageValidator() {
		return messageValidator;
	}

	/**
	 * @param messageValidator セットする messageValidator
	 * @since 1.0
	 */
	public void setMessageValidator(MessageValidator messageValidator) {
		this.messageValidator = messageValidator;
	}

	/**
	 * @return amount
	 * @since 1.0
	 */
	public Amount getAmount() {
		return amount;
	}

	/**
	 * @param amount セットする amount
	 * @since 1.0
	 */
	public void setAmount(Amount amount) {
		this.amount = amount;
	}

	/**
	 * @return walletPassphrase
	 * @since 1.0
	 */
	public WalletPassphrase getWalletPassphrase() {
		return walletPassphrase;
	}

	/**
	 * @param walletPassphrase セットする walletPassphrase
	 * @since 1.0
	 */
	public void setWalletPassphrase(WalletPassphrase walletPassphrase) {
		this.walletPassphrase = walletPassphrase;
	}

	/**
	 * @return walletLock
	 * @since 1.0
	 */
	public WalletLock getWalletLock() {
		return walletLock;
	}

	/**
	 * @param walletLock セットする walletLock
	 * @since 1.0
	 */
	public void setWalletLock(WalletLock walletLock) {
		this.walletLock = walletLock;
	}

	/**
	 * @return sendFrom
	 * @since 1.0
	 */
	public SendFrom getSendFrom() {
		return sendFrom;
	}

	/**
	 * @param sendFrom セットする sendFrom
	 * @since 1.0
	 */
	public void setSendFrom(SendFrom sendFrom) {
		this.sendFrom = sendFrom;
	}



	/**
	 * @return walletInfo
	 * @since 
	 */
	public WalletInfo getWalletInfo() {
		return walletInfo;
	}

	/**
	 * @param walletInfo セットする walletInfo
	 * @since 
	 */
	public void setWalletInfo(WalletInfo walletInfo) {
		this.walletInfo = walletInfo;
	}

	/**
	 * @return validateAddress
	 * @since 
	 */
	public ValidateAddress getValidateAddress() {
		return validateAddress;
	}

	/**
	 * @param validateAddress セットする validateAddress
	 * @since 
	 */
	public void setValidateAddress(ValidateAddress validateAddress) {
		this.validateAddress = validateAddress;
	}


	/**
	 * @return sendToAddress
	 * @since 
	 */
	public SendToAddress getSendToAddress() {
		return sendToAddress;
	}

	/**
	 * @param sendToAddress セットする sendToAddress
	 * @since 
	 */
	public void setSendToAddress(SendToAddress sendToAddress) {
		this.sendToAddress = sendToAddress;
	}

	/**
	 * @return jsonRPCClient
	 * @since 
	 */
	public JsonRPCClient getJsonRPCClient() {
		return jsonRPCClient;
	}

	/**
	 * @param jsonRPCClient セットする jsonRPCClient
	 * @since 
	 */
	public void setJsonRPCClient(JsonRPCClient jsonRPCClient) {
		this.jsonRPCClient = jsonRPCClient;
	}

	
	
}
