/******************************************************************************
 *
 * Copyright (c) 1999	TOSHIYUKI ARAI. ALL RIGHTS RESERVED. 
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 *  
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 *	ScrollBar.cpp
 *
 *****************************************************************************/

// SOL++2000

#include <sol\ScrollBar.h>

#define Max(a,b)    (((a) > (b)) ? (a) : (b))
#define Min(a,b)    (((a) < (b)) ? (a) : (b))

int ScrollBar::HORIZONTAL = (int)SB_HORZ;
int ScrollBar::VERTICAL   = (int)SB_VERT;

ScrollBar::ScrollBar(View* parent, const char* name, Args& args)
	:Primitive(parent, name, 
			args.set(XmNpredefined, TRUE)
			    .set(XmNclassName, "ScrollBar"))
{
	pageIncrement = 4;
	lineIncrement = 1;
	setValues(args);
}


Boolean ScrollBar::create(View* parent, const char* name, Args& args)
{
	Boolean rc = Primitive::create(parent, name, 
			args.set(XmNpredefined, TRUE)
			    .set(XmNclassName, "ScrollBar"));

	pageIncrement = 4;
	lineIncrement = 1;
	setValues(args);

	return rc;
}



int ScrollBar::moveThumb(Event& event)
{
	int pos  = getPos();
	int prev = pos;

	SCROLLINFO scInfo;
	int min, max;
	getScrollRange(&min, &max);
	int range = max - min;

	int request = LOWORD(event.getWParam());
	switch(request) {

	case SB_PAGEDOWN: pos += pageIncrement;
	case SB_LINEDOWN: pos = Min(range, pos + lineIncrement);
		break;

	case SB_PAGEUP:  pos -= pageIncrement;
	case SB_LINEUP:  pos = Max(0, pos - lineIncrement);
		break;

	case SB_TOP: pos = 0;
		break;

	case SB_BOTTOM:	pos = range;
		break;

	case SB_THUMBPOSITION:
	case SB_THUMBTRACK:
		scInfo.fMask = SIF_ALL;
		getScrollInfo(&scInfo);
		pos = scInfo.nTrackPos;
		break;

	default:
		break;
	}
	setPos(pos);
	return  prev - pos;
}


void ScrollBar::getValues(Args& args)
{
	Primitive::getValues(args);

	int num   = args.getCount();
	Arg* arg  = args.getArgList();
	int max, min;

	for (int i = 0; i<num; i++) {
		const char*  name = arg[i].name;
		ulong* val  = (ulong*)arg[i].value;
		if (name == XmNpageIncrement) {
			*val =  pageIncrement;	// 08/02 '94
			continue;
		}
		if (name == XmNlineIncrement) {
			*val =  lineIncrement;
			continue;
		}
		if (name == XmNmaximum) {
			getScrollRange(&min, &max);
			*val = max;
			continue;
		}
		if (name == XmNminimum) {
			getScrollRange(&min, &max);
			*val = min;
			continue;
		}
		if (name == XmNthumbPosition) {
			*val = (ulong)getPos();
			continue;
		}
	}
}


void ScrollBar::setValues(Args& args)
{
	Primitive::setValues(args);

	ulong val;
	if (args.get(XmNpageIncrement, &val))
		pageIncrement = (int)val;
	if (args.get(XmNlineIncrement, &val))
		lineIncrement = (int)val;

	int max, min;
	getScrollRange(&min, &max);
	if (args.get(XmNmaximum, &val))
		setScrollRange(min, (int)val);

	getScrollRange(&min, &max);
	if (args.get(XmNminimum, &val))
		setScrollRange((int)val, max);

	if (args.get(XmNthumbPosition, &val))
		setPos((int)val);
}


LRESULT ScrollBar::defaultProc(Event& event)
{
	return discardInput(event);
}


void ScrollBar::disable()
{
	if(isEditable()) {
		if(!getNewProc()) 
			replaceWindowProc();
	}
	else {
   		View::disable();
	}
}


void ScrollBar::enable()
{
	if(isEditable()) {
		restoreWindowProc();
	}
	else {
	 	View::enable();
	}
}


BOOL ScrollBar::getScrollInfo(SCROLLINFO* scInfo) 
{
	BOOL rc = FALSE;
	if(scInfo) {
		scInfo->cbSize = sizeof(SCROLLINFO);
		rc  = ::GetScrollInfo(getWindow(), SB_CTL, scInfo);
	}
	return rc;
}
	
int	ScrollBar::setScrollInfo(SCROLLINFO* scInfo) 
{
	int pos = 0;
	if(scInfo) {
		scInfo->cbSize = sizeof(SCROLLINFO);
		pos = ::SetScrollInfo(getWindow(), SB_CTL, scInfo, TRUE);
	}
	return pos;
}
