/*
 * Copyright 2006-2008 TSUBAKIMOTO Hiroya <zorac@4000do.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <windows.h>
#include <winnt.h>
#include <wininet.h>
#include <string>
#include <vector>
#include <fcntl.h>
#include <tchar.h>
#include <io.h>
#include "define.h"

#ifdef _UNICODE
typedef std::wstring string_t;
#else
typedef std::string string_t;
#endif

template<typename _Th>
inline _Th complete(_Th h)
{
  if (!h) throw GetLastError();
  return h;
}

struct c_str_t
{
  LPCTSTR c_str;
  c_str_t(LPCTSTR s) : c_str(s) {}
  c_str_t(const string_t& s) : c_str(s.c_str()) {}
};

/*
 * config_t - configuration
 */
class config_t
{
  string_t _path;

  struct _buffer_t
  {
    LPTSTR data;
    
    _buffer_t() : data(NULL) {}
    ~_buffer_t() { delete [] data; }
    operator LPTSTR() { return data; }
    LPTSTR operator()(size_t n)
    {
      delete [] data, data = NULL;
      if (n) data = new TCHAR[n];
      return data;
    }
  };

  string_t _get(LPCTSTR section, LPCTSTR key = NULL, LPCTSTR def = NULL) const
  {
    if (!def) def = TEXT("");
    _buffer_t buf;
    DWORD size = 0;
    for (DWORD n = 0; size + 2 >= n;) {
      n += 256;
      size = GetPrivateProfileString(section, key, def, buf(n), n, _path.c_str());
    }
    size_t i = (size && key ? _tcsspn(buf, TEXT(" \t")) : 0);
    return string_t(buf + i, size - i);
  }

  static string_t _fullpath(c_str_t path)
  {
    _buffer_t buf;
    LPTSTR fp;
    if (!path.c_str || !*path.c_str) path.c_str = TEXT(".");
    DWORD n = complete(GetFullPathName(path.c_str, MAX_PATH, buf(MAX_PATH), &fp));
    n < MAX_PATH || complete(GetFullPathName(path.c_str, n, buf(n), &fp));
    return buf.data;
  }

public:
  config_t() {}
  config_t(c_str_t path) : _path(_fullpath(path)) {}

  std::vector<string_t> operator()(c_str_t section) const
  {
    std::vector<string_t> result;
    string_t s = _get(section.c_str);
    for (string_t::size_type i = 0; i < s.size();) {
      string_t::size_type n = s.find(TCHAR(0), i);
      if (n == string_t::npos) n = s.size();
      result.push_back(s.substr(i, n - i));
      i = n + 1;
    }
    return result;
  }

  int operator()(c_str_t section, c_str_t key, int def) const
  { return GetPrivateProfileInt(section.c_str, key.c_str, def, _path.c_str()); }

  string_t operator()(c_str_t section, c_str_t key, c_str_t def = NULL) const
  { return _get(section.c_str, key.c_str, def.c_str); }

  string_t path(c_str_t section, c_str_t key, c_str_t def = NULL) const
  { return _fullpath(_get(section.c_str, key.c_str, def.c_str)); }

  bool exists(c_str_t section) const
  { return !_get(section.c_str).empty(); }
};

/*
 * console - console object as singleton
 */
class console
{
  static struct state_t
  {
    DWORD id;
    HWND hwnd;
    HANDLE close;

    state_t() : id(GetCurrentThreadId()), hwnd(NULL), close(NULL) {}
  } _state;

  static BOOL CALLBACK _handler(DWORD type)
  {
    switch (type) {
    case CTRL_LOGOFF_EVENT:
    case CTRL_SHUTDOWN_EVENT:
      _state.hwnd = NULL;
      break;
    }
    if (PostThreadMessage(_state.id, WM_QUIT, 0, 0)) {
      WaitForSingleObject(_state.close, 20 * 1000UL);
    }
    return FALSE;
  }

  static BOOL CALLBACK _find(HWND hwnd, LPARAM lParam)
  {
    DWORD pid;
    GetWindowThreadProcessId(hwnd, &pid);
    if (pid != lParam) return TRUE;
    _state.hwnd = hwnd;
    SetForegroundWindow(hwnd);
    return FALSE;
  }

  static void _stdio(FILE* fp, int fd, HANDLE h)
  {
    fclose(fp);
    int d = _open_osfhandle(long(h), O_TEXT | (fd ? O_WRONLY : O_RDONLY));
    _dup2(d, fd), _close(d);
    FILE* p = _fdopen(fd, fd ? "w" : "r");
    *fp = *p;
    setvbuf(fp, NULL, _IONBF, 0);
  }
  // Can not create instance.
  console();
public:
  static void open()
  {
    _state.close = complete(CreateEvent(NULL, TRUE, FALSE, NULL));
    complete(AllocConsole() && SetConsoleCtrlHandler(_handler, TRUE));
    EnumWindows(_find, LPARAM(GetCurrentProcessId()));
    _stdio(stdin, 0, GetStdHandle(STD_INPUT_HANDLE));
    _stdio(stdout, 1, GetStdHandle(STD_OUTPUT_HANDLE));
    _stdio(stderr, 2, GetStdHandle(STD_ERROR_HANDLE));
  }

  static void close()
  {
    SetEvent(_state.close);
    GenerateConsoleCtrlEvent(CTRL_C_EVENT, 0);
    FreeConsole();
  }

  static void show(bool show = true)
  {
    if (_state.hwnd) {
      ShowWindow(_state.hwnd, show ? SW_SHOW : SW_HIDE);
      if (show) SetActiveWindow(_state.hwnd);
    }
  }

  static bool visible()
  {
    return _state.hwnd && IsWindowVisible(_state.hwnd);
  }

  static bool idle()
  {
    for (;;) {
      MSG msg;
      switch (GetMessage(&msg, NULL, 0, 0)) {
      default:
	TranslateMessage(&msg);
	DispatchMessage(&msg);
	continue;
      case 0: break;
      case -1: throw GetLastError();
      }
      break;
    }
    return _state.hwnd;
  }
};
console::state_t console::_state;

/*
 * drive_t - virtual drive for root
 */
class drive_t
{
  string_t _dev, _path;

  BOOL _mount(DWORD flags = 0) const
  { return DefineDosDevice(flags, _dev.c_str(), _path.c_str()); }
  static string_t _avail()
  {
    // Find an available drive-letter backword from "w:".
    const DWORD exists = GetLogicalDrives();
    TCHAR dev[] = TEXT("w:");
    DWORD mask = 1 << (dev[0] - 'a');
    while (exists & mask) --dev[0], mask >>= 1;
    return dev;
  }
public:
  drive_t(const string_t& dev, const string_t& path)
    : _dev(dev.size() ? dev : _avail()), _path(path)
  {
    // Confirm that there isn't any drive with same letter.
    if (QueryDosDevice(_dev.c_str(), NULL, 0) ||
	GetLastError() != ERROR_FILE_NOT_FOUND) {
      throw DWORD(ERROR_INVALID_DRIVE);
    }
    complete(_mount());
    try {
      complete(SetCurrentDirectory((_dev + TEXT("\\")).c_str()));
    } catch (...) {
      _mount(DDD_REMOVE_DEFINITION | DDD_EXACT_MATCH_ON_REMOVE);
      throw;
    }
  }

  ~drive_t()
  {
    SetCurrentDirectory(_path.c_str());
    _mount(DDD_REMOVE_DEFINITION | DDD_EXACT_MATCH_ON_REMOVE);
  }
};

/*
 * task_t
 */
class task_t
{
  static config_t _config;
  std::vector<string_t> _section;
  mutable HINTERNET _inet;
  typedef std::pair<UINT_PTR, string_t> timer_t;
  static std::vector<timer_t> _timer;

  void _exec(c_str_t cmd, bool bg) const
  {
    STARTUPINFO si;
    PROCESS_INFORMATION pi;
    ZeroMemory(&si, sizeof(si));
    si.cb = sizeof(si);
    complete(CreateProcess(NULL, LPTSTR(cmd.c_str),
			   NULL, NULL, TRUE, 0, NULL, NULL, &si, &pi));
    CloseHandle(pi.hThread);
    (bg ? WaitForInputIdle : WaitForSingleObject)(pi.hProcess, INFINITE);
    CloseHandle(pi.hProcess);
  }

  void _fetch(c_str_t url) const
  {
    if (!_inet) {
      _inet = complete(InternetOpen(TEXT("sws"), INTERNET_OPEN_TYPE_PRECONFIG, NULL, NULL, 0));
    }
    for (int retry = 5; retry--; Sleep(2000UL)) {
      HINTERNET h = InternetOpenUrl(_inet, url.c_str, NULL, 0, INTERNET_FLAG_RELOAD, 0);
      if (h) {
	InternetCloseHandle(h);
	break;
      }
    }
  }

  void _open(c_str_t url) const
  {
    ShellExecute(NULL, TEXT("open"), url.c_str, NULL, NULL, SW_SHOWNORMAL);
  }

  static void CALLBACK _time(HWND, UINT, UINT_PTR id, DWORD)
  {
    try {
      std::vector<timer_t>::iterator p;
      for (p = _timer.begin(); p != _timer.end(); ++p) {
	if (p->first == id) {
	  task_t(p->second).execute();
	  return;
	}
      }
      KillTimer(NULL, id);
    } catch (...) {}
  }

  // Can not copy instance.
  task_t(const task_t&);
  task_t& operator=(const task_t&);
public:
  task_t() : _inet(NULL) {}
  task_t(const string_t& section) : _inet(NULL) { _section.push_back(section); }
  ~task_t() { _inet && InternetCloseHandle(_inet); }

  void execute()
  {
    try {
      std::vector<string_t> keys = _config(_section.back());
      std::vector<string_t>::iterator p;
      for (p = keys.begin(); p != keys.end(); ++p) {
	bool echo = p->at(0) != _T('@');
	echo && _tprintf(TEXT("[%s]"), p->c_str());
	execute(_config(_section.back(), *p));
	echo && _tprintf(TEXT("\n"));
      }
    } catch (...) {}
  }

  void execute(string_t cmd)
  {
    try {
      static const TCHAR WS[] = TEXT(" \t");
      string_t::size_type n = cmd.find_first_not_of(WS);
      if (n == string_t::npos) return;
      cmd = cmd.substr(n, cmd.find_last_not_of(WS) - n + 1);
      if (cmd[0] == _T('[') && cmd[cmd.size() - 1] == _T(']')) {
	cmd = cmd.substr(1, cmd.size() - 2);
	if (find(_section.begin(), _section.end(), cmd) == _section.end()) {
	  _section.push_back(cmd);
	  execute();
	  _section.pop_back();
	}
	return;
      }
      n = cmd.find_first_not_of(TEXT("&?"));
      if (n == string_t::npos) return;
      bool bg = false;
      bool fetch = false;
      for (string_t::size_type i = 0; i < n; ++i) {
	switch (cmd[i]) {
	case _T('&'): bg = true; break;
	case _T('?'): fetch = true; break;
	}
      }
      n = cmd.find_first_not_of(WS, n);
      if (n == string_t::npos) return;
      cmd = cmd.substr(n);
      static const TCHAR* const proto[] = {
	TEXT("http://"), TEXT("https://"), TEXT("ftp://"), NULL
      };
      for (int i = 0; proto[i]; ++i) {
	if (cmd.compare(0, lstrlen(proto[i]), proto[i]) == 0) {
	  fetch ? _fetch(cmd) : _open(cmd);
	  return;
	}
      }
      _exec(cmd, bg);
    } catch (DWORD err) {
      _ftprintf(stderr, TEXT(" Error=%lu"), err);
    } catch (...) {}
  }

  static void environment(c_str_t env)
  {
    std::vector<string_t> keys = _config(env);
    std::vector<string_t>::iterator p;
    for (p = keys.begin(); p != keys.end(); ++p) {
      string_t s = _config(env, *p);
      SetEnvironmentVariable(p->c_str(), s.c_str());
      _tprintf(TEXT("%s=%s\n"), p->c_str(), s.c_str());
    }
  }

  static void timer(const string_t& section, int time)
  {
    if (time > 0 && _config.exists(section)) {
      _timer.push_back(timer_t(0, section));
      _timer.back().first = SetTimer(NULL, 0, time, _time);
    }
  }

  static void config(const config_t& config) { _config = config; }
};
config_t task_t::_config;
std::vector<task_t::timer_t> task_t::_timer;

/*
 * menu_t - popup menu
 */
class menu_t
{
  HMENU _hmenu;
  std::vector<string_t> _cmds;

  void _execute(int cmd) const
  {
    switch (cmd) {
    case ID_MENU_EXIT: PostQuitMessage(0); break;
    case ID_MENU_CONSOLE: console::show(!console::visible()); break;
    default:
      if (cmd >= ID_MENU_USER && cmd < ID_MENU_USER + _cmds.size()) {
	task_t().execute(_cmds[cmd - ID_MENU_USER]);
      }
    }
  }

  // Can not copy instance.
  menu_t(const menu_t&);
  menu_t& operator=(const menu_t&);
public:
  menu_t(c_str_t menu)
    : _hmenu(NULL)
  {
    HMENU bar = complete(LoadMenu(GetModuleHandle(NULL), menu.c_str));
    HMENU h = GetSubMenu(bar, 0);
    if (h && !RemoveMenu(bar, 0, MF_BYPOSITION)) h = NULL;
    DestroyMenu(bar);
    _hmenu = complete(h);
#ifdef ID_MENU_DEFAULT
    SetMenuDefaultItem(_hmenu, ID_MENU_DEFAULT, FALSE);
#endif
  }

  ~menu_t() { DestroyMenu(_hmenu); }

  menu_t& load(const config_t& config)
  {
    static const TCHAR MENU[] = TEXT("menu");
    std::vector<string_t> menus = config(MENU);
    if (!menus.empty()) {
      int i = 0;
      std::vector<string_t>::iterator p;
      for (p = menus.begin(); p != menus.end(); ++p) {
	unsigned flag = MF_BYPOSITION;
	string_t cmd;
	if (p->size() == 1 && (*p)[0] == _T('-')) {
	  flag |= MF_SEPARATOR;
	} else {
	  cmd = config(MENU, *p);
	}
	if (cmd.empty()) flag |= MF_GRAYED;
	_cmds.push_back(cmd);
	complete(InsertMenu(_hmenu, i, flag, ID_MENU_USER + i, p->c_str()));
	++i;
      }
      complete(InsertMenu(_hmenu, i, MF_BYPOSITION | MF_SEPARATOR, 0, NULL));
    }
    return *this;
  }

  void action(HWND hwnd, LPARAM type) const
  {
    POINT pt;
    GetCursorPos(&pt);
    switch (type) {
    case WM_LBUTTONDBLCLK:
      SetForegroundWindow(hwnd);
      _execute(GetMenuDefaultItem(_hmenu, FALSE, GMDI_GOINTOPOPUPS));
      break;
    case WM_RBUTTONUP:
      SetForegroundWindow(hwnd);
      CheckMenuItem(_hmenu, ID_MENU_CONSOLE, MF_BYCOMMAND |
		    (console::visible() ? MF_CHECKED : MF_UNCHECKED));
      _execute(TrackPopupMenu(_hmenu, TPM_RIGHTBUTTON | TPM_RETURNCMD,
			      pt.x, pt.y, 0, hwnd, NULL));
      break;
    }
  }
};

/*
 * trayicon_t - tray icon
 */
class trayicon_t
{
  HICON _icon;
  const menu_t& _menu;
  HWND _hwnd;

  HWND _newWindow()
  {
    const static TCHAR id[] = TEXT("sws");
    HINSTANCE instance = GetModuleHandle(NULL);
    WNDCLASSEX wc;
    if (!GetClassInfoEx(instance, id, &wc)) {
      ZeroMemory(&wc, sizeof(wc));
      wc.cbSize = sizeof(wc);
      wc.lpfnWndProc = _dispatch;
      wc.hInstance = instance;
      wc.hbrBackground = HBRUSH(COLOR_WINDOW + 1);
      wc.lpszClassName = id;
      complete(RegisterClassEx(&wc));
    }
    return complete(CreateWindow(id, id, WS_OVERLAPPEDWINDOW,
				 0, 0, 1, 1, NULL, 0, instance, this));
  }

  void _releaseWindow() const
  {
    SetWindowLong(_hwnd, GWL_USERDATA, 0);
    DestroyWindow(_hwnd);
  }

  bool _newIcon() const
  {
    NOTIFYICONDATA ni;
    ZeroMemory(&ni, sizeof(ni));
    ni.cbSize = sizeof(ni);
    ni.hWnd = _hwnd;
    ni.uFlags = NIF_MESSAGE | NIF_ICON;
    ni.uCallbackMessage = WM_APP;
    ni.hIcon = _icon;
    while (!Shell_NotifyIcon(NIM_ADD, &ni)) {
      if (GetLastError() != ERROR_TIMEOUT) return false;
      if (Shell_NotifyIcon(NIM_MODIFY, &ni)) break;
      Sleep(1000);
    }
    return true;
  }

  void _releaseIcon() const
  {
    NOTIFYICONDATA ni;
    ZeroMemory(&ni, sizeof(ni));
    ni.cbSize = sizeof(ni);
    ni.hWnd = _hwnd;
    Shell_NotifyIcon(NIM_DELETE, &ni);
  }

  void _destroy()
  {
    _releaseIcon();
    SetWindowLong(_hwnd, GWL_USERDATA, 0);
    _hwnd = NULL;
    PostQuitMessage(0);
  }

  void _event(LPARAM type)
  {
    if (type > WM_MOUSEFIRST && type <= WM_MOUSELAST) {
      _menu.action(_hwnd, type);
    }
  }

  void _resetIcon() { if (!_newIcon()) console::show(); }

  static LRESULT CALLBACK _dispatch(HWND hwnd, UINT msg,
				    WPARAM wParam, LPARAM lParam)
  {
    static UINT tbc = 0;
    trayicon_t* wp;
#define D_I_S_P_A_T_C_H(func) do {			\
  wp = (trayicon_t*)GetWindowLong(hwnd, GWL_USERDATA);	\
  if (wp) wp->func;					\
} while (0)
    switch (msg) {
    case WM_CREATE:
      SetWindowLong(hwnd, GWL_USERDATA,
		    LONG(LPCREATESTRUCT(lParam)->lpCreateParams));
      tbc = RegisterWindowMessage(TEXT("TaskbarCreated"));
      return 0;
    case WM_DESTROY:
      D_I_S_P_A_T_C_H(_destroy());
      return 0;
    case WM_APP:
      D_I_S_P_A_T_C_H(_event(lParam));
      return 0;
    default:
      if (msg == tbc) D_I_S_P_A_T_C_H(_resetIcon());
      break;
    }
    return DefWindowProc(hwnd, msg, wParam, lParam);
  }
#undef D_I_S_P_A_T_C_H
public:
  trayicon_t(c_str_t name, const menu_t& menu)
    : _icon(complete(LoadIcon(GetModuleHandle(NULL), name.c_str))),
      _menu(menu), _hwnd(_newWindow())
  {
    if (!_newIcon()) {
      _releaseWindow();
      throw GetLastError();
    }
  }

  ~trayicon_t()
  {
    if (_hwnd) {
      _releaseIcon();
      _releaseWindow();
    }
  }
};

/*
 * WinMain - main function
 */
int WINAPI WinMain(HINSTANCE, HINSTANCE, LPSTR, int)
{
  static const TCHAR mutex[] = TEXT("mutex:FE0B973B-6332-4d7c-8AA9-EC20296258F5");
  if (!CreateMutex(NULL, TRUE, mutex) ||
      GetLastError() == ERROR_ALREADY_EXISTS) return 0;
  try {
    console::open();
    {
      static const TCHAR OPTION[] = TEXT("option");
      static const TCHAR CRON[] = TEXT("cron");
      config_t config(TEXT("sws.ini"));
      drive_t drive(config(OPTION, TEXT("drive")),
		    config.path(OPTION, TEXT("basedir")));
      menu_t menu(MAKEINTRESOURCE(1));
      trayicon_t icon(MAKEINTRESOURCE(1), menu.load(config));
      task_t::config(config);
      task_t::environment(TEXT("env"));
      task_t::timer(CRON, config(OPTION, CRON, 10) * 60000);
      _tprintf(TEXT("Boot:\n"));
      task_t(TEXT("boot")).execute();
      console::show(false);
      if (console::idle()) {
	console::show();
	_tprintf(TEXT("Shutdown:\n"));
	task_t(TEXT("shutdown")).execute();
      }
    }
    console::close();
    return 0;
  } catch (DWORD err) {
    _ftprintf(stderr, TEXT("Error=%lu\n"), err);
  } catch (...) {
    _ftprintf(stderr, TEXT("Unknown Error\n"));
  }
  console::show();
  Sleep(3 * 1000UL);
  return -1;
}
