/*
 *  Copyright (C) 2006  Takashi Kasuya <kasuya@sfc.keio.ac.jp>
 *
 * This library is free software; you can redistribute it and/or
 *@modify it under the terms of the GNU Lesser General Public
 *@License as published by the Free Software Foundation; either
 *@version 2.1 of the License, or (at your option) any later version.
 *@This library is distributed in the hope that it will be useful,
 *@but WITHOUT ANY WARRANTY; without even the implied warranty of
 *@MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *@Lesser General Public License for more details.
 *
 *@You should have received a copy of the GNU Lesser General Public
 *@License along with this library; if not, write to the Free Software
 *@Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package jp.ac.naka.ec.sip;

import javax.sip.RequestEvent;
import javax.sip.address.URI;
import javax.sip.header.ContentTypeHeader;
import javax.sip.header.EventHeader;
import javax.sip.header.ExpiresHeader;
import javax.sip.message.Request;
import javax.sip.message.Response;
import javax.xml.parsers.ParserConfigurationException;

import jp.ac.naka.ec.Main;
import jp.ac.naka.ec.db.TupleSpaceAdapter;
import jp.ac.naka.ec.entity.Entity;
import jp.ac.naka.ec.entity.EntityEvent;
import jp.ac.naka.ec.entity.EntityEvent.EventType;
import jp.ac.naka.ec.sip.pidf.PIDFData;
import jp.ac.naka.ec.sip.pidf.PIDFParser;
import jp.ac.naka.ec.sip.pidf.PIDFParserImpl;

import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;
import org.xml.sax.SAXException;

/**
 * 
 * @author Takashi Kasuya
 *
 */
class SubscribeDelegate extends MethodDelegate {
	public static int minExpires = 60;

	private final static String[] acceptPackages = { "presence" };

	private static Logger logger = Logger.getLogger(SubscribeDelegate.class);
	
	private PIDFParser parser;

	
	public SubscribeDelegate() throws ParserConfigurationException,
			SAXException {
		parser = PIDFParserImpl.getInatance();
		core = SipCore.getInstance();
		PropertyConfigurator.configure(Main.log4j_properties);
		dispatcher = TupleSpaceAdapter.getInstance();
	}

	/*
	 * (non-Javadoc)
	 * @see jp.ac.naka.ec.sip.MethodDelegate#forward(javax.sip.RequestEvent)
	 */
	public boolean forward(RequestEvent evt) {

		Request req = evt.getRequest();
		ExpiresHeader expires ;
		try {
		
			// Event, ExpiresHeader
			EventHeader event = (EventHeader) req.getHeader(EventHeader.NAME);
			if (!checkEventHeader(event)) {
				logger.warn("Bad event.");
				core.sendResponse(Response.BAD_EVENT, req);
				return false;
			}
			expires= (ExpiresHeader) req
					.getHeader(ExpiresHeader.NAME);

			if (!checkExpiresHeader(expires)) {
				logger.warn("Interval is too brief.");
				core.sendResponse(Response.INTERVAL_TOO_BRIEF, req);
				return false;
			}
		} catch (Exception e1) {
			logger.warn("Error while send a response.", e1);
			return false;
		}
		ContentTypeHeader type = (ContentTypeHeader) req
				.getHeader("Content-Type");

		if (type.getContentType().equals(PIDFData.CONTENT_TYPE)
				&& type.getContentSubType().equals(PIDFData.CONTENT_SUBTYPE)) {
			byte[] content = req.getRawContent();

			if (content == null) {
				throw new NullPointerException("Notify has no body!");
			}

			PIDFData data;
			try {
			//	String str = new String(content, "UTF-8");
				data = parser.parse(content);
				URI uri = data.getURI();
				
				if (uri.isSipURI()) {
					Entity[] entities = makeEntitiesFromPIDF(data);
					EntityEvent ee = new EntityEvent(entities);
					ee.setEventType(EventType.SUBSCRIBE);
					ee.setMessage(new String(content, "UTF-8"));
					core.sendResponse(Response.OK, evt, uri.toString());
					dispatcher.dispatchEvent(ee);
					
				} else {
					logger.warn("Unsupported URI type :"
							+ uri.toString());
					return false;
				}
				
			} catch (Exception e) {
				logger.warn("Error occures while processing PIDF :" , e);
				return false;
			}
		}
		else  {
			logger.warn("Invalid ContentType: " + type.toString());
			return false;
		} 
		return true;
	}
	
	
	/**
	 * 
	 * @param expires
	 * @return
	 */
	private boolean checkExpiresHeader(ExpiresHeader expires) {
		int expire = expires.getExpires();
		return (expire > minExpires) ? true : false;
	}

	/**
	 * 
	 * @param header
	 * @return
	 */
	private boolean checkEventHeader(EventHeader header) {
		String type = header.getEventType();
		for (String str : acceptPackages) {
			if (str.equals(type)) {
				return true;
			}
		}
		return false;
	}

}
