/*
 * Decompiled with CFR 0.152.
 */
package ow.messaging.tcp;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;
import ow.messaging.InetMessagingAddress;
import ow.messaging.Message;
import ow.messaging.MessageSender;
import ow.messaging.MessagingAddress;
import ow.messaging.Tag;
import ow.messaging.tcp.TCPMessageReceiver;
import ow.stat.MessagingReporter;
import ow.util.AlarmClock;
import ow.util.Timer;
import ow.util.concurrent.ExecutorBlockingMode;
import ow.util.concurrent.SingletonThreadPoolExecutors;

public class TCPMessageSender
implements MessageSender {
    private static final Logger logger = Logger.getLogger("messaging");
    private final TCPMessageReceiver receiver;

    protected TCPMessageSender(TCPMessageReceiver receiver) {
        this.receiver = receiver;
    }

    public void send(MessagingAddress dest, Message msg) throws IOException {
        this.adjustLoopbackAddress((InetMessagingAddress)dest);
        MessagingAddress selfAddress = this.receiver.getSelfAddress();
        if (dest.equals(selfAddress)) {
            this.receiver.processMessage(msg);
            this.receiver.postProcessMessage(msg);
            return;
        }
        InetSocketAddress sockAddr = ((InetMessagingAddress)dest).getInetSocketAddress();
        SocketChannel sock = null;
        int retryCount = 0;
        while (true) {
            try {
                sock = this.receiver.connPool.get(sockAddr);
            }
            catch (IOException e) {
                logger.log(Level.INFO, "Failed to connect: " + dest);
                MessagingReporter msgReporter = this.receiver.getMessagingReporter();
                if (msgReporter != null) {
                    msgReporter.notifyStatCollectorOfDeletedNode(msg.getSource(), dest, msg.getTag());
                }
                throw e;
            }
            try {
                this.send(sock, dest, msg);
            }
            catch (ClosedChannelException e) {
                if (retryCount <= 0) {
                    ++retryCount;
                    continue;
                }
                throw e;
            }
            break;
        }
        this.receiver.connPool.put(sockAddr, sock);
    }

    private void send(SocketChannel sock, MessagingAddress dest, Message msg) throws IOException {
        logger.log(Level.INFO, "send(" + dest + ", " + Tag.getNameByNumber(msg.getTag()) + ")");
        byte[] sig = this.receiver.provider.getMessageSignature();
        msg.setSignature(sig);
        try {
            ByteBuffer buf = Message.encode(sock, msg);
            MessagingReporter msgReporter = this.receiver.getMessagingReporter();
            if (msgReporter != null) {
                msgReporter.notifyStatCollectorOfMessageSent(dest, msg, buf.remaining());
            }
        }
        catch (IOException e) {
            logger.log(Level.WARNING, "Could not write a message.");
            MessagingReporter msgReporter = this.receiver.getMessagingReporter();
            if (msgReporter != null) {
                msgReporter.notifyStatCollectorOfDeletedNode(msg.getSource(), dest, msg.getTag());
            }
            throw e;
        }
    }

    public Message sendAndReceive(MessagingAddress dest, final Message msg) throws IOException {
        long start;
        long timeout;
        this.adjustLoopbackAddress((InetMessagingAddress)dest);
        Message ret = null;
        MessagingAddress selfAddress = this.receiver.getSelfAddress();
        if (dest.equals(selfAddress)) {
            ret = this.receiver.processMessage(msg);
            if (this.receiver.extMessageHandlerRegistered) {
                Runnable r = new Runnable(){

                    public void run() {
                        TCPMessageSender.this.receiver.postProcessMessage(msg);
                    }
                };
                if (this.receiver.config.getUseThreadPool()) {
                    SingletonThreadPoolExecutors.getThreadPool(ExecutorBlockingMode.NON_BLOCKING, false).submit(r);
                } else {
                    Thread t = new Thread(r);
                    t.setName("TCPMessageSender: post-processing thread");
                    t.setDaemon(false);
                    t.start();
                }
            }
            return ret;
        }
        InetSocketAddress sockAddr = ((InetMessagingAddress)dest).getInetSocketAddress();
        SocketChannel sock = null;
        int retryCount = 0;
        while (true) {
            int read;
            try {
                sock = this.receiver.connPool.get(sockAddr);
            }
            catch (IOException e) {
                logger.log(Level.INFO, "Failed to connect: " + dest);
                MessagingReporter msgReporter = this.receiver.getMessagingReporter();
                if (msgReporter != null) {
                    msgReporter.notifyStatCollectorOfDeletedNode(msg.getSource(), dest, msg.getTag());
                }
                throw e;
            }
            ByteBuffer buf = ByteBuffer.allocate(1024);
            sock.configureBlocking(false);
            while ((read = sock.read(buf)) > 0) {
                buf.clear();
                logger.log(Level.INFO, "Data have remained in a pooled stream: " + read);
            }
            sock.configureBlocking(true);
            timeout = this.receiver.provider.getTimeoutCalculator().calculateTimeout(dest);
            start = Timer.currentTimeMillis();
            try {
                this.send(sock, dest, msg);
            }
            catch (ClosedChannelException e) {
                if (retryCount <= 0) {
                    ++retryCount;
                    continue;
                }
                throw e;
            }
            break;
        }
        try {
            AlarmClock.setAlarm(timeout);
            ret = Message.decode(sock);
            AlarmClock.clearAlarm();
            this.receiver.connPool.put(sockAddr, sock);
            this.receiver.provider.getTimeoutCalculator().updateRTT(dest, (int)(Timer.currentTimeMillis() - start));
        }
        catch (Exception e) {
            Thread.interrupted();
            logger.log(Level.INFO, "Timeout: " + timeout + " msec.");
            MessagingReporter msgReporter = this.receiver.getMessagingReporter();
            if (msgReporter != null) {
                msgReporter.notifyStatCollectorOfDeletedNode(msg.getSource(), dest, msg.getTag());
            }
            throw new IOException("Timeout:" + timeout + " msec.");
        }
        return ret;
    }

    private void adjustLoopbackAddress(InetMessagingAddress dest) {
        if (dest.getInetAddress().isLoopbackAddress()) {
            dest.setInetAddress(((InetMessagingAddress)this.receiver.getSelfAddress()).getInetAddress());
            logger.log(Level.INFO, "destination is loopback address and adjusted to " + dest);
        }
    }
}

