<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.beans
 */
/**
 * @file Bean.php
 * @brief This interface is Bean.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: Bean.php 56 2007-10-01 09:13:43Z ishitoya $
 *
 * Abstract Bean.
 * __set and __get mediates $bean->setData() of $bean->getData()
 * on Java or other languge, you must implement
 * $bean->set("Data", $object) or $bean->get("Data") to mediate beans method.
 */

require_once "ficus/lang/Serializable.php";
require_once "ficus/net/URI.php";
require_once "ficus/exception/MethodNotFoundException.php";
require_once "ficus/exception/IllegalArgumentException.php";
require_once "ficus/exception/PropertyNotFoundException.php";
require_once "ficus/exception/NotReadyException.php";
require_once "ficus/beans/BeanComponentFactory.php";
require_once "ficus/beans/annotation/BeanAnnotationAccessor.php";

/**
 * @class Ficus_Bean
 */
abstract class Ficus_Bean implements Ficus_Serializable{
    const METHOD_SIGNATURE = '/^(set|get|addAll|appendAll|pushAll|add|append|delete|shift|push|pop|insert|numOf|count|clear|has|isEmpty|isNot|is|enable|disable)([a-zA-Z][a-zA-Z0-9_]*)/';
    const METHOD_SIGNATURE_SUFFIX = '/^([a-zA-Z][a-zA-Z0-9_]*?)(EqualsTo)/';
    const BEAN_METHOD_SIGNATURE = '/^get([a-zA-Z][a-zA-Z0-9_]*)Bean/';
    
    /**
     * serialize bean
     * @param $type string type of serializer
     * @return string serialized string
     */
    public function serialize(){
        $arguments = func_get_args();
        $type = array_shift($arguments);
        if(is_null($type)){
            $reflection = new ReflectionClass($this);
            $props = $reflection->getDefaultProperties();
            foreach($props as $key => $value){
                $props[$key] = $this->$key;
            }
            return serialize($props);
        }

       if($type instanceof Ficus_BeanSerializer){
            $serializer = $type;
        }else{
            $factory =
              Ficus_BeanComponentFactory::getComponent("BeanSerializerFactory");
            $serializer = $factory->create($type);
        }

        if(empty($arguments)){
            return $serializer->serialize($this);
        }else{
            return $serializer->serialize($this, $arguments);
        }
    }

    public function unserialize($serialized){
        $unserialized = unserialize($serialized);
        foreach($unserialized as $key => $value){
            $this->$key = $value;
        }
    }


    /**
     * deserialize bean
     * @param $type string type of deserializer
     */
    public function deserialize($data, $deserializer=null){
        if(($deserializer instanceof Ficus_BeanDeserializer) == false){
            $factory =
              Ficus_BeanComponentFactory::getComponent(
                      "BeanDeserializerFactory");
            $deserializer = $factory->create($data);
        }
        return $deserializer->deserialize($this, $data);
    }

    /**
     * create clone
     * @return Ficus_Bean cloned object
     */
    public function createClone(){
        $clone = clone $this;
        return $clone;
    }

    /**
     * create hash
     * @return string hash
     */
    public function createHash(){
        return sha1(serialize($this));
    }

    /**
     * overwrite data
     */
    public function overwrite($bean){
        Ficus_Assert::isInstanceOf($bean, "Ficus_Bean");

        $props = $this->beanProperties();
        foreach($props as $name){
            if($bean->has($name)){
                $value = $bean->get($name);
                if($value instanceof Ficus_Bean &&
                   $this->get($name) instanceof Ficus_Bean){
                        $this->get($name)->overwrite($value);
                }else{
                    $this->set($name, $value);
                }
            }
        }
        return $this;
    }
    
    /**
     * set value to specified property
     * @param $name string property name
     * @param $value mixed value to set
     */
    public function set($name, $value){
        if($this->has($name) == false){
            throw new Ficus_PropertyNotFoundException("property $name is not found");
        }
        $this->{$name} = $value;
    }
    
    /**
     * get value
     * @return mixed value
     */
    public function get($name, $index = null){
        if($this->has($name) == false){
            throw new Ficus_PropertyNotFoundException("property $name is not found");
        }
        $method = "get" . ucfirst($name);
        if(method_exists($this, $method)){
            return $this->{$method}();
        }
        if(is_null($index)){
            return $this->{$name};
        }else{
            return $this->{$name}[$index];
        }
    }

    /**
     * has property
     * @param $name string property name
     * @return boolean true if property exists
     */
    public function has($name, $index = null){
        if(is_null($index)){
            return property_exists($this, $name);
        }
        if(property_exists($this, $name)){
            $property = $this->{$name};
            if(isset($property[$index])){
                return true;
            }
        }
        return false;            
    }
    
    /**
     * pop
     * @return mixed value
     */
    public function pop($name){
        if($this->has($name) == false){
            throw new Ficus_PropertyNotFoundException("property $name is not found");
        }
        if(is_array($this->{$name}) == false){
            throw new Ficus_IllegalArgumentException("property $name is not an array");
        }
        
        return array_pop($this->{$name});
    }

    /**
     * shift
     * @return mixed value
     */
    public function shift($name){
        if($this->has($name) == false){
            throw new Ficus_PropertyNotFoundException("property $name is not found");
        }
        if(is_array($this->{$name}) == false){
            throw new Ficus_IllegalArgumentException("property $name is not an array");
        }
        
        return array_shift($this->{$name});
    }

    /**
     * push
     * @return mixed value
     */
    public function push($name, $value){
        if($this->has($name) == false){
            throw new Ficus_PropertyNotFoundException("property $name is not found");
        }
        if(is_array($this->{$name}) == false){
            throw new Ficus_IllegalArgumentException("property $name is not an array");
        }

        array_push($this->{$name}, $value);
        return $this;
    }

    /**
     * push
     * @return mixed value
     */
    public function pushAll($name, $value){
        if($this->has($name) == false){
            throw new Ficus_PropertyNotFoundException("property $name is not found");
        }
        if(is_array($this->{$name}) == false){
            throw new Ficus_IllegalArgumentException("property $name is not an array");
        }
        if(is_array($value) == false){
            throw new Ficus_IllegalArgumentException("value must be an array");
        }

        $this->{$name} = array_merge($this->{$name}, $value);
        return $this;
    }

    /**
     * clear
     */
    public function clear($name){
        if($this->has($name) == false){
            throw new Ficus_PropertyNotFoundException("property $name is not found");
        }
        if(is_array($this->{$name})){
            $this->{$name} = array();
        }else{
            $this->{$name} = null;
        }
        return $this;
    }

    /**
     * clear
     */
    public function isEmpty($name){
        if($this->has($name) == false){
            throw new Ficus_PropertyNotFoundException("property $name is not found");
        }
        return empty($this->{$name});
    }

    /**
     * count
     */
    public function count($name){
        if($this->has($name) == false){
            throw new Ficus_PropertyNotFoundException("property $name is not found");
        }
        return count($this->{$name});
    }        

    /**
     * get properties
     * @param $pattern string to get
     * @return array of string properties
     */
    public function beanProperties($pattern = null){
        $propNames = array();
        $props = get_object_vars($this);
        foreach($props as $name => $prop){
            $propNames[$name] = $name;
        }
        if(is_null($pattern)){
            return $propNames;
        }
        
        $result = array();
        foreach($propNames as $prop){
            if(preg_match($pattern, $prop)){
                $result[] = $prop;
            }
        }
        return $result;
    }

    /**
     * other functions
     */
    public function __call($name, $arguments){
        return $this->__onCall($name, $arguments);
    }

    /**
     * handle call
     */
    protected function __onCall($name, $arguments){
        $class = new ReflectionClass($this);
        if($class->hasProperty($name)){
            $property = $class->getProperty($name);
            if($property->isStatic()){
                $values = $class->getStaticProperties();
                $value = $values[$name];
            }else{
                $value = $this->{$name};
            }
            if(isset($arguments[0])){
                return $value[$arguments[0]];
            }else{
                return $value;
            }
        }
        if($this->has($name)){
            $value = $this->{$name};
            if(isset($arguments[0])){
                return $value[$arguments[0]];
            }else{
                return $value;
            }
        }

        if(preg_match(self::METHOD_SIGNATURE, $name, $matches)){
            $operator = $matches[1];
            $property = strtolower($matches[2][0]) . substr($matches[2], 1);

            switch($operator){
            case "has" :
                if(isset($arguments[0])){
                    return $this->has($property, $arguments[0]);
                }
                return $this->has($property);
            }

            if($this->has($property)){
                switch($operator){
                case "set" :
                    $this->{$property} = $arguments[0];
                    return $this;
                case "get" :
                    if(isset($arguments[0])){
                        if(is_null($arguments[0])){
                            return null;
                        }
                        if(isset($this->{$property}[$arguments[0]])){
                            return $this->{$property}[$arguments[0]];
                        }else{
                            return null;
                        }
                    }
                    return $this->{$property};
                case "add" :
                case "push" :
                case "append" :
                    return $this->push($property, $arguments[0]);
                case "addAll" :
                case "pushAll" :
                case "appendAll" :
                    return $this->pushAll($property, $arguments[0]);
                case "shift" :
                    return $this->shift($property);
                case "pop" :
                    return $this->pop($property);
                case "delete" :
                    if(isset($arguments[0])){
                        if((is_numeric($arguments[0]) ||
                            is_string($arguments[0])) &&
                           isset($this->{$property}[$arguments[0]])){
                            $temp = $this->{$property}[$arguments[0]];
                            unset($this->{$property}[$arguments[0]]);
                            return $temp;
                        }else{
                            foreach($this->{$property} as $key => $item){
                                if($item == $arguments[0]){
                                    unset($this->{$property}[$key]);
                                    return $item;
                                }
                            }
                            return $this;
                        }
                    }
                    break;
                case "insert" :
                    if(isset($arguments[0])){
                        $this->{$property}[$arguments[0]] = $arguments[1];
                        return $this;
                    }
                    break;
                case "numOf" :
                case "count" :
                    return $this->count($property);
                case "clear" :
                    return $this->clear($property);
                case "isEmpty" :
                    return $this->isEmpty($property); 
                case "isNot" :
                    return ((boolean)$this->{$property} == false);
                case "is" :
                    return (boolean)$this->{$property};
                case "enable" :
                    $this->{$property} = true;
                    return $this;
                case "disable" :
                     $this->{$property} = false;
                    return $this;
               }
            }
        }
        if(preg_match(self::METHOD_SIGNATURE_SUFFIX, $name, $matches)){
            $operator = $matches[2];
            $property = strtolower($matches[1][0]) . substr($matches[1], 1);
            if($this->has($property)){
                switch($operator){
                case "EqualsTo" :
                    return Ficus_Types::equalsByToString($arguments[0],
                                                        $this->{$property});
                }
            }
        }
        if(method_exists($this, $name)){
            call_user_func_array(array($this, $name), $arguments);
        }else if(isset($property) &&
                 $this->has($property) == false){
            throw new Ficus_PropertyNotFoundException("property $property is not found.");
        }else{
            throw new Ficus_MethodNotFoundException("method $name is not found.");
        }
    }
}
