<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages
 */
/**
 * @file S2ContainerComponentFactory.php
 * @brief S2Container component factory
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: S2ContainerComponentFactory.php 48 2007-09-20 07:21:07Z ishitoya $
 * 
 * S2Container component factory
 */

require_once("ficus/io/File.php");
require_once("ficus/config/Registry.php");
require_once("ficus/di/ComponentFactory.php");

/**
 * @class Ficus_S2ContainerComponentFactory
 */
abstract class Ficus_S2ContainerComponentFactory
implements Ficus_ComponentFactory{
    const DICON_PATH = "s2container.dicon_path";
    
    /**
     * @var S2Container S2Container.
     */
    private static $containers = array();

    /**
     * @var Ficus_S2ContainerComponentFactory
     */
    protected static $factory = null;

    /**
     * @var string dicon path
     */
    protected $diconPath = null;

    /**
     * get Dicon file name from registry
     * @return string dicon filename
     */
    protected abstract function getDiconFileNameRegistry();

    /**
     * get Dicon namespace from registry
     * @return string dicon namespace
     */
    protected abstract function getDiconNameSpaceRegistry();

    /**
     * get dicon file name
     * @return string dicon file name
     */
    protected abstract function getDefaultDiconFileName();
    
    /**
     * get defoult dicon namespace
     * @return string dicon namespace
     */
    protected abstract function getDefaultDiconNameSpace();

    /**
     * instanciate factory
     */
    protected static function getInstance($class){
        return new $class;
    }
    
    /**
     * check for registry
     */
    protected function checkForRegistry(){
        if(Ficus_Registry::search(self::DICON_PATH) == false){
            throw new Ficus_NotReadyException("s2container.dicon_path must be declared in Ficus_Registry before runch any function");
        }
        return;
    }
    
    /**
     * Get dicon path.
     *
     * return string dicon path..
     */
    protected function getDiconPath() {
        $diconPath = Ficus_Registry::search($this->getDiconFileNameRegistry());
        if($diconPath == false){
            $diconPath = $this->getDefaultDiconFileName();
        }else{
            $candidates = Ficus_ClassPath::search($diconPath);
            if(empty($candidates)){
                throw new Ficus_IllegalArgumentException("$diconPath is not resonable file name, check out class path");
            }
            $diconPath = $candidates[0];
        }
        
        return $diconPath;
    }

    /**
     * Get S2Container.
     *
     * return S2Container S2Container.
     */
    protected function getS2Container() {
        if (empty(self::$containers)){
            Ficus_AutoLoad::add(new Ficus_S2ContainerAutoLoad());
        }
        if (is_null($this->diconPath) ||
            isset(self::$containers[$this->diconPath]) == false) {
            $this->checkForRegistry();
            $root = Ficus_Registry::search(self::DICON_PATH);
            if(defined("DICON_PATH") == false){
                $candidates = Ficus_ClassPath::search($root);
                if(empty($candidates)){
                    throw new Ficus_IllegalArgumentException("$root is not resonable dicon path, check out class path");
                }
                define("DICON_PATH", $candidates[0]);
            }

            $this->diconPath = $this->getDiconPath();
            self::$containers[$this->diconPath]
                = S2ContainerFactory::create($this->diconPath);
        }
        return self::$containers[$this->diconPath];
    }

    /**
     * get component
     *
     * @return component
     */
    public static function getComponent($name, $class = __CLASS__){
        if(is_null(self::$factory)){
            self::$factory = self::getInstance($class);
            if(is_null(self::$factory)){
                throw new Ficus_NotImplementedException("extended class must implement getInstance method");
            }                
        }
        return self::$factory->getS2Container()->getComponent($name);
    }
}
?>
