<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file ClassLoader.php
 * @brief ClassLoader for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: ClassLoader.php 2 2007-07-11 10:37:48Z ishitoya $
 *
 * Class Loader for php
 */

require_once("ficus/exception/IllegalArgumentException.php");
require_once("ficus/exception/ClassNotFoundException.php");
require_once("ficus/exception/MultipleClassDefinitionFoundException.php");
require_once("ficus/exception/NotReadyException.php");
require_once("ficus/lang/ClassPath.php");
require_once("ficus/io/Dir.php");
require_once("ficus/test/PathInitializer.php");

/**
 * @class Ficus_ClassLoader
 */
class Ficus_ClassLoader
{
    const CACHE_IMPORT = "import";
    const CACHE_CONSTRUCT = "construct";

    /**
     * @var $cache Array cache
     */
    private static $cache = array();

    /**
     * import class
     * @param $classPath string class name to import
     * @param $classname string class if classqname and classname are not same
     * @throw Ficus_NotReadyException no initialized.
     * @throw Ficus_ClassNotFoundException class not found.
     * @throw Ficus_MultipleClassDefinitionFoundException multiple definition.
     */
    public static function import($classPath, $classname=""){
        $index = $classPath;

        if(empty($classname)){
            if(preg_match('/.*?\.?([^\.]+)$/', $classPath, $regs) == false){
                throw new Ficus_IllegalArgumentException("class path $classPath is illegal");
            }
            $classname = $regs[1];
        }
        
        if(class_exists($classname)){
            return;
        }

        $result = self::normalizeClassPath($classPath, $classname);
        $classPath = $result[0];
        $classname = $result[1];
        $classPath = self::searchForClass($classPath);

        self::$cache[self::CACHE_IMPORT][$index] = $classPath;
        require_once($classPath);

        if(!class_exists($classname)){
            throw new Ficus_ClassNotFoundException(__FILE__ . ": "
                                                  . "class " . $classname . " not found in " . $classPath . ".");
        }
    }

    /**
     * normalize Class path
     * @param $classPath string class name to import
     * @param $classname string class if classqname and classname are not same
     * @throw Ficus_ClassNotFoundException class not found.
     */
    protected static function normalizeClassPath($classPath, $classname){
        $classPath = Ficus_ClassPath::packageToDirectory($classPath);
        $classPath =
          ereg_replace(Ficus_ClassPath::ESCAPED_DOT,
                       Ficus_ClassPath::PACKAGE_SEPARATOR,
                       $classPath);

        $regs = array();
        //case hogehoge/ugeuge/moge.php or the hogehoge/moge.php
        if(preg_match("/(.*)\/(([^\/]*_)?([^\/]*))$/", $classPath, $regs)){
            $classname = empty($classname) ? $regs[2] : $classname;
            $shortname = $regs[4];
            $rootpath = array();
            //$classPath = self::$classPath . self::$defaultDir . "/";
            $classPath = $regs[1] . "/" . $shortname . ".php";
            //class is placed in root path then class path is class name.
        }else if(ereg("_([^\/_]*)", $classPath, $regs)){
            $classname = empty($classname) ? $classPath : $classname;
            $shortname = $regs[1];
            $classPath = $shortname . ".php";
        }else{
            throw new Ficus_ClassNotFoundException(__FILE__ . ": "
                                                  . "class " . $classPath . " not found.");
        }
        return array($classPath, $classname);
    }

    /**
     * search for class
     * @param $classPath string class name to import
     * @throw Ficus_ClassNotFoundException class not found.
     * @throw Ficus_MultipleClassDefinitionFoundException multiple definition.
     */
    protected static function searchForClass($classPath){
        $foundClasses = Ficus_ClassPath::search($classPath);
        
        if(count($foundClasses) >= 2){
            throw new Ficus_MultipleClassDefinitionFoundException("Multiple Class Definition Found. found classes are " . implode(",", $foundClasses) . ".");
        }else if(empty($foundClasses)){
            throw new Ficus_ClassNotFoundException(__FILE__ . ": " . "class " . $classPath . " not found.");
        }
        return $foundClasses[0];
    }

    /**
     * not only import but load class.
     * @param $classQName String qualified class name
     * @param $parameter array parameters for constructor
     * @return Mixed instance of class
     */
    public static function load($classQName, $parameter = null){
        if(isset(self::$cache[self::CACHE_IMPORT][$classQName]) == false){
            self::import($classQName);
        }
        if(preg_match("/\.?([^.]*)$/", $classQName, $regs)){
             return self::constructClass($regs[1], $parameter);
        }

        return null;
    }

    /**
     * load class with class name.
     * Use this method when class file name and class name is differ.
     * @param $classPath string class to import
     * @param $className string class name to load
     * @param $parameter array parameters for constructor
     * @return Mixed instance of class
     */
    public static function loadWithClassName($classPath,
                                             $className,
                                             $parameter = null){
        self::import($classPath, $className);
        return self::constructClass($className, $parameter);
    }

    /**
     * load class from file
     * Use this method when class is not saved yet.
     * @param $source string source code.
     * @param $className string class name to load
     * @param $alternate boolean when the class name is already exists.
     * @param $parameter array parameters for constructor
     * @return Mixed instance of class
     */
    public static function loadFromFile($source, $classname,
                                        $alternate = false, $parameter = null){
        $source = str_replace(array("<?php", "?>"), "", $source);
        if($alternate){
            $alt = self::getAlternateClassname($classname);
            $source =
              preg_replace("/^class $classname/m", "class $alt", $source);
            $classname = $alt;
        }

        eval($source);
        return self::constructClass($classname, $parameter);
    }

    /**
     * get Alternate classname
     * @param $classname string class name
     * @return string alternate classname
     */
    private function getAlternateClassname($classname){
        for($i = 1; class_exists($classname); $i++){
            $classname = "{$classname}_1";
        }
        return $classname;
    }

    /**
     * construct class with a constructor parameters.
     * please give me an idea to make better this code
     * @param $className string className
     * @param $param array constructor parameter
     * @throw Ficus_IllegalArgumentException not instantiable.
     */
    private function constructClass($className, $param){
        $start = microtime(true);
        if(isset(self::$cache[self::CACHE_CONSTRUCT][$className]) == false){
            $class = new ReflectionClass($className);
            if($class->isAbstract() || $class->isInterface()){
                throw new Ficus_IllegalArgumentException("this class $className can not instantiable");
            }

            $requiredParamCount = 0;
            $paramCount = 0;
            $constructor = $class->getConstructor();
            self::$cache[self::CACHE_CONSTRUCT][$className] = $constructor;
        }else{
            $constructor = self::$cache[self::CACHE_CONSTRUCT][$className];
        }
        
        if(is_null($constructor) == false){
            $requiredParamCount =$constructor->getNumberOfRequiredParameters();
            $paramCount = $constructor->getNumberOfParameters();
        }

        if(is_null($param)){
            if($requiredParamCount != 0){
                throw new Ficus_IllegalArgumentException("this class's constructor have required parameters.");
            }
            return new $className();
        }
        if(is_array($param) == false){
            throw new Ficus_IllegalArgumentException("parameter must be array");
        }

        $count = count($param);
        if($count < $requiredParamCount ||
           $count > $paramCount){
            throw new Ficus_IllegalArgumentException("wrong parameter count.");
        }

        switch($count){
          case (0)  : return new $className();
          case (1)  : return new $className($param[0]);
          case (2)  : return new $className($param[0], $param[1]);
          case (3)  : return new $className($param[0], $param[1], $param[2]);
          case (4)  : return new $className(
              $param[0], $param[1], $param[2], $param[3]);
          case (5)  : return new $className(
              $param[0], $param[1], $param[2], $param[3], $param[4]);
          case (6)  : return new $className(
              $param[0], $param[1], $param[2], $param[3], $param[4],
              $param[5]);
          case (7)  : return new $className(
              $param[0], $param[1], $param[2], $param[3], $param[4],
              $param[5], $param[6]);
          case (8)  : return new $className(
              $param[0], $param[1], $param[2], $param[3], $param[4],
              $param[5], $param[6], $param[7]);
          case (9)  : return new $className(
              $param[0], $param[1], $param[2], $param[3], $param[4],
              $param[5], $param[6], $param[7], $param[8]);
          case (10) : return new $className(
              $param[0], $param[1], $param[2], $param[3], $param[4],
              $param[5], $param[6], $param[7], $param[8], $param[9]);
          default   : throw new Ficus_IllegalArgumentException("This method supports only max 10 parameter now.");
        }
    }
}
?>
