<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.net.rmi
 */
/**
 * @file AbstractRemoteMethodInvocation.php
 * @brief abstract class for remote method invocation
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: AbstractRemoteMethodInvocation.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * remote method invocation abstract class. implements some http methods.
 */

require_once("HTTP/Request.php");
require_once("ficus/net/rmi/RemoteMethodInvocation.php");

/**
 * @class Ficus_AbstractRemoteMethodInvocation
 */
abstract class Ficus_AbstractRemoteMethodInvocation
         implements Ficus_RemoteMethodInvocation
{
    const HTTP_VERSION_11 = "1.1";
    const USER_AGENT = "Infocrew Accessor";

    const ENCODING_EUC = "EUC-JP";
    const ENCODING_UTF8 = "UTF-8";
    const ENCODING_ISOJP = "ISO-2022-JP";
    
    /**
     * @var $endpoint string endpoint
     */
    protected $endpoint = "";

    /**
     * @var $response string response
     */
    protected $response = "";
        
    /**
     * HTTP_Request object
     */
    protected $http = null;

    /**
     * user
     */
    protected $user = "";

    /**
     * password
     */
    protected $password = "";

    /**
     * agent
     */
    protected $agent = self::USER_AGENT;

    /**
     * method
     */
    protected $method = HTTP_REQUEST_METHOD_POST;

    /**
     * headers
     */
    protected $headers = array();

    /**
     * httpver
     */
    protected $httpVersion = self::HTTP_VERSION_11;

    /**
     * @var $encoding string pukiwiki encoding
     */
    protected $encoding = self::ENCODING_EUC;

    /**
     * invoke remote method
     * @param $request array of request
     * @return string result
     */
    public final function invoke($request){
        if(empty($this->endpoint)){
            throw new Ficus_IllegalArgumentException("endpoint must not be empty");
        }
        $this->http = new HTTP_Request($this->endpoint);
        $this->http->setHttpVer($this->httpVersion);
        $this->http->setMethod($this->method);
        $this->http->addHeader('User-Agent', $this->agent);
        if(empty($this->user) == false){
            $this->http->setBasicAuth($this->user, $this->password);
        }
        $this->onInvocation($request);

        $response = $this->http->sendRequest();

        if(PEAR::isError($response)){
            return $this->processError($response);
        }else{
            return $this->processResponse($this->http->getResponseBody());
        }
    }

    /**
     * invoke remote method
     * @param $request array of request
     */
    public function onInvocation($request){
        if(is_array($request)){
            foreach($request as $name => $value){
                $this->addPostData($name, $value);
            }
        }
    }


    /**
     * add post data
     * @param $name string name of data
     * @param $value string value of data
     */
    protected function addPostData($name, $value){
        $this->http->addPostData($name, $value);
    }
    
    /**
     * process Response
     * @param $response string response of remote method
     * @return string result
     */
    protected final function processResponse($response){
        return $this->onProcessResponse($response);
    }

    /**
     * process Response
     * @param $response Object response of remote method
     * @return string result
     */
    protected function onProcessResponse($response){
        return $response;
    }


    /**
     * process Response
     * @param $response string response of remote method
     * @return string result
     */
    protected final function processError($response){
        return $this->onProcessError($response);
    }

    /**
     * process Error
     * @param $response Object response of remote method
     * @return string result
     */
    protected function onProcessError($response){
        return $response->getMessage();
    }

    /**
     * get response
     * @return string response of last invocation
     */
    public function getResponse(){
        return $this->response;
    }

    /**
     * set endpoint
     * @param $endpoint endpoint of method
     */
    public function setEndpoint($endpoint){
        $this->endpoint = $endpoint;
    }

    /**
     * set user
     * @param $user string user name
     */
    public function setUser($user){
        $this->user = $user;
    }

    /**
     * set password
     * @param $password string password
     */
    public function setPassword($password){
        $this->password = $password;
    }

    /**
     * set User agent
     * @param $agent string agent
     */
    public function setAgent($agent){
        $this->agent = $agent;
    }

    /**
     * set Http version
     * @param $version string version of http
     */
    public function setHTTPVersion($version){
        $this->httpVersion = $version;
    }

    /**
     * set Method
     * @param $method string method of http
     */
    public function setMethod($method){
        $this->method = $method;
    }

    /**
     * set encoding
     * @param $encoding string encoding
     */
    public function setEncoding($encoding){
        $this->encoding = $encoding;
    }

    /**
     * encode
     * @param $data string to encode
     * @return string encoded data
     */
    protected function getEncoded($data){
        $src = mb_detect_encoding($data);
        return mb_convert_encoding($data, $this->encoding, $src);
    }                              
}
?>
