<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages
 */
/**
 * @file AbstractPageAuthenticationManager.php
 * @brief authentication manager
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: AbstractPageAuthenticationManager.php 19 2007-07-25 18:20:09Z ishitoya $
 */

/**
 * @class Ficus_AbstractPageAuthenticationManager
 */
abstract class Ficus_AbstractPageAuthenticationManager
implements Ficus_PageAuthenticationManager, S2AnA_AuthenticationContext{
    const TOKEN_ALL = "*";

    protected $current = null;
    
    /**
     * authenticate user
     * @param $visit Ficus_PageVisitBean information of user
     * @return Ficus_PageVisitBean information of user
     */
    public function authenticate($visit = null){
        $result = $this->onAuthenticate($visit);
        return $result;
    }

    /**
     * on authenticate user
     * @param $visit Ficus_PageVisitBean information of user
     * @return Ficus_PageVisitBean information of user
     */
    abstract protected function onAuthenticate($visit = null);
    
    /**
     * authorize user
     * @param $page Ficus_Page page
     * @param $visit Ficus_PageVisitBean bean
     * @return Ficus_PageVisitBean bean
     */
    public function authorize($page, $visit = null){
        if(is_null($visit)){
            $visit = $this->authenticate();
        }
        $context = $this->getAuthenticationContext($page);
        $visit->setContext($context);

        $method = $context["method"];
        $this->current = ($visit);
        switch($method){
        case("requireLogin"):
            $visit = $this->onAuthorizeRequireLogin($page, $visit);
            break;
        case("allow"):
            $visit = $this->onAuthorizeAllow($page, $visit);
            break;
        case("deny"):
            $visit = $this->onAuthorizeDeny($page, $visit);
            break;
        }
        return $this->onAuthorized($page, $visit);
    }

    /**
     * on Authorized
     */
    protected function onAuthorized($page, $visit){
        return $visit;
    }
    
    /**
     * on authorize
     * @param $page Ficus_Page page
     * @param $visit Ficus_PageVisitBean bean
     * @return Ficus_PageVisitBean bean
     */
    protected function onAuthorizeRequireLogin($page, $visit){
        $context = $visit->context();
        $visit->disableLogined();
        if(isset($context["role"]) && $visit->isAuthenticated() &&
           $this->isResonableUser($visit)){
            $visit->enableLogined();
            $visit->disableGuest();
        }else{
            if(isset($context["redirectTo"]) == false){
                $context["redirectTo"] = $this->getPageName("login");
            }
            $visit->setContext($context);
        }
        return $visit;
    }
    
    /**
     * on authorize
     * @param $page Ficus_Page page
     * @param $visit Ficus_PageVisitBean bean
     * @return Ficus_PageVisitBean bean
     */
    protected function onAuthorizeAllow($page, $visit){
        $context = $visit->context();
        $visit->disableLogined();
        if(isset($context["role"]) && $visit->isAuthenticated() &&
           $this->isResonableUser($visit)){
            $visit->enableLogined();
            $visit->disableGuest();
        }else{
            if(isset($context["redirectTo"]) == false){
                $context["redirectTo"] = $this->getPageName("restricted");
            }
            $visit->setContext($context);
        }
        return $visit;
    }
        

    /**
     * on authorize
     * @param $page Ficus_Page page
     * @param $visit Ficus_PageVisitBean bean
     * @return Ficus_PageVisitBean bean
     */
    protected function onAuthorizeDeny($page, $visit){
        $context = $visit->context();
        $visit->enableLogined();
        $visit->disableGuest();
        if(isset($context["role"]) && $visit->isAuthenticated() &&
           $this->isResonableUser($visit)){
            $visit->disableLogined();
            if(isset($context["redirectTo"]) == false){
                $context["redirectTo"] = $this->getPageName("restricted");
            }
            $visit->setContext($context);
        }

        return $visit;
    }

    /**
     * get page authentication settings
     */
    public function getAuthenticationContext($page){
        if($page instanceof Ficus_Page){
            $pagename = $page->pagename();
        }else{
            $pagename = $page;
        }
        $contexts = Ficus_Registry::search("pages.authentication.contexts");
        if($contexts == false){
            return null;
        }
        $context = null;
        foreach($contexts as $key => $setting){
            foreach($setting as $item){
                if($item["page"] === $pagename){
                    $context = $item;
                    $context["method"] = $key;
                    break 2;
                }
            }
        }
        if(is_null($context)){
            foreach($contexts as $key => $setting){
                foreach($setting as $item){
                    if($item["page"] === self::TOKEN_ALL){
                        $context = $item;
                        $context["method"] = $key;
                        break 2;
                    }
                }
            }
        }
        if(isset($context["role"])){
            $roles = explode(",", $context["role"]);
            $context["role"] = array();
            foreach($roles as $role){
                $role = trim($role);
                $context["role"][$role] = $role;
            }
        }
        return $context;
    }

    /**
     * get Page
     */
    public function getPageName($name){
        $pages = Ficus_Registry::search("pages.authentication.pages");
        if(isset($pages[$name])){
            return $pages[$name];
        }
        return null;
    }

    /**
     * isSystemPage
     */
    public function isNoAuthenticationPage($page){
        $pages = Ficus_Registry::search("pages.authentication.pages");
        $context = $this->getAuthenticationContext($page);
        $name = $page->pagename();
        
        if(is_null($context) ||
           ($context["page"] == self::TOKEN_ALL &&
            in_array($name, $pages))){
            return true;
        }
        return false;
    }

    /**
     * get bean
     */
    public function getBean(){
        return new Ficus_PageVisitBean();
    }
    //s2ana functions
    
    /**
     * isAuthorized
     */
    public function isAuthenticated(){
        return $this->getUserPrincipal()->isAuthenticated();
    }

    /**
     * get user principal
     */
    public function getUserPrincipal(){
        return $this->current;
    }

    /**
     * is user in role
     */
    public function isUserInRole($roleName){
        return $this->current->role() == $roleName;
    }

    /**
     * is acceptable user
     */
    public function isResonableUser($visit){
        $context = $visit->context();
        $roles = $context["role"];
        if(isset($roles[$visit->role()])){
            return true;
        }
        if(isset($roles[self::TOKEN_ALL])){
            return true;
        }
        return false;
    }
}
?>