<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages.scaffold
 */
/**
 * @file ScaffoldTableConfiguration.php
 * @brief ScaffoldTableConfiguration.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: ScaffoldTableConfiguration.php 54 2007-09-27 19:37:15Z ishitoya $
 *
 */

/**
 * @class Ficus_ScaffoldTableConfiguration
 */
class Ficus_ScaffoldTableConfiguration extends Ficus_Bean
implements Ficus_ScaffoldConstants, Ficus_S2DaoModelConstants{
    const TOKEN_ALL = "*";
    
    /**
     * exclude pattern
     */
    protected $excludes = array();

    /**
     * exclude pattern
     */
    protected $patternExcludes = array();

    /**
     * no label
     */
    protected $nolabels = array();

    /**
     * no children
     */
    protected $noChildren = array("*" => true);

    /**
     * transitions
     */
    protected $transitions = array();

    /**
     * current Transition
     */
    protected $currentTransition = array();

    /**
     * default values
     */
    protected $defaultValues = array();

    /**
     * property settings
     */
    protected $properties = array();

    /**
     * parent Part
     */
    protected $parentPart = null;

    /**
     * assists
     */
    protected $assists = null;

    /**
     * context
     */
    protected $context = null;

    /**
     * check for exclude
     * @param $part Ficus_ScaffoldPart part to check
     */
    public function isExcludeProperty($part){
        $context = $this->context;
        $qname = $part->qname();

        $page = Ficus_PageComponentFactory::getPageController()->chain()
                                                          ->current()->page();

        if(isset($this->excludes[$page . "." . self::TOKEN_ALL]) ||
           isset($this->excludes[$page . "." . $context]) ||
           isset($this->patternExcludes[$page . "." . self::TOKEN_ALL]) ||
           isset($this->patternExcludes[$page . "," . $context])){
            if(isset($this->excludes[$page . "." . self::TOKEN_ALL][$qname]) ||
               isset($this->excludes[$page . "." . $context][$qname])){
                return true;
            }
            if(isset($this->patternExcludes[$page . "." . self::TOKEN_ALL])){
                foreach($this->patternExcludes[$page . "." . self::TOKEN_ALL]
                        as $pattern){
                    if(preg_match('/' . $pattern . '/', $qname)){
                        return true;
                    }
                }
            }
            if(isset($this->patternExcludes[$page . "." . $context])){
                foreach($this->patternExcludes[$page . "." . $context]
                        as $pattern){
                    if(preg_match('/' . $pattern . '/', $qname)){
                        return true;
                    }
                }
            }
            return false;
        }            
        
        if((isset($this->excludes[self::TOKEN_ALL]) &&
            isset($this->excludes[self::TOKEN_ALL][$qname]))){
            return true;
        }

        if(isset($this->excludes[$context]) &&
           isset($this->excludes[$context][$qname])){
            return true;
        }


        if(isset($this->patternExcludes[self::TOKEN_ALL])){
            foreach($this->patternExcludes[self::TOKEN_ALL] as $pattern){
                if(preg_match('/' . $pattern . '/', $qname)){
                    return true;
                }
            }
        }
        if(isset($this->patternExcludes[$context])){
            foreach($this->patternExcludes[$context] as $pattern){
                if(preg_match('/' . $pattern . '/', $qname)){
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * check for exclude
     * @param $part Ficus_ScaffoldPart part to check
     */
    public function isNoLabelProperty($part){
        $entityName = $part->entityName();
        if(array_key_exists($entityName, $this->nolabels) &&
           $this->nolabels[$entityName] == $part->name()){
            return true;
        }
        return false;
    }

    /**
     * check for transition property
     * @param $entity Ficus_ScaffoldEntity entity to check
     */
    public function isTransitionProperty($entity){
        $entityName = $entity->parentEntityName($entity);

        if(empty($this->currentTransition)){
            if($entity->hasParent() == false){
                return true;
            }
            return false;
        }
        if($entity->isComposite() &&
           in_array($entityName, $this->currentTransition)){
            return true;
        }
        if(in_array($entityName, $this->currentTransition) == false){
            return false;
        }

        return true;
    }

    /**
     * is not in transition
     */
    public function isExistsInTransition($entity){
        if(in_array($entity->parentEntityName(), $this->transitions)){
            return true;
        }
        if(empty($this->transitions)){
            return false;
        }
        return false;
    }

    /**
     * check is transition entity
     */
    public function isTransitionEntity($part){
        if($part->isTypeForeign() &&
           in_array($part->entityName(), $this->currentTransition)){
            return true;
        }
        return false;
    }

    /**
     * check is parent of transition property
     */
    public function isTransitionParentProperty($part){
        if($part->isTypeForeign()){
            $parent = $part->entityName();
            foreach($this->currentTransition as $transition){
                if(strpos($transition, $parent) === 0){
                    return true;
                }
            }
        }
        return false;            
    }
    
    /**
     * get default value
     * @param $entity Ficus_ScaffoldEntity entity to check
     */
    public function getDefaultValue($entity){
        if($entity->isTypeList() == false && $entity->isTypeDirect() == false){
            return null;
        }
        $qname = $entity->qname();
        if(preg_match('/Direct$/', $qname) == false){
            $qname .= "Direct";
        }
        if(isset($this->defaultValues[$qname])){
            return $this->defaultValues[$qname];
        }
        return null;
    }

    /**
     * is no children
     */
    public function isNoChildren(){
        $context = $this->context;
        $page = Ficus_PageComponentFactory::getPageController()->chain()
                                                          ->current()->page();
        if(isset($this->noChildren[$page])){
            return $this->noChildren[$page];
        }
        if(isset($this->noChildren[$context])){
            return $this->noChildren[$context];
        }
        if(isset($this->noChildren[self::TOKEN_ALL])){
            return $this->noChildren[self::TOKEN_ALL];
        }
        return false;
    }

    /**
     * ignore parts pattern
     * @param $entityname string ignore entity name
     * @param $property string when null, ignore class
     */
    public function addExcludeProperty($context, $qname){
        $contexts = explode(",", $context);
        foreach($contexts as $context){
            if(is_string($context) == false){
                throw new Ficus_InvalidConfigurationException("exclude property must be string");
            }
            $context = trim($context);
            $this->excludes[$context][$qname] = $qname;
            if(strpos($qname, "*") !== false){
                $this->patternExcludes[$context][] =
                    str_replace("*", ".*", $qname);
            }
        }
    }

    /**
     * no label pattern
     */
    public function addNoLabelProperty($entityname, $property = null){
        $this->nolabels[$entityname] = $property;
    }

    /**
     * set no children
     * @param $noChildren boolean set to true, ignore child entities
     */
    public function addNoChildren($context, $noChildren){
        $contexts = explode(",", $context);
        foreach($contexts as $context){
            $context = trim($context);
            $this->noChildren[$context] = $noChildren;
        }
    }

    /**
     * set transition
     * @param $transition string transition
     */
    public function setCurrentTransition($transition){
        $transition = explode(",", $transition);
        $this->currentTransition = array();
        foreach($transition as $trans){
            $this->currentTransition[] = trim($trans);
        }
    }

    /**
     * set transitions
     * @param $transitions array transitions
     */
    public function setTransitions($transitions){
        $this->transitions = $transitions;
    }

    /**
     * set default values
     */
    public function addDefaultValues($column, $value){
        $this->defaultValues[$column] = $value;
    }

    /**
     * get Validators
     */
    public function validators($property, $name = null){
        if(isset($this->properties[$property]["validators"])){
            if(in_array($name, $this->properties[$property]["validators"])){
                return true;
            }else if(is_null($name)){
                return $this->properties[$property]["validators"];
            }
        }
        return false;
    }
    


    /**
     * get Next transition
     * @param $key string current transition, if ommited, return first one
     * @return string next transition
     */
    public function getNextTransition($key = null){
        $transitions = $this->transitions;
        if(empty($transitions)){
            return false;
        }
        if(is_null($key)){
            return array_shift($transitions);
        }else{
            reset($transitions);
            while($value = each($transitions)){
                if($value["value"] == $key){
                    $next = each($transitions);
                    if($next == false){
                        return false;
                    }
                    return $next["value"];
                }
            }
        }
        return false;
    }

    /**
     * merge
     */
    public function merge($parent){
        $properties = array_merge($parent->properties(), $this->properties());
        $this->setProperties($properties);
        return $this;
    }
}
?>
