<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages.scaffold
 */
/**
 * @file Form bean.php
 * @brief Form bean.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: ScaffoldEntity.php 19 2007-07-25 18:20:09Z ishitoya $
 *
 */

require_once("ficus/db/s2dao/models/S2DaoModelConstants.php");

/**
 * @class Ficus_ScaffoldEntity
 */
class Ficus_ScaffoldEntity extends Ficus_ConcreteBeanAggregator
implements Ficus_ScaffoldConstants, Ficus_S2DaoModelConstants{
    /**
     * parent
     */
    protected $parent;
    
    /**
     * entity name
     */
    protected $entity;

    /**
     * name of part
     */
    protected $name;

    /**
     * type of part
     */
    protected $type;

    /**
     * value of the part
     */
    protected $value;

    /**
     * label of the part
     */
    protected $label;

    /**
     * remark of the part
     */
    protected $remark;

    /**
     * composite parts
     */
    protected $parts = array();

    /**
     * mediator
     */
    protected static $mediator = null;

    /**
     * constructor
     */
    public function __construct(){
        parent::__construct("parts");
    }

    public function setMediator($mediator){
        if(is_null(self::$mediator)){
            self::$mediator = $mediator;
        }
    }

    public function config($name = null){
        return self::$mediator->table($name);
    }
    
    /**
     * returns qualified name
     * @return string name
     */
    public function qname(){
        return $this->parentEntityName() . "." . $this->name;
    }

    /**
     * returns html name
     */
    public function serializeQname(){
        return str_replace(".", "__", $this->qname());
    }

    /**
     * get Entity qname
     */
    public function entityName(){
        $qname = $this->parentEntityName();
        if(($this->value() instanceof Ficus_S2DaoEntity)){
            $entityName = $this->value()->getEntityName();
            return $qname . "." . $entityName;
        }
        return $qname;
    }

    /**
     * get qname
     */
    public function parentEntityName(){
        $entityName = $this->entity()->getEntityName();
        if($this->hasParent()){
            $entityName =
                $this->parent()->parentEntityName() . "." . $entityName;
        }
        return $entityName;
    }

    /**
     * check has parent
     * @return boolean true if has parent
     */
    public function hasParent(){
        if(is_null($this->parent)){
            return false;
        }
        return true;
    }

    /**
     * serialize
     */
    public function serialize(){
        $arguments = func_get_args();
        if(empty($arguments)){
            $type = null;
        }else{
            $type = array_shift($arguments);
        }
        if(is_null($type)){
            if($this->config()->context() == self::ACTION_VIEW){
                return parent::serialize("ScaffoldData");
            }else{
                if($this->isHidden()){
                    return parent::serialize("ScaffoldHidden");
                }else if($this->isSelect()){
                    return parent::serialize("ScaffoldSelect");
                }else if($this->isTextArea()){
                    return parent::serialize("ScaffoldTextArea");
                }else{
                    return parent::serialize("ScaffoldEdit");
                }
            }
        }else{
            if(empty($arguments)){
                return parent::serialize($type);
            }else{
                return parent::serialize($type, $arguments);
            }
        }
        throw new Ficus_IllegalArgumentException("serializer is not found");
    }

    /**
     * check is composite part
     * @return boolean true if composite
     */
    public function isComposite(){
        if($this->value instanceof Ficus_S2DaoEntity){
            return true;
        }
        return false;
    }

    /**
     * is hidden
     */
    public function isHidden(){
        if($this->isContextView()){
            if($this->config()->isExcludeProperty($this) ||
               $this->isTypeDirect()){
                return true;
            }
            return false;
        }
        if($this->isTypeDirect() ||
           ($this->config()->isExcludeProperty($this) &&
            $this->config()->isTransitionParentProperty($this) == false)){
            return true;
        }
        if($this->isComposite() &&
           $this->isTypeList() == false &&
           $this->config()->isNoChildren()){
            return true;
        }else if($this->config()->isTransitionProperty($this) == false &&
           $this->config()->isTransitionParentProperty($this) == false){
            return true;
        }
        if($this->hasParent() &&
           $this->config()->isExcludeProperty($this->parent) &&
           $this->parent->isTransitionParent() == false){
            return true;
        }
        return false;
    }

    /**
     * validators
     */
    public function validators(){
        $table = $this->parentEntityName();
        $v = $this->config($table)->validators($this->name());
        if(is_array($v)){
            return $v;
        }
        return array();
    }

    /**
     * is type list
     */
    public function isTypeList(){
        return $this->type == self::TYPE_LIST;
    }

    /**
     * is type list
     */
    public function isTypeDirect(){
        return $this->type == self::TYPE_DIRECT;
    }

    /**
     * is type list
     */
    public function isTypeForeign(){
        return $this->type == self::TYPE_FOREIGN;
    }

    /**
     * is root
     */
    public function isRoot(){
        if(strpos(".", $this->qname()) === false){
            return true;
        }
        return false;
    }
    
    /**
     * transition
     */
    public function isTransition(){
        return $this->config()->isTransitionProperty($this);
    }

    /**
     * transition
     */
    public function isExistsInTransition(){
        if($this->hasParent() &&
           $this->parent()->isExistsInTransition()){
            return true;
        }
        if($this->config()->isExistsInTransition($this)){
            return true;
        }
        return false;
    }

    /**
     * transition
     */
    public function isTransitionEntity(){
        return $this->config()->isTransitionEntity($this);
    }

    /**
     * transition
     */
    public function isTransitionParent(){
        return $this->config()->isTransitionParentProperty($this);
    }

    /**
     * is no child
     */
    public function isNoChildren(){
        return $this->config()->isNoChildren();
    }

    /**
     * is select
     */
    public function isSelect(){
        if($this->isComposite() &&
           $this->isTypeList()){
            return true;
        }
        return false;
    }

    /**
     * is textarea
     */
    public function isTextArea(){
        if(in_array("html", $this->validators())){
            return true;
        }
        return false;
    }

    /**
     * is arbitrary
     */
    public function isArbitrary(){
        if($this->isComposite() &&
           $this->isTransitionEntity() &&
           in_array("required", $this->validators()) == false){
            return true;
        }
        return false;
    }
    
    /**
     * is no label
     */
    public function isNoLabel(){
        return $this->config()->isNoLabelProperty($this);
    }

    /**
     * initialize
     */
    public function initialize(){
        if(($this->isBlank() &&
            $this->isExistsInTransition()) &&
            $this->isTransition() == false){
            return;
        }
        if($this->isTypeList()){
            $id = $this->value()->id();
            $key = $this->value()->name();
            if(is_null($key) && is_null($id)){
                $id = $this->config()->getDefaultValue($this);
                if(is_null($id) == false){
                    $this->entity()->set($this->name() . "Direct", $id);
                }
            }
        }else if($this->isTypeDirect() &&
                 $this->isEmptyValue()){
            preg_match('/^(.*?)Direct$/', $this->name(), $regs);
            $orig = $regs[1];
            $v = $this->entity()->get($orig);
            $d = $this->entity()->get($this->name());
            $def = $this->config()->getDefaultValue($this);
            if(is_null($v) == false && $d != $v->id()){
                $this->setValue($d);
                $this->entity()->set($orig, $d);
            }else if(is_null($v->id()) == false){
                $this->setValue($v->id());
            }else if(is_null($def) == false){
                $this->setValue($def);
                $this->entity()->set($this->name(), $def);
            }
        }
    }

    public function hasAssists(){
        return $this->config($this->entity->getEntityName())->isEmptyAssists() == false;
    }

    public function assists(){
        return $this->config($this->entity->getEntityName())->assists();
    }
    
    public function isBlank(){
        if($this->isComposite() &&
           $this->value->isBlank()){
            return true;
        }else if(is_null($this->value)){
            return true;
        }
        return false;
    }
    
    /**
     * search
     */
    public function search($name){
        if($this->qname() == $name){
            return $this;
        }
        if($this->entityName() == $name){
            return $this;
        }
        foreach($this as $entity){
            $ret = $entity->search($name);
            if(is_null($ret) == false){
                return $ret;
            }
        }
        return null;
    }

    /**
     * is context
     */
    public function isContextView(){
        return $this->config()->context() == self::ACTION_VIEW;
    }
}
?>
