<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.parameters.visitors
 */
/**
 * @file ParameterValueSetter.php
 * @brief ParameterValueSetter for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: ParameterValueSetter.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * ParameterValueSetter.
 */

require_once("ficus/lang/Assert.php");
require_once("ficus/lang/Types.php");
require_once("ficus/parameters/Parameter.php");
require_once("ficus/parameters/visitors/Acceptor.php");
require_once("ficus/parameters/visitors/Visitor.php");
require_once("ficus/validators/Validatable.php");

/**
 * @class Ficus_ParameterValueSetter
 */
class Ficus_ParameterValueSetter extends Ficus_Visitor
{
	/**
	 * @var $parameter Ficus_Parameter Parameter to set Value
	 */
	private $parameter = null;

	/**
	 * @var $values array array of values
	 */
	private $values = array();

	/**
	 * set values
	 * @param $parameter Ficus_Parameter target Parameter
	 * @param $values array array of values to set
	 */
	public function __construct($parameter = null, $values = array()){
		if(is_null($parameter) === false){
			$this->set($parameter, $values);
		}
	}

	/**
	 * set values
	 * @param $parameter Ficus_Parameter target Parameter
	 * @param $values array array of values to set
	 */
	public function set($parameter, $values){
		Ficus_Assert::TypeHinting("Ficus_Parameter", $parameter);
		Ficus_Assert::isPrimitiveType($values, "array");
	    $this->parameter = null;
		$this->values = array();

		$this->parameter = $parameter;
		$this->values = $values;
		$this->parameter->accept($this);
	}		
	
	/**
	 * visit parameters and validate
	 * @param $parameter Ficus_Acceptor validatables
	 */
	public function visit($parameter){
		Ficus_Assert::isInstanceOf($parameter, "Ficus_Acceptor");
		Ficus_Assert::isInstanceOf($parameter, "Ficus_Parameter");
		$name = $parameter->name();
			
		if(array_key_exists($name, $this->values)){
			$value = $this->values[$name];
			if(is_object($value)){
				$value = Ficus_Types::toArrayOf($value);
				$this->values = array_merge($this->values, $value);
			}else if(is_array($value) &&
					 $parameter instanceof Ficus_ArrayParameter){
				$setter = new Ficus_ParameterValueSetter();
                $parameters = $parameter->getParameters();
				$template = current($parameters);
				foreach($value as $v){
					$t = clone $template;
					if(is_array($v)){
						$setter->set($t, $v);
					}else{
						$t->setValue($v);
					}
					$parameter->setValue($t);
				}
			}else{
				$parameter->setValue($value);
			}
		}else{
			$parameter->setValue(null);
		}
		
		//check for child parameters
		if($parameter instanceof Ficus_ComplexParameter &&
			($parameter instanceof Ficus_ArrayParameter) === false){
			$children = $parameter->getParameters();
			foreach($children as $child){
				if($child instanceof Ficus_Acceptor){
					$child->accept($this);
				}
			}
		}
	}

	/**
	 * return filled parameter
	 * @return Ficus_Parameter value filled parameter
	 */
	public function getParameter(){
		return $this->parameter;
	}

	/**
	 * return parameter with in accessor
	 * @return Ficus_ParameterAccessor accessor
	 */
	public function getAccessor(){
		return new Ficus_ParameterAccessor($this->parameter);
	}
}
?>
