#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2010 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

package Lib::Archive;

use strict;
use warnings;

use constant {
    LOCK_SH => 1,
    LOCK_EX => 2,
    LOCK_NB => 4,
    LOCK_UN => 8,
};

use Lib::Logger;
use Lib::User;
use Lib::DateTime;
use Lib::Topic;
use Lib::Keyword;
use Lib::Util;
use Lib::Conf;
use Lib::Language;

my $logger = Lib::Logger->new();

# コンストラクタ
sub new {
    my $self = {};

    use Lib::Conf;
    my $conf = Lib::Conf->new();

    $self->{system_dir} = $conf->get_system_dir();
    $self->{data_dir} = $conf->get_data_dir();
    $self->{docroot_dir} = $conf->get_docroot_dir();
    $self->{site_title} = $conf->get_site_title();
    $self->{sub_title} = $conf->get_sub_title();
    $self->{new_item_num} = $conf->get_new_item_num();
    $self->{topic_num} = $conf->get_topic_num();
    $self->{static_html} = $conf->get_static_html();

    bless($self);
    return $self;
}

# 追加
sub add {
    my $self = shift;
    my $topicid = shift;

    my @new_topics = $self->get_newtopics();
    @new_topics = grep { !/\A$topicid\z/ } @new_topics;

    # 新規データを先頭に追加
    unshift @new_topics, $topicid;
    my $item_num = $self->{new_item_num};
    if (scalar(@new_topics) > $item_num) {
        @new_topics = @new_topics[0..$item_num];
    }

    # 更新情報データファイルを更新
    $self->update_newtopics(@new_topics);
}

# 削除
sub delete {
    my $self = shift;
    my $topicid = shift;

    my @new_topics = $self->get_newtopics();
    @new_topics = grep { !/\A$topicid\z/ } @new_topics;

    # 更新情報データファイルを更新
    $self->update_newtopics(@new_topics);
}

# 更新情報データリストの取得
sub get_newtopics {
    my $self = shift;

    my $datafile = "$self->{data_dir}/newtopic.data";
    my @new_topics = ();

    if (-f $datafile) { # ファイルが存在する場合
        open my $infh, '<', $datafile;
        while (my $data = <$infh>) {
            chomp $data;
            if ($data ne '') {
                push @new_topics, $data;
            }
        }
        close $infh;
    }

    return @new_topics;
}

# 更新情報データファイルの更新
sub update_newtopics {
    my $self = shift;
    my @new_topics = @_;

    # 更新情報データファイルを更新
    my $datafile = "$self->{data_dir}/newtopic.data";
    open my $outfh, '>', $datafile;
    flock $outfh, LOCK_EX;
    foreach my $data (@new_topics) {
        print {$outfh} "$data\n";
    }
    flock $outfh, LOCK_UN;
    close $outfh;

    chmod 0766, $datafile;

    # RSSフィーードを更新
    $self->outputrss2();
}

# 更新情報の一覧を作成する
sub new_topiclist {
    my $self = shift;

    my @new_topics = $self->get_newtopics();

    my $topic = Lib::Topic->new();

    my $list = '';
    for my $i (0 .. $#new_topics) {
        last if ($i >= $self->{topic_num});
        my $topicid = $new_topics[$i];
        $topic->load($topicid);
        my $title = $topic->get_title();
        $list .= <<"END_LIST";
<a href="./?func=detail&id=$topicid">$title</a><br />
END_LIST
    }

    if ($list ne '') {
        my $language = Lib::Language->new();
        my $updateinfo = $language->get_statictext('update-info');
        $list = <<"END_LIST";
<h3>$updateinfo</h3>
$list
<a href="./">もっと見る...</a><br />
<br />
END_LIST
    }

    return $list;
}

# トピックをアーカイブファイルに追加する
sub add_archive {
    my $self = shift;
    my $topicid = shift;
    my $lastdate = shift;
    my $yyyy = substr($lastdate, 0, 4); # 年
    my $mm = substr($lastdate, 5, 2);   # 月

    # アーカイブデータファイルに追加
    my $archivefile = "$self->{data_dir}/$yyyy$mm" . "ar.data";
    open my $archivefh, '>>', $archivefile;
    flock $archivefh, LOCK_EX;
    print {$archivefh} "$topicid,$lastdate\n";
    flock $archivefh, LOCK_UN;
    close $archivefh;

	chmod 0766, $archivefile;
}

# トピックをアーカイブファイルから削除する
sub delete_archive {
    my $self = shift;
    my $topicid = shift;
    my $lastdate = shift;
    my $yyyy = substr($lastdate, 0, 4); # 年
    my $mm = substr($lastdate, 5, 2);   # 月
    my @archives = ();

    my $archivefile = "$self->{data_dir}/$yyyy$mm" . "ar.data";
    open my $infh, '<', $archivefile;
    while (my $data = <$infh>) {
        chomp $data;
        if ($data ne '' && $data ne $topicid) {
            push @archives, $data;
        }
    }
    close $infh;

    # アーカイブファイルを更新
    open my $outfh, '>', $archivefile;
    flock $outfh, LOCK_EX;
    foreach my $archive (@archives) {
        print {$outfh} "$archive\n";
    }
    flock $outfh, LOCK_UN;
    close $outfh;
}

# アーカイブトピックリストの取得
sub get_archivetopics {
    my $self = shift;
    my ($year, $month, $day) = @_;

    my $archivefile = "$self->{data_dir}/$year$month" . "ar.data";
    my @archive_topics = ();

    if (-f $archivefile) { # ファイルが存在する場合
        open my $infh, '<', $archivefile;
        while (my $data = <$infh>) {
            chomp $data ;
            if ($data ne '') {
                my ($topicid, $lastdate) = split(/\,/, $data);
                if ($day eq '') {
                    push @archive_topics, $topicid;
                }
                else {
                    my $date = "$year/$month/$day";
                    if (substr($lastdate, 0, 10) eq $date) {
                        push @archive_topics, $topicid;
                    }
                }
            }
        }
        close $infh;
    }

    return @archive_topics;
}

# アーカイブ表の取得(HTML)
sub archive_list {
    my $self = shift;

    my $datadir = $self->{data_dir};

    my $dir = "$datadir/*ar.data";
    my @files = <${dir}>;
    @files = reverse (sort @files);

    my $list = '';
    foreach my $filename (@files) {
        $filename =~ s/$datadir\///g;
        my $yyyy = substr($filename, 0, 4);
        my $mm = substr($filename, 4, 2);
        $list .= <<"END_LIST";
<a href="./?date=$yyyy$mm">$yyyy/$mm</a><br />
END_LIST
    }

    if ($list ne '') {
        my $language = Lib::Language->new();
        my $archivelog = $language->get_statictext('archive-log');
        $list = "<h3>$archivelog</h3>" . $list;
    }

    return $list;
}

# 現在のトピックカウンター値を取得する
sub get_counter {
    my $self = shift;
    my $counter = 0;

    my $counterfile = "$self->{system_dir}/topiccounter.txt";
    if (-f $counterfile) {
        open my $counterfh, '<', $counterfile;
        my $data = <$counterfh>;
        chomp $data;
        if ($data ne '') {
            $counter = $data;
        }
        close $counterfh;
    }

    return $counter;
}

# トピックカウンター値を更新する
sub update_counter {
    my $self = shift;
    my $counter = shift;

    my $counterfile = "$self->{system_dir}/topiccounter.txt";
    open my $counterfh, '>', $counterfile;
    flock $counterfh, LOCK_EX;
    print {$counterfh} "$counter\n";
    flock $counterfh, LOCK_UN;
    close $counterfh;
}

# 更新情報をXMLファイルとして保存
sub output_xml {
    my $self = shift;
    my @idlist = @_;

    my $siteurl = $self->{docroot_dir} . "/";
    my $topic = Lib::Topic->new();

    my $itemlist = '';
    foreach my $topicid (@idlist) {
        my $ret = $topic->load($topicid, 0);
        next if (!$ret);

        my $title = $topic->get_title();
        my $topicurl = $siteurl . "?func=detail&amp;id=$topicid";

        $title =~ s/&/&amp;/g;
        $title =~ s/"/&quot;/g;
        $title =~ s/</&lt;/g;
        $title =~ s/>/&gt;/g;

        $itemlist .= <<"END_ITEM";
  <item>
    <title>$title</title>
    <link>$topicurl</link>
  </item>
END_ITEM
    }

    my $xml = <<"END_XML";
<?xml version="1.0" encoding="UTF-8" ?>
<items>
$itemlist
</items>
END_XML

    my $xmlfile = "update.xml";
    open my $xmlfh, '>', $xmlfile;
    flock $xmlfh, LOCK_EX;
    print {$xmlfh} $xml;
    flock $xmlfh, LOCK_UN;
    close $xmlfh;
}

# 更新情報ファイルを読んでRSS 2.0 ファイルを出力する
sub outputrss2 {
    my $self = shift;

    my @new_topics = $self->get_newtopics();
    my $rssitemnum = $self->{topic_num};
    $rssitemnum--;
    @new_topics = @new_topics[0 .. $rssitemnum];
    my $rssxml = $self->get_rss2(@new_topics);

    my $rssfile = "rss.xml";
    open my $rssfh, '>', $rssfile;
    flock $rssfh, LOCK_EX;
    print {$rssfh} $rssxml;
    flock $rssfh, LOCK_UN;
    close $rssfh;

#    $self->output_xml(@new_topics);
}

# 指定されたトピックIDの情報をRSS 2.0 形式に編集
sub get_rss2 {
    my $self = shift;
    my @idlist = @_;

    my $siteurl = $self->{docroot_dir} . "/";
    my $sitetitle = $self->{site_title};
    my $subtitle = $self->{sub_title};

    my $util = Lib::Util->new();
    $sitetitle = $util->tag_invalidate($sitetitle);
    $subtitle = $util->tag_invalidate($subtitle);

    my $topic = Lib::Topic->new();
    my $keywordobj = Lib::Keyword->new();

    my $dt = Lib::DateTime->new();
    my $last_builddate = $dt->local_datetime(3);

    my $itemlist = '';
    foreach my $topicid (@idlist) {
        my $ret = $topic->load($topicid, 0);
        next if (!$ret);

        my $lastdate = $topic->get_lastdate();
        my $title = $topic->get_title();
        my $text = $topic->get_text();
        my $editor = $topic->get_editor();
        my $commentnum = $topic->get_commentnum();
        my $trackbacknum = $topic->get_trackbacknum();
        my $linknum = $topic->get_linknum();
        my $keyword = $topic->get_keyword();
        my @keyworditems = split(/ /, $keyword);
        my $keywordlist = $keywordobj->list(@keyworditems);

        # 改行コードを削除
        $text =~ s/\n//g;

        # トピックのURL
        my $topicurl = $siteurl . "?func=detail&amp;id=$topicid";

        $text .= "<br />"; # 改行
        $text .= "<a href=\"$topicurl#link\" target=\"_blank\">Link($linknum)</a> | ";
        $text .= "<a href=\"$topicurl#trackback\" target=\"_blank\">Trackback($trackbacknum)</a> | ";
        $text .= "<a href=\"$topicurl#comment\" target=\"_blank\">Comment($commentnum)</a>";

        $text .= "<br /><br />";
        $text .= $keywordlist;

        # 関連リンクを追加
        my $linklist = $topic->get_linklist();
        $text .= $linklist;

        # サイトへのリンクを追加
        $text .= "<br /><a href=\"$siteurl\">$sitetitle - $subtitle</a>";

#        $title = $util->tag_invalidate($title);
        if ($editor ne '') {
            $text = $util->tag_invalidate($text);
        }

        $dt->set_datetime($lastdate);
        my $pubdate = $dt->format_datetime(3);

        $itemlist .= <<"END_ITEM";
  <item>
    <title>$title</title>
    <description>$text</description>
    <link>$topicurl</link>
    <pubDate>$pubdate</pubDate>
  </item>
END_ITEM
    }

    open my $templatefh, '<', "$self->{system_dir}/tmpl/rss2.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$SITEURL\$/$siteurl/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$PUBDATE\$/$last_builddate/g;
    $template =~ s/\$LASTBUILDDATE\$/$last_builddate/g;
    $template =~ s/\$ITEMLIST\$/$itemlist/g;

    return $template;
}

1;
# End of Archive.pm
