#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use URI::Escape;
use Encode;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::DateTime;
use Lib::String;
use Lib::Util;
use Lib::Filter;
use Lib::Mail;
use Lib::Topic;
use Lib::Archive;
use Lib::Search;
use Lib::Conf;
use Lib::Language;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $conf = Lib::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    if ($act eq 'exec_mobile') {
        _exec_mobile($cgi);
    }
    else {
        _view_mobile($cgi);
    }
}

sub _view_mobile() {
    my ($cgi) = @_;
    my $func = $cgi->param('func');

    # func: list, detail
    if ($func eq 'detail') {
        _show_detail($cgi);
    }
    else {
        _show_list($cgi);
    }
}

sub _exec_mobile() {
    my ($cgi) = @_;
    my $func = $cgi->param('func');

    # func: comment
    if ($func eq 'comment') {
        _add_comment($cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

# リストページの表示
sub _show_list() {
    my ($cgi) = @_;

#   $logger->write("_show_list");   # for debug

    my $q = $cgi->param('q');
    my $page = $cgi->param('page');
    my $util = Lib::Util->new();
    $q = $util->tag_invalidate($q);
    $q =~ s/　/ /g;

    # 入力チェック
    if ($page ne '' && $page !~ /\A[0-9]+\z/) {
        $error->print_error('00002e');
    }
    $page = ($page eq '') ? 1 : $page;
#    $logger->write("_show_list - $page:$page");   # for debug

    my $archive = Lib::Archive->new();

    my @items = ();
    if ($q eq'') {
        @items = $archive->get_newtopics();
    }
    else {
        # 検索
        if ($q =~ /\A[0-9\,]+\z/) { # 数字とコンマだけの場合はID指定
            @items = split(/\,/, $q);
            # データファイルが存在するかチェック
            my $datadir = $conf->get_data_dir();
            my @checkeditems = ();
            foreach my $item (@items) {
                my $filename = "$datadir/$item" . "tp.data";
                if (-f $filename) {
                    push @checkeditems, $item;
                }
            }
            @items = @checkeditems;
        }
        else {
            my $search = Lib::Search->new();
            @items = $search->search($q);
        }
        @items = reverse (sort { $a <=> $b; } @items); # 降順にソート
    }

    my $resulthtml = _mobile_html($q, $page, @items);

    print "Content-Type: text/html\n\n";
    print $resulthtml;
}

# mobileページのhtmlソース作成
sub _mobile_html {
    my $q = shift;
    my $page = shift;
    my @result = @_;

#    $logger->write("_mobile_html");   # for debug

    my $docrootdir = $conf->get_docroot_dir();
    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();
    my $language = Lib::Language->new();

    my $caption = $sitetitle;
    my $listtitle = $language->get_statictext('update-info');
    my $metadisc = "<meta name=\"description\" content=\"$subtitle\">";
    my $metakeywords = "<meta name=\"keywords\" content=\"bbs,forum\">";

    my $currentq = '';
    if ($q ne '') {
        $listtitle = $language->get_statictext('search-result');
        $listtitle =~ s/\$QUERY\$/$q/g;
        $caption = "$listtitle : $sitetitle";
        $metadisc = "<meta name=\"description\" content=\"$listtitle\">";
        $metakeywords = "<meta name=\"keywords\" content=\"$q\">";
        $currentq = uri_escape($q);
    }

    my $prevlist = $language->get_statictext('prev-list');
    my $nextlist = $language->get_statictext('next-list');

    my $resultlist = $language->get_message('not-found');
    my $pagenavi = '';
    if (scalar(@result) > 0) { # ページナビゲーションを作成

        # 総ページ数の算出
        my $itemnum = 5;
        my $pagenum = int(scalar(@result) / $itemnum); 
        if (scalar(@result) / $itemnum > $pagenum) {
            $pagenum++;
        }

        # 該当ページのユーザーIDを取り出す
        my @items = ();
        for my $i (0 .. $#result) {
            if (($i >= ($page - 1) * $itemnum) && ($i < $page * $itemnum)) {
                push @items, $result[$i];
            }
        }

        # トピック一覧を取得
        $resultlist = _get_topiclist($currentq, $page, @items);

        # ページナビゲーションの作成
        my $base = 1;
        my $currentnum = ($pagenum < 3) ? $pagenum : 3;
        if ($page > 3) {
            $base = $page - 2;
        }
        my @pagelist = ();
        for my $pagecount ($base .. ($base + $currentnum - 1)) {
            push @pagelist, $pagecount;
        }
        my $current = '';
        foreach my $pagecount (@pagelist) {
            if ($pagecount != $page) {
                $current .= "<a href=\"./?q=$currentq&page=$pagecount\">$pagecount</a>&nbsp;";
            }
            else {
                $current .= "$pagecount&nbsp;";
            }
        }
        my $prev = ''; 
        if ($page > 1) {
            my $prevpage = $page - 1;
            $prev .= "<a href=\"./?q=$currentq&page=$prevpage\">$prevlist</a>";
        }
        my $next = ''; 
        if ($page < $pagenum) {
            my $nextpage = $page + 1;
            $next = "<a href=\"./?q=$currentq&page=$nextpage\">$nextlist</a>";
        }
        $pagenavi = <<"END_PAGENAVI";
<p><strong>$prev&nbsp;&nbsp;$current&nbsp;$next</strong></p>
END_PAGENAVI
    }

    my $content = <<"END_CONTENT";
<p><b>$listtitle</b></p>
$resultlist
END_CONTENT

    my $searchbutton = $language->get_statictext('search');
    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/mobile.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$PREVQ\$/$q/g;
    $template =~ s/\$SEARCHBUTTON\$/$searchbutton/g;
    $template =~ s/\$CONTENT\$/$content/g;
    $template =~ s/\$PAGENAVI\$/$pagenavi/g;

    return $template;
}

sub _get_topiclist {
    my $q = shift;
    my $page = shift;
    my @items = @_;

#    $logger->write("_get_topiclist");   # for debug

    my $string = Lib::String->new();
    my $topic = Lib::Topic->new();

    my $topiclist = '';
    my $number = 1;
    foreach my $topicid (@items) {
        my $ret = $topic->load($topicid, 0);
        next if (!$ret);
        my $title = $topic->get_title();
        my $summary = $string->cut($title, 24);
        my $akey = "accesskey=\"$number\"";
        $topiclist .= <<"END_TOPICLIST";
$number.<a href="./?func=detail&id=$topicid&q=$q&page=$page" $akey>$summary</a><br />
END_TOPICLIST
        $number++;
    }

    return $topiclist;
}

# 詳細ページの表示
sub _show_detail {
    my ($cgi) = @_;

    my $topicid = $cgi->param('id');
    if ($topicid !~ /\A[0-9]+\z/) {
        $error->print_error('00002e');
    }
    my $q = $cgi->param('q');
    my $page = $cgi->param('page');
    my $util = Lib::Util->new();
    $q = $util->tag_invalidate($q);
    if ($page ne '' && $page !~ /\A[0-9]+\z/) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($topicid, 1);
    my $topicurl = "./?func=detail&id=$topicid";

    my $status = $topic->get_status();
    my $title = $topic->get_title();
    my $text = $topic->get_text();
    my $point = $topic->get_point();
    my $commentnum = $topic->get_commentnum();
#    my $trackbacknum = $topic->get_trackbacknum();
    my $linknum = $topic->get_linknum();
    my $lastdate = $topic->get_lastdate();

    my $url = $topic->get_url();
    my $bookmark = '';
    my $topictitle = $title;
    if ($url ne '') {
        $topictitle = "<a href=\"$url\">$title</a>";
    }

    my $language = Lib::Language->new();
    my $deletenotice = $language->get_statictext('delete-notice');
    if ($status == 9) { # ステータスが削除の場合
        $title = $deletenotice;
        $text = $deletenotice;
    }

    my $editor = $topic->get_editor();
    my $handle = $topic->get_handle();
    if ($editor eq '') {
        $editor = $handle;
        if ($editor eq '') {
            $editor = $conf->get_anonymous();
        }
    }
    my $enceditor = uri_escape($editor);
    $editor = "<a href=\"./?q=$enceditor\">$editor</a>";

    my $currentq = uri_escape($q);

    my @links = $topic->get_links( );
    my $linklist = _get_linklist($topicid, $currentq, $page, @links);
    if ($linklist ne '') {
        my $topiclink = $language->get_statictext('topic-link');
        $linklist = "<p><h4><a name=\"link\">$title$topiclink</a></h4>$linklist</p>";
    }

    my @comments = $topic->get_comments( );
    my $commentlist = _get_commentlist($topicid, $currentq, $page, @comments);
    my $commentform = '';
    if ($status == 1 || $status == 3) {
        $commentform = _comment_form($topicid);
    }

    my $topiccomment = $language->get_statictext('topic-comment');
    my $returnlist = $language->get_statictext('return-list');
    my $content = <<"END_CONTENT";
<h3>$topictitle</h3>
<p>
$text
</p>
<p>
[$topicid] Posted by $editor at <font class="datetime">$lastdate</font><br />
<a href="$topicurl#link">Link ($linknum)</a> | <a href="$topicurl#comment">Comment ($commentnum)</a>
</p>
$linklist
<p>
<h4><a name="comment" href="$topicurl">$title</a>$topiccomment</h4>
$commentlist
</p>
<p>
$commentform
</p>
<p>
<a href="?q=$currentq&page=$page">$returnlist</a>
</p>
END_CONTENT

    my $searchbutton = $language->get_statictext('search');
    my $systemdir = $conf->get_system_dir();
    my $sitetitle = $conf->get_site_title();
    my $caption = $sitetitle;
    my $pagenavi = _page_navi($topicid);

    open my $templatefh, '<', "$systemdir/tmpl/mobile.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$PREVQ\$/$q/g;
    $template =~ s/\$SEARCHBUTTON\$/$searchbutton/g;
    $template =~ s/\$CONTENT\$/$content/g;
    $template =~ s/\$PAGENAVI\$/$pagenavi/g;

    print "Content-Type: text/html\n\n";
    print $template;
}

sub _get_linklist {
    my $topicid = shift;
    my $q = shift;
    my $page = shift;
    my @linkdata = @_;

    my $list = '';
    my @links = @linkdata;
    for my $i (0 .. $#links) {
        my $link = $links[$i];
        my ($linkid, $title, $url, $summary) = split(/\,/, $link);

        $title =~ s/enc_conma/,/g;
        $url =~ s/enc_conma/,/g;

        $list .= <<"END_LIST";
<a href="$url">$title</a><br />
END_LIST
    }

    return $list;
}

# トピックのコメント一覧を作成する
sub _get_commentlist {
    my $topicid = shift;
    my $q = shift;
    my $page = shift;
    my @commentdata = @_;

    my $list = '';
    my $anonymous = $conf->get_anonymous();
    my $topicurl = "./?func=detail&id=$topicid&q=$q&page=$page";

    foreach my $comment (@commentdata) {
        my ($commentid, $status, $postdate, $author, $mailaddr, $hpurl, $text, $evaluate, $ipaddr) = split(/\,/, $comment);
        next if ($status == 2);

        # 匿名の置き換え
        if ($author eq '') {
            $author = $anonymous;
        }

        # 改行を削除
        $text =~ s/<br.+?\/>enc_crlf/enc_crlf/g;
        $text =~ s/enc_crlf/<br \/>/g;

        # コンマをデコード
        $text =~ s/enc_conma/,/g;

        # >>$commentidに対してリンクを張る。
        $text =~ s/>>([0-9]+)/<a href="$topicurl#comment$1">&gt;&gt;$1<\/a>/g;
        $text =~ s/&gt;&gt;([0-9]+)/<a href="$topicurl#comment$1">&gt;&gt;$1<\/a>/g;

        # URLをリンク
        my $util = Lib::Util->new();
        $text = $util->url_link($text, "");

        # メールアドレスやホームページURLが入力されていた場合はリンク
        if ($hpurl ne '') {
            $author = "<a href=\"$hpurl\">$author</a>";
        }
        else {
            if ($mailaddr ne '') {
                $author = "<a href=\"mailto:$mailaddr\">$author</a>";
            }
        }

        $list .= <<"END_LIST";
<p>
<a name="comment$commentid"></a><b>$commentid</b> posted by <b>$author</b> at $postdate</td><br />
</p>
<p>
$text
</p>
END_LIST
    }

    return $list;
}

# コメント入力用フォームの生成
sub _comment_form {
    my $topicid = shift;

    my $language = Lib::Language->new();
    my $recommend = $language->get_statictext('recommend');
    my $commentlabel = $language->get_statictext('comment');
    my $namelabel = $language->get_statictext('name');
    my $mailaddrlabel = $language->get_statictext('mail-address');
    my $postbutton = $language->get_statictext('post');

    my $form = <<"END_FORM";
<a name="commentform"></a>
<form name="showform" method="post" action="./">
<input type="hidden" name="act" value="comment_mobile">
<input type="hidden" name="topicid" value="$topicid">
$namelabel :<br />
<input type="text" name="author" size="30" value=""><br />
$mailaddrlabel :<br />
<input type="text" name="mailaddr" size="30"><br />
<input type="checkbox" name="evaluate" value="1">$recommend<br />
$commentlabel :<br />
<textarea name="text" rows="12" cols="21"></textarea>
<br />
<input type="submit" value="$postbutton">
</form>
END_FORM

    return $form;
}

# ページナビゲーション
sub _page_navi {
    my $topicid = shift;

    my $archive = Lib::Archive->new();
    my $lastid = $archive->get_counter();

    my $language = Lib::Language->new();
    my $prevpost = $language->get_statictext('prev-post');
    my $nextpost = $language->get_statictext('next-post');

    my $datadir = $conf->get_data_dir();
    my $prevpage = '';
    for (my $id = $topicid - 1; $id >= 0; $id--) {
        my $filename = "$datadir/$id" . "tp.data";
        if (-f $filename) {
            $prevpage = "<a href=\"./?func=detail&id=$id\">$prevpost</a>";
            last;
        }
    }
    my $nextpage = '';
    for (my $id = $topicid + 1; $id <= $lastid; $id++) {
        my $filename = "$datadir/$id" . "tp.data";
        if (-f $filename) {
            $nextpage = "<a href=\"./?func=detail&id=$id\">$nextpost</a>";
            last;
        }
    }

    my $pagenavi = "$prevpage&nbsp;&nbsp;$nextpage";
    return $pagenavi;
}

sub _add_comment() {
    my ($cgi) = @_;

    my $util = Lib::Util->new();
    my $dt = Lib::DateTime->new();

    my $commentdata;
    $commentdata->{topicid}  = $cgi->param('topicid');
    if ($commentdata->{topicid} !~ /\A[0-9]+\z/) {
        $error->print_error('00002e');
    }
    $commentdata->{status}   = 1;
    $commentdata->{postdate} = $dt->local_datetime(0);
    $commentdata->{author}   = $cgi->param('author');
    $commentdata->{mailaddr} = $cgi->param('mailaddr');
    $commentdata->{text}     = $cgi->param('text');
    $commentdata->{evaluate} = $cgi->param('evaluate');
    $commentdata->{ipaddr}   = $ENV{'REMOTE_ADDR'};
    if ($commentdata->{evaluate} eq '') {
        $commentdata->{evaluate} = 0;
    }

    _check_input($commentdata);

    $commentdata->{text} = $util->tag_invalidate($commentdata->{text});
    $commentdata->{text} =~ s/\r?\n/enc_crlf/g;
    $commentdata->{text} =~ s/,/enc_conma/g;
    $commentdata->{author} = $util->tag_invalidate($commentdata->{author});
    $commentdata->{author} =~ s/\r?\n//g;
    $commentdata->{author} =~ s/,/enc_conma/g;
    $commentdata->{mailaddr} = $util->tag_invalidate($commentdata->{mailaddr});
    $commentdata->{mailaddr} =~ s/\r?\n//g;
    $commentdata->{mailaddr} =~ s/,/enc_conma/g;

    my $language = Lib::Language->new();

    # スパムチェック
    if (_is_spam($commentdata->{ipaddr}, $commentdata->{text}, $commentdata->{postdate})) {
        _show_error($commentdata, $language->get_usererror('deny-post'));
    }

    my $topic = Lib::Topic->new();
    $topic->load($commentdata->{topicid}, 1);
    my $topic_status = $topic->get_status();
    if ($topic_status != 1 && $topic_status != 3) {
        _show_error($commentdata, $language->get_usererror('deny-post'));
    }

    if ($topic_status == 3) {
        $commentdata->{status} = 2;  # 承認待ちステータスを設定
    }

    my $editor = $topic->get_editor();
    my $ipaddr = $topic->get_ipaddr();

    # コメントIDの取得
    my @comments = $topic->get_comments();
    my $commentid = 0;
    if (scalar(@comments) > 0) {
        $commentid = (split(/\,/, $comments[$#comments]))[0];
    }
    $commentid++;

    # メール通知先をリストアップ
    my @mailinglist = ();
    foreach my $item (@comments) {
        my ($id, $mailaddr) = (split(/\,/, $item))[0,4];
        next if (!$id);
        next if (!$mailaddr);
        $mailaddr =~ s/enc_conma/,/g;
        push @mailinglist, $mailaddr;
    }

    my $commentrec = "$commentid,$commentdata->{status},$commentdata->{postdate},$commentdata->{author},";
    $commentrec .= "$commentdata->{mailaddr},,$commentdata->{text},";
    $commentrec .= "$commentdata->{evaluate},$commentdata->{ipaddr}";
    push @comments, $commentrec;
    $topic->set_comments(@comments);

    $topic->set_commentnum(scalar(@comments));
    my $point = $topic->get_point() ;
    if ($commentdata->{evaluate} > 0) {
        $point += $conf->get_good_point();
    }
    else {
        $point += $conf->get_normal_point();
    }
    $topic->set_point($point);
    $topic->update();

    $topic->get_html($conf->get_static_html(), 0);

    # おすすめの場合は更新情報に追加
    if (($commentdata->{evaluate} == 1) && ($ipaddr ne $commentdata->{ipaddr})) {
        my $archive = Lib::Archive->new();
        $archive->add($commentdata->{topicid});
    }

    # ログに保存
    _write_log($commentdata->{topicid}, $commentdata->{postdate},
            $commentdata->{ipaddr}, $commentdata->{author}, $commentdata->{text});

    my $topicurl = $conf->get_docroot_dir() . "/?func=detail&id=$commentdata->{topicid}";
    my $topictitle = $topic->get_title();

    $commentdata->{author} =~ s/enc_conma/,/g;
    $commentdata->{mailaddr} =~ s/enc_conma/,/g;
    $commentdata->{text} =~ s/enc_conma/,/g;
    $commentdata->{text} =~ s/enc_crlf/\n/g;

    my $postperson = $language->get_statictext('post-person');
    my $maillabel = $language->get_statictext('mail-address');
    my $commentlabel = $language->get_statictext('comment');
    my $commentinfo = $language->get_message('comment-info');
    my $commentinfobody1 = $language->get_message('comment-infobody1');
    my $commentinfobody2 = $language->get_message('comment-infobody2');
    my $commentinfobody3 = $language->get_message('comment-infobody3');

    $commentinfo =~ s/\$TOPICTITLE\$/$topictitle/g;
    $commentinfobody1 =~ s/\$TOPICTITLE\$/$topictitle/g;

    my $subject = $commentinfo;
    my $body = <<"END_BODY";
$commentinfobody1
$topicurl

----------

$postperson :
$commentdata->{author}

$maillabel :
$commentdata->{mailaddr}

$commentlabel :
$commentdata->{text}

---
$commentinfobody2
$commentinfobody3

----------
END_BODY

    # send comment information mail to user and webmaster.
    if ($editor ne '') {
        my $user = Lib::User->new();
        $user->load($editor);
        my $mailaddr = $user->get_mailaddr();
        push @mailinglist, $mailaddr;
    }

    my $adminaddr = $conf->get_admin_address();
    if ($conf->get_notify_mail() && $adminaddr) {
        push @mailinglist, $adminaddr;
    }

    my $mail = Lib::Mail->new();
    my %count;
    @mailinglist = grep {!$count{$_}++} @mailinglist;
    foreach my $mailaddr (@mailinglist) {
        $mail->send($subject, $body, $mailaddr);
    }

    print "Location: $topicurl", "\n\n";
}

sub _write_log() {
    my ($topicid, $postdate, $ipaddr, $author, $text) = @_;
    my $yyyy = substr($postdate, 0, 4);
    my $mm = substr($postdate, 5, 2);
    my $dd = substr($postdate, 8, 2);
    my $systemdir = $conf->get_system_dir();
    my $logfile = "$systemdir/log/c_$yyyy$mm$dd.log";
    my $datarec = "$topicid,$postdate,$ipaddr,$author,$text";

    open my $lfh, '>>', $logfile;
    print {$lfh} "$datarec\n";
    close $lfh;
}

sub _is_spam() {
    my ($ip, $cmtext, $cmdate) = @_;

    my $limit = 3;
    my $iplimit = 12;
    my $yyyy = substr($cmdate, 0, 4);
    my $mm = substr($cmdate, 5, 2);
    my $dd = substr($cmdate, 8, 2);
    my $systemdir = $conf->get_system_dir();
    my $logfile = "$systemdir/log/c_$yyyy$mm$dd.log";
    my $count = 0;
    my $ipcount = 0;
    if (-f $logfile) {
        open my $lfh, '<', $logfile;
        while (my $data = <$lfh>) {
            last if ($count >= $limit);
            last if ($ipcount >= $iplimit);
            chomp $data;
            my ($topicid, $postdate, $ipaddr, $author, $text) = split(/\,/, $data);
            if ($ip eq $ipaddr) {
                $ipcount++;
                if (substr($cmtext, 0, 10) eq substr($text, 0, 10)) {
                    $count++;
                }
            }
        }
        close $lfh;
    }

    if (($count >= $limit) || ($ipcount >= $iplimit)) {
        my $denyrec = "deny from $ip";
        my $accessfile = ".htaccess";
        open my $afh, '>>', $accessfile;
        print {$afh} "$denyrec\n";
        close $afh;
        return 1; # is spam
    }

    return 0; # no sapm
}

sub _check_input() {
    my ($commentdata) = @_;

    my $filter = Lib::Filter->new();
    my $docrootdir = $conf->get_docroot_dir();

    my $language = Lib::Language->new();

    my $referer = $ENV{'HTTP_REFERER'};
    if ($referer !~ /\A$docrootdir/) {
        _show_error($commentdata, $language->get_usererror('deny-post'));
    }

    if ($filter->check_filter($commentdata->{ipaddr})) {
        _show_error($commentdata, $language->get_usererror('deny-post'));
    }

    if ($commentdata->{author} ne '') {
        $commentdata->{author} =~ s/[\r\n]//g;
        if ($filter->check_filter($commentdata->{author})) {
            _show_error($commentdata, $language->get_usererror('deny-post'));
        }
    }

    if ($commentdata->{mailaddr} ne '') {
        $commentdata->{mailaddr} =~ s/[\r\n]//g;
        if ($commentdata->{mailaddr} !~ /.+\@.+[.].+/) {
            _show_error($commentdata, $language->get_usererror('mailaddr-format'));
        }
        if ($filter->check_filter($commentdata->{mailaddr})) {
            _show_error($commentdata, $language->get_usererror('deny-post'));
        }
    }

    if ($commentdata->{text} eq '') {
        _show_error($commentdata, $language->get_usererror('input-comment'));
    }
    else {
        my $langcode = $language->get_langcode();
        if (($langcode eq 'ja') && ($commentdata->{text} =~ /\A[\x20-\x7E\r\n]+\z/)) {
            _show_error($commentdata, $language->get_usererror('deny-post'));
        }
        if ($commentdata->{text} =~ /<[aA] .+>.*<\/[aA]>/) {
            _show_error($commentdata, $language->get_usererror('deny-post'));
        }
        if ($filter->check_filter($commentdata->{text})) {
            _show_error($commentdata, $language->get_usererror('deny-post'));
        }
    }
}

sub _show_error() {
    my ($commentdata, $errormsg) = @_;

    my $caption = $errormsg;

    my $msg = "<p><b><font color=\"red\">$errormsg</font></b></p>";

    my $systemdir = $conf->get_system_dir();
    my $sitetitle = $conf->get_site_title();

    my $language = Lib::Language->new();
    my $searchbutton = $language->get_statictext('search');
    my $content = <<"END_CONTENT";
$msg
END_CONTENT

    open my $templatefh, '<', "$systemdir/tmpl/mobile.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$PREVQ\$//g;
    $template =~ s/\$SEARCHBUTTON\$/$searchbutton/g;
    $template =~ s/\$CONTENT\$/$content/g;
    $template =~ s/\$PAGENAVI\$//g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
