#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2010 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

package Lib::Topic;

use strict;
use warnings;

use constant {
    LOCK_SH => 1,
    LOCK_EX => 2,
    LOCK_NB => 4,
    LOCK_UN => 8,
};

use URI::Escape;

use Lib::Logger;
use Lib::User;
use Lib::DateTime;
use Lib::String;
use Lib::Util;
use Lib::Calendar;
use Lib::Ads;
use Lib::Topic;
use Lib::Archive;
use Lib::Keyword;
use Lib::Conf;
use Lib::Language;

my $logger = Lib::Logger->new();

# コンストラクタ
sub new {
    my $self = {};

    use Lib::Conf;
    my $conf = Lib::Conf->new();

    $self->{system_dir} = $conf->get_system_dir();
    $self->{data_dir} = $conf->get_data_dir();
    $self->{docroot_dir} = $conf->get_docroot_dir();
    $self->{files_dir} = $conf->get_files_dir();
    $self->{site_title} = $conf->get_site_title();
    $self->{sub_title} = $conf->get_sub_title();
    $self->{link_target} = $conf->get_link_target();
    $self->{trackback_target} = $conf->get_trackback_target();
    $self->{comment_target} = $conf->get_comment_target();
    $self->{bookmark_target} = $conf->get_bookmark_target();
    $self->{anonymous} = $conf->get_anonymous();
    $self->{static_html} = $conf->get_static_html();
    $self->{free_signup} = $conf->get_free_signup();

    $self->{topicid} = '';    # トピックID
    $self->{lastdate} = '';   # 最終更新日 YYYY/MM/DD HH:MM:SS
    $self->{status} = 0;      # 状態
    $self->{editor} = '';     # 編集者
    $self->{handle} = '';     # ハンドル名
    $self->{title} = '';      # タイトル
    $self->{text} = '';       # 本文
    $self->{commentnum} = 0;  # コメント数
    $self->{trackbacknum} = 0;# トラックバック数
    $self->{linknum} = 0;     # 関連リンク数
    $self->{ad1} = '';        # 広告コード1（幅468、記事上）
    $self->{ad2} = '';        # 広告コード2（幅468、記事下）
    $self->{ad3} = '';        # 広告コード3（幅300、サイドバー）
    $self->{point} = 0;       # 累計ポイント
    $self->{keyword} = '';    # キーワード
    $self->{url} = '';        # ブックマークURL
    $self->{ipaddr} = '';     # 投稿者のIPアドレス

    $self->{links} = [];      # 関連リンクの配列
    $self->{comments} = [];   # コメントの配列
    $self->{trackbacks} = []; # トラックバックの配列

    bless($self);
    return $self;
}

# トピックIDの取得
sub get_topicid {
    my $self = shift;
    return $self->{topicid};
}

# トピックIDの設定
sub set_topicid {
    my $self = shift;
    $self->{topicid} = shift;
}

# トピックの最終更新日の取得
sub get_lastdate {
    my $self = shift;
    return $self->{lastdate};
}

# トピックの最終更新日の設定
sub set_lastdate {
    my $self = shift;
    $self->{lastdate} = shift;
}

# 状態の取得
sub get_status {
    my $self = shift;
    return $self->{status};
}

# 状態の設定
sub set_status {
    my $self = shift;
    $self->{status} = shift;
}

# 編集者の取得
sub get_editor {
    my $self = shift;
    return $self->{editor};
}

# 編集者の設定
sub set_editor {
    my $self = shift;
    $self->{editor} = shift;
}

# ハンドル名の取得
sub get_handle {
    my $self = shift;
    return $self->{handle};
}

# ハンドル名の設定
sub set_handle {
    my $self = shift;
    $self->{handle} = shift;
}

# タイトルの取得
sub get_title {
    my $self = shift;
    return $self->{title};
}

# タイトルの設定
sub set_title {
    my $self = shift;
    $self->{title} = shift;
}

# 本文の取得
sub get_text {
    my $self = shift;
    return $self->{text};
}

# 本文の設定
sub set_text {
    my $self = shift;
    $self->{text} = shift;
}

# コメント数の取得
sub get_commentnum {
    my $self = shift;
    return $self->{commentnum};
}

# コメント数の設定
sub set_commentnum {
    my $self = shift;
    $self->{commentnum} = shift;
}

# トラックバック数の取得
sub get_trackbacknum {
    my $self = shift;
    return $self->{trackbacknum};
}

# トラックバック数の設定
sub set_trackbacknum {
    my $self = shift;
    $self->{trackbacknum} = shift;
}

# 関連リンク数の取得
sub get_linknum {
    my $self = shift;
    return $self->{linknum};
}

# 関連リンク数の設定
sub set_linknum {
    my $self = shift;
    $self->{linknum} = shift;
}

# 広告コード1の取得
sub get_ad1 {
    my $self = shift;
    return $self->{ad1};
}

# 広告コード1の設定
sub set_ad1 {
    my $self = shift;
    $self->{ad1} = shift;
}

# 広告コード2の取得
sub get_ad2 {
    my $self = shift;
    return $self->{ad2};
}

# 広告コード2の設定
sub set_ad2 {
    my $self = shift;
    $self->{ad2} = shift;
}

# 広告コード3の取得
sub get_ad3 {
    my $self = shift;
    return $self->{ad3};
}

# 広告コード3の設定
sub set_ad3 {
    my $self = shift;
    $self->{ad3} = shift;
}

# ポイントの取得
sub get_point {
    my $self = shift;
    return $self->{point};
}

# ポイントの設定
sub set_point {
    my $self = shift;
    $self->{point} = shift;
}

# キーワードの取得
sub get_keyword {
    my $self = shift;
    return $self->{keyword};
}

# キーワードの設定
sub set_keyword {
    my $self = shift;
    $self->{keyword} = shift;
}

# ブックマークURLの取得
sub get_url {
    my $self = shift;
    return $self->{url};
}

# ブックマークURLの設定
sub set_url {
    my $self = shift;
    $self->{url} = shift;
}

# IPアドレスの取得
sub get_ipaddr {
    my $self = shift;
    return $self->{ipaddr};
}

# IPアドレスの設定
sub set_ipaddr {
    my $self = shift;
    $self->{ipaddr} = shift;
}

# 関連リンクの取得
sub get_links {
    my $self = shift;
    return @{ $self->{links} };
}

# 関連リンクの設定
sub set_links {
    my $self = shift;
    @{ $self->{links} } = @_;
}

# コメントの取得
sub get_comments {
    my $self = shift;
    return @{ $self->{comments} };
}

# コメントの設定
sub set_comments {
    my $self = shift;
    @{ $self->{comments} } = @_;
}

# トラックバックの取得
sub get_trackbacks {
    my $self = shift;
    return @{ $self->{trackbacks} };
}

# トラックバックの設定
sub set_trackbacks {
    my $self = shift;
    @{ $self->{trackbacks} } = @_;
}

# トピックデータの読込み
sub load {
    my $self = shift;
    my ($topicid, $mode) = @_;

    # トピックデータを読み込む
    my $datafile = "$self->{data_dir}/$topicid" . "tp.data";
    return $self->read($datafile, $mode);
}

# ファイル名指定して読込み
sub read {
    my $self = shift;
    my ($datafile, $mode) = @_;
    my $result = 0;

    # トピックデータを読み込む
    if (-f $datafile) {  # ファイルが存在する場合
        my @links = ();
        my @comments = ();
        my @trackbacks = ();

        open my $datafh, '<', $datafile;
        while (my $data = <$datafh>) {
            chomp($data);
            if ($data ne '') {
                my $rectype = substr($data, 0, 1);
                if ($rectype == 0) {    # トピックデータ
                    ($self->{topicid},
                     $self->{lastdate},
                     $self->{status},
                     $self->{editor},
                     $self->{handle},
                     $self->{title},
                     $self->{text},
                     $self->{commentnum},
                     $self->{linknum},
                     $self->{trackbacknum},
                     $self->{ad1},
                     $self->{ad2},
                     $self->{ad3},
                     $self->{point},
                     $self->{keyword},
                     $self->{url},
                     $self->{ipaddr}) = (split(/\,/, $data))[1..17];
                }
                elsif ($rectype == 2) { # 関連リンク
                    push @links, substr($data, 2);
                }
                elsif ($rectype == 3) { # コメント
                    push @comments, substr($data, 2);
                }
                elsif ($rectype == 4) { # トラックバック
                    push @trackbacks, substr($data, 2);
                }
            }
            last if (!$mode);
        }
        close $datafh;
        $result = 1;

        @{ $self->{links} } = @links;
        @{ $self->{comments} } = @comments;
        @{ $self->{trackbacks} } = @trackbacks;

        $self->{title}   =~ s/enc_conma/,/g;
        $self->{text}    =~ s/enc_conma/,/g;
        $self->{text}    =~ s/enc_crlf/\n/g;

        $self->{url} =~ s/enc_conma/,/g;

        $self->{ad1} =~ s/enc_conma/,/g;
        $self->{ad2} =~ s/enc_conma/,/g;
        $self->{ad3} =~ s/enc_conma/,/g;
        $self->{ad1} =~ s/enc_crlf/\n/g;
        $self->{ad2} =~ s/enc_crlf/\n/g;
        $self->{ad3} =~ s/enc_crlf/\n/g;
    }

    return $result;
}

# トピックデータの更新（必ずmode='1'で読み込んでから更新する）
sub update {
    my $self = shift;

    my $datafile = "$self->{data_dir}/$self->{topicid}tp.data";

    my @links = @{ $self->{links} };
    my @comments = @{ $self->{comments} };
    my @trackbacks = @{ $self->{trackbacks} };

    $self->{title}   =~ s/,/enc_conma/g;
    $self->{text}    =~ s/,/enc_conma/g;
    $self->{text}    =~ s/\r?\n/enc_crlf/g;

    $self->{url} =~ s/,/enc_conma/g;

    $self->{ad1} =~ s/,/enc_conma/g;
    $self->{ad2} =~ s/,/enc_conma/g;
    $self->{ad3} =~ s/,/enc_conma/g;
    $self->{ad1} =~ s/\r?\n/enc_crlf/g;
    $self->{ad2} =~ s/\r?\n/enc_crlf/g;
    $self->{ad3} =~ s/\r?\n/enc_crlf/g;

    $self->{commentnum} = scalar(@comments);
    $self->{linknum} = scalar(@links);
    $self->{trackbacknum} = scalar(@trackbacks);

    my $topicdata = "0,$self->{topicid},$self->{lastdate},$self->{status},$self->{editor},$self->{handle},";
    $topicdata .= "$self->{title},$self->{text},";
    $topicdata .= "$self->{commentnum},$self->{linknum},$self->{trackbacknum},";
    $topicdata .= "$self->{ad1},$self->{ad2},$self->{ad3},";
    $topicdata .= "$self->{point},$self->{keyword},$self->{url},$self->{ipaddr}";

    # トピックデータファイルを更新
    open my $datafh, '>', $datafile;
    flock $datafh, LOCK_EX;
    print {$datafh} "$topicdata\n";

    if (scalar(@links) > 0) { # 関連リンク
        foreach my $link (@links) {
            print {$datafh} "2,$link\n";
        }
    }
    if (scalar(@comments) > 0) { # コメント
        foreach my $comment (@comments) {
            print {$datafh} "3,$comment\n";
        }
    }
    if (scalar(@trackbacks) > 0) { # トラックバック
        foreach my $trackback (@trackbacks) {
            print {$datafh} "4,$trackback\n";
        }
    }
    flock $datafh, LOCK_UN;
    close $datafh;

    chmod 0766, $datafile;

    if (($self->{status} != 0) && ($self->{status} != 9)) {
        $self->output_html();
    }
}

sub output_html {
    my $self = shift;
    if ($self->{static_html}) {
        $self->get_html($self->{static_html}, '');
    }
}

# トピック詳細ページのhtmlソース作成
sub get_html {
    my $self = shift;
    my ($static, $session_userid) = @_;

    my $docrootdir = $self->{docroot_dir};
    my $sitetitle = $self->{site_title};
    my $subtitle = $self->{sub_title};

    my $topicid = $self->{topicid};
    $self->load($topicid, 1);

    my $language = Lib::Language->new();
    if ($self->{status} == 9) {
        $self->{title} = $language->get_statictext('delete-notice');
        $self->{text} = $language->get_statictext('delete-notice');
    }

    my $topictitle = $self->{title};
    my $pagetitle = $topictitle;
    $topictitle =~ s/&amp;/&/g;
    my $caption = $topictitle;

    # トピックHTMLファイル名を生成
    my $permalink = "$docrootdir/?func=detail&id=$topicid";
    my $rssurl = "$docrootdir/rss.xml";
    my $trackbackurl = "$docrootdir/rcvtb.cgi/$topicid";

    my $signup = $language->get_statictext('signup');
    my $subscribe = $language->get_statictext('subscribe');
#    my $post = $language->get_statictext('post');
    my $post = $language->get_statictext('new-thread');
    my $setting = $language->get_statictext('setting');
    my $manage = $language->get_statictext('manage');
    my $login = $language->get_statictext('login');
    my $logout = $language->get_statictext('logout');

    my $editlabel = $language->get_statictext('edit');
    my $linklabel = $language->get_statictext('link');

    my $usersignup = '';
    if ($self->{free_signup}) {
        $usersignup = "<a class=\"menu\" href=\"./?func=signup\">$signup</a>&nbsp;";
    }

    my $user = Lib::User->new();
    my $mode = 0;
    if ($session_userid ne '') {
        $mode = (($session_userid eq $self->{editor}) || ($user->is_admin($session_userid))) ? 1 : 2;
    }

    my $menu = <<"END_MENU";
<a class="menu" href="$rssurl">$subscribe</a>&nbsp;<a class="menu" href="./?act=new_post&topicid=$topicid">$post</a>&nbsp;
END_MENU
    if ($static) {
        $menu .= "$usersignup<a class=\"menu\" href=\"./?func=login\">$login</a>";
    }
    else {
        if ($mode == 1) { # 編集あり
            $menu = <<"END_MENU";
<a class="menu" href="./?act=view_editor&topicid=$topicid">$post</a>&nbsp;
<a class="menu" href="./?act=view_setting">$setting</a>&nbsp;
<a class="menu" href="./?act=exec_main&func=logout">$logout</a>
END_MENU
            if ($user->is_admin($session_userid)) {
                $menu = <<"END_MENU";
<a class="menu" href="./?act=view_editor&topicid=$topicid">$post</a>&nbsp;
<a class="menu" href="./?act=view_setting">$setting</a>&nbsp;
<a class="menu" href="./?act=view_manage">$manage</a>&nbsp;
<a class="menu" href="./?act=exec_main&func=logout">$logout</a>
END_MENU
            }
        }
        elsif ($mode == 2) { # 編集権なし
            $menu = <<"END_MENU";
<a class="menu" href="./?act=view_editor&topicid=$topicid">$post</a>&nbsp;
<a class="menu" href="./?act=view_setting">$setting</a>&nbsp;
<a class="menu" href="./?act=exec_main&func=logout">$logout</a>
END_MENU
        }
        else { # 閲覧
            $menu .= "$usersignup<a class=\"menu\" href=\"./?func=login\">$login</a>";
        }
    }

    my $editmenu = '';
    if (!$static && $mode == 1) {
        $editmenu .= <<"END_EDITMENU";
&nbsp;<a href="./?act=view_editor&func=edittopic&topicid=$topicid">$editlabel</a>&nbsp;
<a href="./?act=view_editor&func=newlink&topicid=$topicid">$linklabel</a>
END_EDITMENU
    }

    my $topicdetail = $self->_topic_detail($editmenu);  # トピック詳細を取得
    my $linklist = $self->_link_list($mode);            # 関連リンク一覧を取得
    my $trackbacklist = $self->_trackback_list($mode);  # トラックバック一覧を取得
    my $commentlist = $self->_comment_list($mode);      # コメント一覧を取得

    my $keywordlist ='';
    my $text = $self->{text};
    my $keyword = $self->{keyword};
    my $lastdate = $self->{lastdate};

    # メタタグ"description"の設定
    my $string = Lib::String->new();
    $string->set_tail('');
    my $descstr = $string->cut($text, 250);
    my $metadisc = "<meta name=\"description\" content=\"$descstr\">";

    my $metakeywords = '';
    my $keywordobj = Lib::Keyword->new();

    # 関連キーワードをメタタグの"keywords"を設定
    if ($keyword ne '') {
        $metakeywords = $keyword;
        $metakeywords =~ s/[ ]/,/g;
        $metakeywords = "<meta name=\"keywords\" content=\"$metakeywords\">";

        # キーワードリストの作成
        my @keywords = split(/ /, $keyword);
        $keywordlist = $keywordobj->list(@keywords);
    }

    # ホームページへのリンク
    my $homepage = '';
    my $editor = $self->{editor};
    if ($editor ne '') {
        my $user = Lib::User->new();
        $user->load($editor);
        my $hpurl = $user->get_hpurl();
        my $username = $user->get_username();
        if ($username ne '' && $hpurl ne '') {
            $homepage = "<p><a href=\"$hpurl\" target=\"_blank\">$username" . $language->get_statictext('user-homepage') . "</a></p>";
        }
    }

    # カレンダーを作成
    my ($year, $month, $day) = (split(/\//, substr($lastdate, 0, 10)));
    my $cal = Lib::Calendar->new();
    my $calendar = $cal->get_calendar($year, $month);

    # 最新アーカイブ、更新情報一覧を取得
    my $archive = Lib::Archive->new();
    my $archivelist = $archive->archive_list();

    # 広告の設定
    my $ad1 = $self->{ad1};
    my $ad2 = $self->{ad2};
    my $ad3 = $self->{ad3};
    my $status = $self->{status};

    my $ads = Lib::Ads->new();
    my $blogparts = '';
    if ($editor ne '') {
        my $user = Lib::User->new();
        $user->load($editor);
        if ($ad1 eq '') {
            $ad1 = $user->get_ad1();
        }
        if ($ad2 eq '') {
            $ad2 = $user->get_ad2();
        }
        if ($ad3 eq '') {
            $ad3 = $user->get_ad3();
        }
        $blogparts = $user->get_parts();
    }
    else {
        if ($ad1 eq '') {
            $ad1 = $ads->get_ad1();
        }
        if ($ad2 eq '') {
            $ad2 = $ads->get_ad2();
        }
        if ($ad3 eq '') {
            $ad3 = $ads->get_ad3();
        }
    }
    my $adlist = $ads->get_adlist_table();

    my $clouds = $keywordobj->clouds(1);

    my $pagenavi = $self->_page_navi($topicid);

    my $systemdir = $self->{system_dir};

    my $homelabel = $language->get_statictext('home');

    $string->set_tail('...');
    my $pathtitle = $string->cut($topictitle, 50);
    my $path = <<"END_PATH";
<a href="./" class="path">$homelabel</a>&nbsp;
&gt;&nbsp;
$pathtitle
END_PATH

    my $commentform = '';
    if ($status == 1 || $status == 3) {
        $commentform = $self->_comment_form();
    }

    my $topictrackback = $language->get_statictext('topic-trackback');
    my $trackbacklabel = $language->get_statictext('trackback-url');
    my $topiccomment = $language->get_statictext('topic-comment');

    my $maincontent = <<"END_MAINCONTENT";
$topicdetail
<br />

$keywordlist

$homepage

<center>
<p>
$ad2
</p>
</center>

$linklist

<br />
<h3><a name="trackback">$topictrackback</a></h3>
<p><font size="-1"><b>$trackbacklabel :</b></font><input type="text" size="60" value="$trackbackurl" readonly onfocus="this.select()"></p>
$trackbacklist

<h3><a name="comment">$topiccomment</a></h3>
$commentlist
<br />

$commentform
END_MAINCONTENT

    my $searchbutton = $language->get_statictext('search');
    my $updateinfo = $language->get_statictext('update-info');

    open my $templatefh, '<', "$systemdir/tmpl/main.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$METADISC\$/$metadisc/g;
    $template =~ s/\$METAKEYWORDS\$/$metakeywords/g;
    $template =~ s/\$RSSURL\$/$rssurl/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MENU\$/$menu/g;
    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$PREVQ\$//g;
    $template =~ s/\$SEARCHBUTTON\$/$searchbutton/g;
    $template =~ s/\$MAIN\$/$maincontent/g;
    $template =~ s/\$PAGENAVI\$/$pagenavi/g;
    $template =~ s/\$AD1\$/$ad1/g;
    $template =~ s/\$AD2\$//g;
    $template =~ s/\$AD3\$/$ad3/g;
    $template =~ s/\$ADLIST\$/$adlist/g;
    $template =~ s/\$CALENDAR\$/$calendar/g;
    $template =~ s/\$ARCHIVELIST\$/$archivelist/g;
    $template =~ s/\$CLOUDS\$/$clouds/g;
    $template =~ s/\$BLOGPARTS\$/$blogparts/g;
    $template =~ s/\$UPDATEINFO\$/$updateinfo/g;
#    $template =~ s/\$RECENTTRACKBACK\$/$recenttrackback/g;
#    $template =~ s/\$RECENTCOMMENT\$/$recentcomment/g;

    if ($static) {
        $template =~ s/\$AUTHOR\$//g;
        $template =~ s/\$HPURL\$//g;

        # HTMLファイルを出力
        my $topicurl = "detail$topicid.html";
        open my $htmlfh, '>', $topicurl;
        flock $htmlfh, LOCK_EX;
        print {$htmlfh} $template;
        flock $htmlfh, LOCK_UN;
        close $htmlfh;
        chmod 0766, $topicurl;
    }

    return $template;
}

# トピック詳細を作成し、作成したHTMLを返す、loadが前の処理で呼び出されていること
sub _topic_detail {
    my $self = shift;
    my $editmenu = shift;

    my $topicid = $self->{topicid};
    my $topicurl = "./?func=detail&id=$topicid";
    my $status = $self->{status};
    my $text = $self->{text};
    my $topictitle = $self->{title};
    my $lastdate = $self->{lastdate};
    my $commentnum = $self->{commentnum};
    my $trackbacknum = $self->{trackbacknum};
    my $linknum = $self->{linknum};
    my $point = $self->{point};
    my $editor = $self->{editor};
    my $handle = $self->{handle};
    my $url = $self->{url};

#    $logger->write("_topic_detail");   # for debug

    if ($url ne '') {
        $topictitle = "<div class=\"hreview\"><h2><span class=\"item\"><a class=\"url fn\" href=\"$url\" $self->{bookmark_target}>$topictitle</a></span></h2>";
    }
    else {
        $topictitle = "<div><h2>$topictitle</h2>";
    }

    if ($editor eq '') {
        $editor = $handle;
        if ($editor eq '') {
            $editor = $self->{anonymous};
            $editor = "<a href=\"./\">$editor</a>";
        }

        my $util = Lib::Util->new();
        $text = $util->url_link($text, "target=\"_blank\" rel=\"nofollow\"");

        my $keyword = Lib::Keyword->new();
        $text = $keyword->link_keyword($self->{keyword}, $text);
    }
    else {
        my $enceditor = uri_escape($editor);
        $editor = "<a href=\"./?q=$enceditor\">$editor</a>";
    }

    my $language = Lib::Language->new();
    my $statusline = $language->get_statictext('delete-status');
    if ($status == 0) {
        $statusline = "<font color=\"Orange\">" . $language->get_statictext('draft-status') . "</font>";
    }
    elsif ($status == 1) {
        $statusline = "<font color=\"Darkgreen\">" . $language->get_statictext('open-status') . "</font>";
    }
    elsif ($status == 2) {
        $statusline = "<font color=\"Darkred\">" . $language->get_statictext('close-status') . "</font>";
    }
    elsif ($status == 3) {
        $statusline = "<font color=\"Darkblue\">" . $language->get_statictext('limit-status') . "</font>";
    }

    my $detail = <<"END_DETAIL";
$topictitle
$text
<div>
[<a href="./?q=id:$topicid">$topicid</a>] Posted by <b>$editor</b> at <font class="datetime">$lastdate</font> $editmenu<br/>
$statusline |
<b><font class="point" size="+1">$point</font> point</b> |
<a href="$topicurl#link">Link ($linknum)</a> | 
<a href="$topicurl#trackback">Trackback ($trackbacknum)</a> | 
<a href="$topicurl#comment">Comment ($commentnum)</a>
</div>
END_DETAIL

    return $detail;
}

# トピックの関連リンクを取得
sub get_linklist {
    my $self = shift;

    my $list = $self->_link_list(0);

    return $list;
}

# トピックの関連リンク一覧を作成する
sub _link_list {
    my $self = shift;
    my ($mode) = @_;

    my $list = '';
    my $docrootdir = $self->{docroot_dir} . "/";
    my $link_target = $self->{link_target};
    my $topicid = $self->{topicid};

    my $language = Lib::Language->new();
    my $topiclink = $language->get_statictext('topic-link');

    my @links = @{ $self->{links} };
    for my $i (0 .. $#links) {
        my $link = $links[$i];
        my ($linkid, $title, $url, $summary) = split(/\,/, $link);

        $url =~ s/\A(detail[0-9][.]html)\z/$docrootdir$1/g;
        $url =~ s/\A(\.\/\?func=detail.+)\z/$docrootdir$1/g;

        $summary =~ s/enc_crlf/<br \/>/g;

        $title =~ s/enc_conma/,/g;
        $url =~ s/enc_conma/,/g;
        $summary =~ s/enc_conma/,/g;

        if ($summary !~ m/<a .+>.+<\/a>/i) {
            # キーワードをリンク
            my $keyword = Lib::Keyword->new();
            $summary = $keyword->link_keyword($self->{keyword}, $summary, 0);
        }

        my $menu = '';
        if ($mode == 1) {   # 編集モード
            my $edit = $language->get_statictext('edit');
            my $delete = $language->get_statictext('delete');
            $menu = <<"END_MENU";
<a href="./?act=view_editor&func=editlink&topicid=$topicid&linkid=$linkid">$edit</a>&nbsp;&nbsp;<a href="./?act=view_editor&func=dellink&topicid=$topicid&id=$linkid">$delete</a>
END_MENU
        }

        $list .= <<"END_LIST";
<table border="0" width="100%" cellpadding="0" cellspacing="0">
<tr>
  <td><a href="$url" $link_target>$title</a></td>
  <td width="90" align="right">$menu</td></tr>
<tr>
  <td colspan="2">$summary</td>
</tr>
</table>
END_LIST
    }

    if ($list ne '') {
       $list = <<"END_LIST";
<br />
<h3><a name="link">$topiclink</a></h3>
$list
END_LIST
    }

    return $list;
}

# トピックのトラックバック一覧を作成する
sub _trackback_list {
    my $self = shift;
    my ($mode) = @_;

    my $list = '';
    my $trackback_target = $self->{trackback_target};
    my $topicid = $self->{topicid};

    my $language = Lib::Language->new();

    my @trackbacks = @{ $self->{trackbacks} };
    for my $i (0 .. $#trackbacks) {
        my $trackback = $trackbacks[$i];
        my ($trackbackid, $status, $recvdate, $url, $blogname, $title, $excerpt, $ipaddr) = split(/\,/, $trackback);

        # ステータスが無効なトラックバックは表示しない
        next if ($status == 0);

        # 編集モード以外ではステータスが承認待ちのトラックバックは表示しない
        next if ($mode != 1 && $status == 2);

        # コンマをデコード
        $title =~ s/enc_conma/,/g;
        $excerpt =~ s/enc_conma/,/g;
        $url =~ s/enc_conma/,/g;
        $blogname =~ s/enc_conma/,/g;

        # キーワードをリンク
        my $keyword = Lib::Keyword->new();
        $excerpt = $keyword->link_keyword($self->{keyword}, $excerpt, 0);

        my $menu = '';
        if ($mode == 1) {   # 編集モード
            my $delete = $language->get_statictext('delete');
            my $validate = $language->get_statictext('validate');
            $menu = "<a href=\"./?act=view_editor&func=deltb&topicid=$topicid&id=$trackbackid\">$delete</a>";
            if ($status == 2) {
                $menu .= "&nbsp;&nbsp;<a href=\"./?act=exec_editor&func=admittb&topicid=$topicid&id=$trackbackid\">$validate</a>";
            }
        }

        $list .= <<"END_LIST";
<table border="0" width="100%" cellpadding="0" cellspacing="0">
<tr>
  <td><a name="trackback$trackbackid"></a>Linked from <b>$blogname</b> : <a href="$url" $trackback_target>$title</a> at $recvdate</td>
  <td width="90" align="right" nowrap>$menu</td></tr>
<tr>
  <td colspan="2">$excerpt</td>
</tr>
</table>
END_LIST
    }

    return $list;
}

# トピックのコメント一覧を作成する
sub _comment_list {
    my $self = shift;
    my ($mode) = @_;

    my $list = '';
    my $comment_target = $self->{comment_target};
    my $anonymous = $self->{anonymous};
    my $topicid = $self->{topicid};
    my $topicurl = "./?func=detail&id=$topicid";

    my $language = Lib::Language->new();

    my @comments = @{ $self->{comments} };
    for my $i (0 .. $#comments) {
        my $comment = $comments[$i];
        my ($commentid, $status, $postdate, $author, $mailaddr, $hpurl, $text, $evaluate, $ipaddr) = split(/\,/, $comment);

        # ステータスが無効なコメントは表示しない
        next if ($status == 0);

        # 編集モード以外ではステータスが承認待ちのコメントは表示しない
        next if ($mode != 1 && $status == 2);

        # 匿名の置き換え
        if ($author eq '') {
            $author = $anonymous;
        }

        # 改行をデコード
        $text =~ s/<br.+?\/>enc_crlf/enc_crlf/g;
        $text =~ s/enc_crlf/<br \/>/g;

        # コンマをデコード
        $text =~ s/enc_conma/,/g;

        # >>$commentidに対してリンクを張る。
        $text =~ s/>>([0-9]+)/<a href="$topicurl#comment$1">&gt;&gt;$1<\/a>/g;
        $text =~ s/&gt;&gt;([0-9]+)/<a href="$topicurl#comment$1">&gt;&gt;$1<\/a>/g;

        # URLをリンク
        my $util = Lib::Util->new();
        $text = $util->url_link($text, "$comment_target rel=\"nofollow\"");

        # キーワードをリンク
        my $keyword = Lib::Keyword->new();
        $text = $keyword->link_keyword($self->{keyword}, $text, 0);

        # メールアドレスやホームページURLが入力されていた場合はリンク
        if ($hpurl ne '') {
            $author = "<a href=\"$hpurl\" $comment_target rel=\"nofollow\">$author</a>";
        }
        else {
            if ($mailaddr ne '') {
                $author = "<a href=\"mailto:$mailaddr\">$author</a>";
            }
        }

        my $menu = '';
        if ($mode == 1) {   # 編集モード
            my $delete = $language->get_statictext('delete');
            my $validate = $language->get_statictext('validate');
            $menu = "<a href=\"./?act=view_editor&func=delcomment&topicid=$topicid&id=$commentid\">$delete</a>";
            if ($status == 2) {
                $menu .= "&nbsp;&nbsp;<a href=\"./?act=exec_editor&func=admitcomment&topicid=$topicid&id=$commentid\">$validate</a>";
            }
        }

        $list .= <<"END_LIST";
<table border="0" width="100%" cellpadding="0" cellspacing="0">
<tr>
  <td><a name="comment$commentid"></a><b>$commentid</b> Posted by <b>$author</b> at $postdate</td>
  <td width="90" align="right" nowrap>$menu</td></tr>
<tr>
  <td colspan="2">$text</td>
</tr>
</table>
END_LIST
    }

    return $list;
}

# コメント入力用フォームの生成
sub _comment_form {
    my $self = shift;

    my $language = Lib::Language->new();

    my $showdetail = $language->get_statictext('show-detail');
    my $hidedetail = $language->get_statictext('hide-detail');
    my $recommend = $language->get_statictext('recommend');
    my $recommendguide = $language->get_statictext('recommend-guide');
    my $commentlabel = $language->get_statictext('comment');
    my $namelabel = $language->get_statictext('name');
    my $mailaddrlabel = $language->get_statictext('mail-address');
    my $homepagelabel = $language->get_statictext('homepage-url');
    my $postbutton = $language->get_statictext('post');

    my $topicid = $self->{topicid};
    my $formpart = <<"END_FORMPART";
  <input type="hidden" name="act" value="add_comment">
  <input type="hidden" name="topicid" value="$topicid">
END_FORMPART

    my $form = <<"END_FORM";
<a name="commentform"></a>
<div id="toggleHide" style="display:block">
<form name="hideform" method="post" action="./">
$formpart
<a href="javascript:;" onmousedown="toggleInput('toggleShow','toggleHide');">$showdetail</a>
<br /><br />
<input type="checkbox" name="evaluate" value="1"><b>$recommend</b> &nbsp;$recommendguide<br />
<input type="hidden" name="author" value="\$AUTHOR\$">
<input type="hidden" name="mailaddr" value="">
<input type="hidden" name="hpurl" value="\$HPURL\$">
<b>$commentlabel :</b><br />
<textarea name="text" rows="10" cols="60"></textarea>
<br />
<input type="submit" value="$postbutton">
</form>
</div>
<div id="toggleShow" style="display:none">
<form name="showform" method="post" action="./">
$formpart
<a href="javascript:;" onmousedown="toggleInput('toggleShow','toggleHide');">$hidedetail</a>
<br /><br />
$namelabel :<br />
<input type="text" name="author" size="60" value="\$AUTHOR\$"><br />
$mailaddrlabel :<br />
<input type="text" name="mailaddr" size="60"><br />
$homepagelabel :<br />
<input type="text" name="hpurl" size="60" value="\$HPURL\$"><br />
<br />
<input type="checkbox" name="evaluate" value="1"><b>$recommend</b> &nbsp;$recommendguide<br />
<b>$commentlabel :</b><br />
<textarea name="text" rows="10" cols="60"></textarea>
<br />
<input type="submit" value="$postbutton">
</form>
</div>
END_FORM

    return $form;
}

# ページナビゲーション
sub _page_navi {
    my $self = shift;
    my $topicid = shift;

    my $archive = Lib::Archive->new();
    my $lastid = $archive->get_counter();

    my $language = Lib::Language->new();
    my $prevpost = $language->get_statictext('prev-post');
    my $nextpost = $language->get_statictext('next-post');

    my $prevpage = '';
    for (my $id = $topicid - 1; $id >= 0; $id--) {
        my $filename = "$self->{data_dir}/$id" . "tp.data";
        if (-f $filename) {
            $prevpage = "<a href=\"./?func=detail&id=$id\">$prevpost</a>";
            last;
        }
    }
    my $nextpage = '';
    for (my $id = $topicid + 1; $id <= $lastid; $id++) {
        my $filename = "$self->{data_dir}/$id" . "tp.data";
        if (-f $filename) {
            $nextpage = "<a href=\"./?func=detail&id=$id\">$nextpost</a>";
            last;
        }
    }

    my $pagenavi = "$prevpage&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;$nextpage";
    return $pagenavi;
}

1;
# End of Topic.pm
