#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2010 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::DateTime;
use Lib::String;
use Lib::Util;
use Lib::Filter;
use Lib::Mail;
use Lib::Topic;
use Lib::Archive;
use Lib::Conf;
use Lib::Language;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $conf = Lib::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    my $topicid = $cgi->param('topicid');
    if ($topicid !~ /\A[0-9]+\z/) {
        $error->print_error('00002e');
    }
    my $topic = Lib::Topic->new();
    my $ret = $topic->load($topicid, 0);
    if (!$ret) {
        $error->print_error('00002e');
    }

    if ($act eq 'add_comment') {
        _add_comment($cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

sub _add_comment() {
    my ($cgi) = @_;

    my $util = Lib::Util->new();
    my $dt = Lib::DateTime->new();

    my $commentdata;
    $commentdata->{topicid}  = $cgi->param('topicid');
    $commentdata->{status}   = 1;
    $commentdata->{postdate} = $dt->local_datetime(0);
    $commentdata->{author}   = $cgi->param('author');
    $commentdata->{mailaddr} = $cgi->param('mailaddr');
    $commentdata->{hpurl}    = $cgi->param('hpurl');
    $commentdata->{text}     = $cgi->param('text');
    $commentdata->{evaluate} = $cgi->param('evaluate');
    $commentdata->{ipaddr}   = $ENV{'REMOTE_ADDR'};
    if ($commentdata->{evaluate} eq '') {
        $commentdata->{evaluate} = 0;
    }

    _check_input($commentdata);

    $commentdata->{text} = $util->tag_invalidate($commentdata->{text});
    $commentdata->{text} =~ s/\r?\n/enc_crlf/g;
    $commentdata->{text} =~ s/,/enc_conma/g;
    $commentdata->{author} = $util->tag_invalidate($commentdata->{author});
    $commentdata->{author} =~ s/\r?\n//g;
    $commentdata->{author} =~ s/,/enc_conma/g;
    $commentdata->{mailaddr} = $util->tag_invalidate($commentdata->{mailaddr});
    $commentdata->{mailaddr} =~ s/\r?\n//g;
    $commentdata->{mailaddr} =~ s/,/enc_conma/g;
    $commentdata->{hpurl} = $util->tag_invalidate($commentdata->{hpurl});
    $commentdata->{hpurl} =~ s/\r?\n//g;
    $commentdata->{hpurl} =~ s/,/enc_conma/g;

    my $language = Lib::Language->new();

    # スパムチェック
    if (_is_spam($commentdata->{ipaddr}, $commentdata->{text}, $commentdata->{postdate})) {
        _show_error($commentdata, $language->get_usererror('deny-post'));
    }

    my $topic = Lib::Topic->new();
    $topic->load($commentdata->{topicid}, 1);
    my $topic_status = $topic->get_status();
    if ($topic_status != 1 && $topic_status != 3) {
        _show_error($commentdata, $language->get_usererror('deny-post'));
    }

    if ($topic_status == 3) {
        $commentdata->{status} = 2;  # 承認待ちステータスを設定
    }

    my $editor = $topic->get_editor();

    # コメントIDの取得
    my @comments = $topic->get_comments();
    my $commentid = 0;
    if (scalar(@comments) > 0) {
        $commentid = (split(/\,/, $comments[$#comments]))[0];
    }
    $commentid++;

    # メール通知先をリストアップ
    my @mailinglist = ();
    foreach my $item (@comments) {
        my ($id, $mailaddr) = (split(/\,/, $item))[0,4];
        next if (!$id);
        next if (!$mailaddr);
        $mailaddr =~ s/enc_conma/,/g;
        push @mailinglist, $mailaddr;
    }

    my $commentrec = "$commentid,$commentdata->{status},$commentdata->{postdate},$commentdata->{author},";
    $commentrec .= "$commentdata->{mailaddr},$commentdata->{hpurl},$commentdata->{text},";
    $commentrec .= "$commentdata->{evaluate},$commentdata->{ipaddr}";
    push @comments, $commentrec;
    $topic->set_comments(@comments);

    $topic->set_commentnum(scalar(@comments));
    my $point = $topic->get_point() ;
    if ($commentdata->{evaluate} > 0) {
        $point += $conf->get_good_point();
    }
    else {
        $point += $conf->get_normal_point();
    }
    $topic->set_point($point);
    $topic->update();

    # おすすめの場合は更新情報に追加
    if ($commentdata->{evaluate} == 1) {
        my $archive = Lib::Archive->new();
        $archive->add($commentdata->{topicid});
    }

    # ログに保存
    _write_log($commentdata->{topicid}, $commentdata->{postdate},
            $commentdata->{ipaddr}, $commentdata->{author}, $commentdata->{text});

    my $topicurl = $conf->get_docroot_dir() . "/?func=detail&id=$commentdata->{topicid}";
    my $topictitle = $topic->get_title();

    $commentdata->{author} =~ s/enc_conma/,/g;
    $commentdata->{mailaddr} =~ s/enc_conma/,/g;
    $commentdata->{hpurl} =~ s/enc_conma/,/g;
    $commentdata->{text} =~ s/enc_conma/,/g;
    $commentdata->{text} =~ s/enc_crlf/\n/g;

    my $postperson = $language->get_statictext('post-person');
    my $maillabel = $language->get_statictext('mail-address');
    my $homepagelabel = $language->get_statictext('homepage-url');
    my $commentlabel = $language->get_statictext('comment');
    my $commentinfo = $language->get_message('comment-info');
    my $commentinfobody1 = $language->get_message('comment-infobody1');
    my $commentinfobody2 = $language->get_message('comment-infobody2');
    my $commentinfobody3 = $language->get_message('comment-infobody3');

    $commentinfo =~ s/\$TOPICTITLE\$/$topictitle/g;
    $commentinfobody1 =~ s/\$TOPICTITLE\$/$topictitle/g;

    my $subject = $commentinfo;
    my $body = <<"END_BODY";
$commentinfobody1
$topicurl

----------

$postperson :
$commentdata->{author}

$maillabel :
$commentdata->{mailaddr}

$homepagelabel :
$commentdata->{hpurl}

$commentlabel :
$commentdata->{text}

---
$commentinfobody2
$commentinfobody3

----------
END_BODY

    # send comment information mail to user and webmaster.
    if ($editor ne '') {
        my $user = Lib::User->new();
        $user->load($editor);
        my $mailaddr = $user->get_mailaddr();
        push @mailinglist, $mailaddr;
    }

    my $adminaddr = $conf->get_admin_address();
    if ($conf->get_notify_mail() && $adminaddr) {
        push @mailinglist, $adminaddr;
    }

    my $mail = Lib::Mail->new();
    my %count;
    @mailinglist = grep {!$count{$_}++} @mailinglist;
    foreach my $mailaddr (@mailinglist) {
        $mail->send($subject, $body, $mailaddr);
    }

    print "Location: $topicurl", "\n\n";
}

# ログに記録
sub _write_log() {
    my ($topicid, $postdate, $ipaddr, $author, $text) = @_;
    my $yyyy = substr($postdate, 0, 4);
    my $mm = substr($postdate, 5, 2);
    my $dd = substr($postdate, 8, 2);
    my $systemdir = $conf->get_system_dir();
    my $logfile = "$systemdir/log/c_$yyyy$mm$dd.log";
    my $datarec = "$topicid,$postdate,$ipaddr,$author,$text";

    open my $lfh, '>>', $logfile;
    print {$lfh} "$datarec\n";
    close $lfh;
}

# スパム判定
sub _is_spam() {
    my ($ip, $cmtext, $cmdate) = @_;

    my $iplimit = 10;
    my $cmlimit = 3;
    my $yyyy = substr($cmdate, 0, 4);
    my $mm = substr($cmdate, 5, 2);
    my $dd = substr($cmdate, 8, 2);
    my $systemdir = $conf->get_system_dir();
    my $logfile = "$systemdir/log/c_$yyyy$mm$dd.log";
    my $ipcount = 0;
    my $cmcount = 0;
    if (-f $logfile) {
        open my $lfh, '<', $logfile;
        while (my $data = <$lfh>) {
            last if ($cmcount >= $cmlimit);
            last if ($ipcount >= $iplimit);
            chomp $data;
            my ($topicid, $postdate, $ipaddr, $author, $text) = split(/\,/, $data);
            if ($ip eq $ipaddr) {
                $ipcount++;
            }
            if (substr($cmtext, 0, 10) eq substr($text, 0, 10)) {
                $cmcount++;
            }
        }
        close $lfh;
    }

    if ($cmcount >= $cmlimit) {
        return 1; # is spam
    }
    if ($ipcount >= $iplimit) {
        my $denyrec = "deny from $ip";
        my $accessfile = ".htaccess";
        open my $afh, '>>', $accessfile;
        print {$afh} "$denyrec\n";
        close $afh;
        return 1; # is spam
    }

    return 0; # no sapm
}

sub _check_input() {
    my ($commentdata) = @_;

    my $filter = Lib::Filter->new();
    my $docrootdir = $conf->get_docroot_dir();

    my $language = Lib::Language->new();

    my $referer = $ENV{'HTTP_REFERER'};
    if ($referer !~ /\A$docrootdir/) {
        _show_error($commentdata, $language->get_usererror('deny-post'));
    }

    if ($filter->check_filter($commentdata->{ipaddr})) {
        _show_error($commentdata, $language->get_usererror('deny-post'));
    }

    if ($commentdata->{author} ne '') {
        $commentdata->{author} =~ s/[\r\n]//g;
        if ($filter->check_filter($commentdata->{author})) {
            _show_error($commentdata, $language->get_usererror('deny-post'));
        }
    }

    if ($commentdata->{mailaddr} ne '') {
        $commentdata->{mailaddr} =~ s/[\r\n]//g;
        if ($commentdata->{mailaddr} !~ /.+\@.+[.].+/) {
            _show_error($commentdata, $language->get_usererror('mailaddr-format'));
        }
        if ($filter->check_filter($commentdata->{mailaddr})) {
            _show_error($commentdata, $language->get_usererror('deny-post'));
        }
    }

    if ($commentdata->{hpurl} ne '') {
        $commentdata->{hpurl} =~ s/[\r\n]//g;
        my $pattern = "s?https?://[-_.!~*'()a-zA-Z0-9;/?:@&=+$,%#]+";
        if ($commentdata->{hpurl} !~ /$pattern/) {
            _show_error($commentdata, $language->get_usererror('url-format'));
        }
        if ($filter->check_filter($commentdata->{hpurl})) {
            _show_error($commentdata, $language->get_usererror('deny-post'));
        }
    }

    if ($commentdata->{text} eq '') {
        _show_error($commentdata, $language->get_usererror('input-comment'));
    }
    else {
        my $langcode = $language->get_langcode();
        if (($langcode eq 'ja') && ($commentdata->{text} =~ /\A[\x20-\x7E\r\n]+\z/)) {
            _show_error($commentdata, $language->get_usererror('deny-post'));
        }
        if ($commentdata->{text} =~ /<[aA] .+>.*<\/[aA]>/) {
            _show_error($commentdata, $language->get_usererror('deny-post'));
        }
        if ($filter->check_filter($commentdata->{text})) {
            _show_error($commentdata, $language->get_usererror('deny-post'));
        }
    }
}

sub _show_error() {
    my ($commentdata, $msg) = @_;

    my $pathtitle = _get_pathtitle($commentdata->{topicid});

    my $language = Lib::Language->new();
    my $homelabel = $language->get_statictext('home');

    my $caption = $msg;
    my $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
<a class="path" href="./?func=detail&id=$commentdata->{topicid}">$pathtitle</a>&nbsp;
&gt;&nbsp;
$caption
END_PATH

    $msg = "<p><div id=\"message\">$msg</div></p>";

    my $systemdir = $conf->get_system_dir();
    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();

    open my $templatefh, '<', "$systemdir/tmpl/message.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$MSG\$/$msg/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

sub _get_pathtitle() {
    my ($topicid) = @_;
    my $string = Lib::String->new();
    my $topic = Lib::Topic->new();
    $topic->load($topicid, 0);
    my $topictitle = $topic->get_title();
    my $adjusttitle = $string->cut($topictitle, 50);
    return $adjusttitle;
}

1;
