//  vorbis_encoder.hpp: vorbis_enc device adapter

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef HAMIGAKI_AUDIO_VORBIS_ENCODER_HPP
#define HAMIGAKI_AUDIO_VORBIS_ENCODER_HPP

#include <hamigaki/audio/detail/endian.hpp>
#include <hamigaki/audio/pcm/format.hpp>
#include <hamigaki/iostreams/device/file.hpp>
#include <boost/iostreams/close.hpp>
#include <boost/iostreams/write.hpp>
#include <boost/noncopyable.hpp>

#include <vorbis/vorbisenc.h>
#include <cstdlib>
#include <ctime>

#include <boost/config/abi_prefix.hpp>

namespace hamigaki { namespace audio {

namespace detail
{

template<typename Sink>
class vorbis_encoder_impl : boost::noncopyable
{
public:
    typedef char char_type;

    vorbis_encoder_impl(const Sink& sink, const pcm_format& fmt)
        : sink_(sink), format_(fmt), close_(false)
    {
        ::vorbis_info_init(&vi_);

        ::vorbis_encode_init_vbr(&vi_, fmt.channels, fmt.rate, 0.1);

        ::vorbis_comment_init(&vc_);

        ::vorbis_analysis_init(&vd_, &vi_);
        ::vorbis_block_init(&vd_, &vb_);

        std::srand(std::time(0));
        ::ogg_stream_init(&os_, std::rand());

        ::ogg_packet header;
        ::ogg_packet header_comm;
        ::ogg_packet header_code;

        ::vorbis_analysis_headerout(
            &vd_, &vc_, &header, &header_comm, &header_code);
        ::ogg_stream_packetin(&os_, &header);
        ::ogg_stream_packetin(&os_, &header_comm);
        ::ogg_stream_packetin(&os_, &header_code);

        while (::ogg_stream_flush(&os_, &og_) != 0)
        {
            boost::iostreams::write(sink_,
                reinterpret_cast<const char*>(og_.header), og_.header_len);
            boost::iostreams::write(sink_,
                reinterpret_cast<const char*>(og_.body), og_.body_len);
        }
    }

    ~vorbis_encoder_impl()
    {
        try
        {
            close();
        }
        catch (...)
        {
        }

        ::ogg_stream_clear(&os_);
        ::vorbis_block_clear(&vb_);
        ::vorbis_dsp_clear(&vd_);
        ::vorbis_comment_clear(&vc_);
        ::vorbis_info_clear(&vi_);
    }

    std::size_t block_size() const
    {
        return format_.channels * ((format_.bits + 7) / 8);
    }

    const pcm_format& format() const
    {
        return format_;
    }

    std::streamsize write(const char_type* s, std::streamsize n)
    {
        std::size_t block = block_size();

        if (n % block != 0)
            throw BOOST_IOSTREAMS_FAILURE("invalid write size");

        const std::size_t count = n / block;
        float** buffer = ::vorbis_analysis_buffer(&vd_, count);

        if (format_.bits == 8)
        {
            for (std::size_t i = 0; i < count; ++i)
            {
                for (std::size_t j = 0; j < format_.channels; ++j)
                {
                    buffer[j][i] =
                        (static_cast<unsigned char>(s[0]) - 128) / 128.f;
                    ++s;
                }
            }
        }
        else
        {
            for (std::size_t i = 0; i < count; ++i)
            {
                for (std::size_t j = 0; j < format_.channels; ++j)
                {
                    boost::uint16_t tmp = decode_little_endian16(s);

                    if (tmp & 0x8000)
                        buffer[j][i] = ((~tmp & 0xFFFF) + 1) / (-32768.f);
                    else
                        buffer[j][i] = tmp / 32768.f;

                    s += 2;
                }
            }
        }

        write_to_downstream(count);

        return n;
    }

    void close()
    {
        if (!close_)
        {
            write_to_downstream(0);
            close_ = true;
        }
    }

private:
    Sink sink_;
    pcm_format format_;
    bool close_;
    ::ogg_stream_state  os_;
    ::ogg_page          og_;
    ::ogg_packet        op_;
    ::vorbis_info       vi_;
    ::vorbis_comment    vc_;
    ::vorbis_dsp_state  vd_;
    ::vorbis_block      vb_;

    void write_to_downstream(int count)
    {
        ::vorbis_analysis_wrote(&vd_, count);

        while (::vorbis_analysis_blockout(&vd_, &vb_) == 1)
        {
            ::vorbis_analysis(&vb_, 0);
            ::vorbis_bitrate_addblock(&vb_);

            while (::vorbis_bitrate_flushpacket(&vd_, &op_))
            {
                ::ogg_stream_packetin(&os_, &op_);

                bool eos = false;
                while (!eos)
                {
                    int result = ::ogg_stream_pageout(&os_, &og_);
                    if (result == 0)
                        break;

                    boost::iostreams::write(
                        sink_,
                        reinterpret_cast<const char*>(og_.header),
                        og_.header_len);

                    boost::iostreams::write(
                        sink_,
                        reinterpret_cast<const char*>(og_.body),
                        og_.body_len);

                    if (::ogg_page_eos(&og_))
                        eos = true;
                }
            }
        }
    }
};

} // namespace detail

template<typename Sink>
struct basic_vorbis_encoder
{
    typedef detail::vorbis_encoder_impl<Sink> impl_type;

public:
    typedef char char_type;

    struct category :
        boost::iostreams::optimally_buffered_tag,
        boost::iostreams::output,
        boost::iostreams::device_tag,
        boost::iostreams::closable_tag {};

    basic_vorbis_encoder(const Sink& sink, const pcm_format& fmt)
        : pimpl_(new impl_type(sink, fmt))
    {
    }

    std::streamsize optimal_buffer_size() const
    {
        return pimpl_->block_size() * (pimpl_->format().rate / 5);
    }

    std::size_t block_size() const
    {
        return pimpl_->block_size();
    }

    const pcm_format& format() const
    {
        return pimpl_->format();
    }

    std::streamsize write(const char_type* s, std::streamsize n)
    {
        return pimpl_->write(s, n);
    }

    void close()
    {
        pimpl_->close();
    }

private:
    boost::shared_ptr<impl_type> pimpl_;
};

class vorbis_encoder
    : public basic_vorbis_encoder<hamigaki::iostreams::file_sink>
{
private:
    typedef basic_vorbis_encoder<hamigaki::iostreams::file_sink> base_type;

public:
    explicit vorbis_encoder(const std::string& path, const pcm_format& fmt)
        : base_type(hamigaki::iostreams::file_sink(
            path, BOOST_IOS::out|BOOST_IOS::binary), fmt)
    {
    }
};

template<typename Sink>
inline basic_vorbis_encoder<Sink>
make_vorbis_encoder(const Sink& sink, const pcm_format& fmt)
{
    return basic_vorbis_encoder<Sink>(sink, fmt);
}

} } // End namespaces audio, hamigaki.

#include <boost/config/abi_suffix.hpp>

#endif // HAMIGAKI_AUDIO_VORBIS_ENCODER_HPP
