/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.factory;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.jmx.adaptor.rmi.RMIAdaptor;

import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.accesscontrol.ejb.entity.RoleLocal;
import com.clustercontrol.accesscontrol.ejb.entity.RoleUtil;
import com.clustercontrol.accesscontrol.ejb.entity.UserLocal;
import com.clustercontrol.accesscontrol.ejb.entity.UserUtil;
import com.clustercontrol.bean.Property;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ユーザを操作するクラスです。
 *
 * @version 2.0.0
 * @since 2.0.0
 */
public class ModifyUser {
	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog(ModifyUser.class);
	/** JMXのルックアップ名 */
	public static final String LOOKUP_NAME = "jmx/invoker/RMIAdaptor";
	/** JMXのオブジェクト名 */
	public static final String OBJECT_NAME = "jboss.security:service=JaasSecurityManager";
	/** JMXのオブジェクトのメソッド名 */
	public static final String OPERATION_NAME = "flushAuthenticationCache";
	
	/**
	 * ユーザを追加します。
	 *  
	 * @param property ユーザ用プロパティ
	 * @param userName 作成ユーザ
	 * @throws NamingException
	 * @throws CreateException
	 * @throws FinderException
	 * @throws ParseException
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.AccessLock#lock(String)
	 */
	public void addUser(Property property, String userName) throws NamingException, CreateException, FinderException, ParseException  {
    	//アクセスロック
    	AccessLock.lock(AccessLock.ACCESS);
		
		String uid = null;
	    try {
			ArrayList values = null;
			//現在日時を取得
			Date now = new Date();
			
			//UID取得
			values = PropertyUtil.getPropertyValue(property, UserProperty.UID);
			uid = (String)values.get(0);
			//UIDの有効性をチェック
			checkValidUid(uid);
					
			//Userを作成
			UserLocal user = UserUtil.getLocalHome().create(uid);

			//名前を取得し、Userに設定
			values = PropertyUtil.getPropertyValue(property, UserProperty.NAME);
			user.setCn(checkString(values.get(0)));
			//説明を取得し、Userに設定
			values = PropertyUtil.getPropertyValue(property, UserProperty.DESCRIPTION);
			user.setDescription(checkString(values.get(0)));
			//作成者を取得し、Userに設定
			user.setCreatorsName(checkString(userName));
			//作成日時を取得し、Userに設定
			user.setCreateTimestamp(now);
			//更新者を取得し、Userに設定
			user.setModifiersName(checkString(userName));
			//更新日時を取得し、Userに設定
			user.setModifyTimestamp(now);

			//HinemosUser
			assignUser(RoleConstant.HINEMOS_USER, user.getDn());
			
			//アクセス権を取得し、Userに設定
			List<String> roleList = RoleConstant.getRoles();
			for(int i = 0; i < roleList.size(); i++){
				String roleName = roleList.get(i);
				
				values = PropertyUtil.getPropertyValue(property, roleName);
				if(values.size() > 0 && ((Boolean)values.get(0)).booleanValue()){
					assignUser(roleName, user.getDn());
				}
			}
			
		} catch (EJBException e) {
            AplLogger apllog = new AplLogger("ACC", "acc");
            String[] args = {uid};
            apllog.put("SYS", "006", args);
            
            m_log.debug("addUser() : " + e.getMessage());
			throw e;
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("ACC", "acc");
            String[] args = {uid};
            apllog.put("SYS", "006", args);
            
            m_log.debug("addUser() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("ACC", "acc");
            String[] args = {uid};
            apllog.put("SYS", "006", args);
            
            m_log.debug("addUser() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * ユーザを変更します。
	 * 
	 * @param property ユーザ用プロパティ
	 * @param userName 変更ユーザ
	 * @throws NamingException
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.AccessLock#lock(String)
	 */
	public void modifyUser(Property property, String userName) throws NamingException, FinderException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.ACCESS);
    	
    	String uid = null;
	    try {
			ArrayList values = null;
			Date now = new Date();
			
			//UID取得
			values = PropertyUtil.getPropertyValue(property, UserProperty.UID);
			uid = (String)values.get(0);
			
			//Userを検索し取得
			UserLocal user = UserUtil.getLocalHome().findByUid(uid);
			//名前を取得し、Userに設定
			values = PropertyUtil.getPropertyValue(property, UserProperty.NAME);
			user.setCn(checkString(values.get(0)));
			//説明を取得し、Userに設定
			values = PropertyUtil.getPropertyValue(property, UserProperty.DESCRIPTION);
			user.setDescription(checkString(values.get(0)));
			//更新者を取得し、Userに設定
			user.setModifiersName(checkString(userName));
			//更新日時を取得し、Userに設定
			user.setModifyTimestamp(now);
			
			//アクセス権を取得し、Userに設定
			List<String> roleList = RoleConstant.getRoles();
			for(int i = 0; i < roleList.size(); i++){
				String roleName = roleList.get(i);
				
				values = PropertyUtil.getPropertyValue(property, roleName);
				if(values.size() > 0){
					releaseUser(roleName, user.getDn());
					if(((Boolean)values.get(0)).booleanValue()){
						assignUser(roleName, user.getDn());
					}
				}
			}
			
			//認証キャッシュ更新
			flushAuthenticationCache();
			
		} catch (EJBException e) {
            AplLogger apllog = new AplLogger("ACC", "acc");
            String[] args = {uid};
            apllog.put("SYS", "008", args);
            
            m_log.debug("modifyUser() : " + e.getMessage());
			throw e;
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("ACC", "acc");
            String[] args = {uid};
            apllog.put("SYS", "008", args);
            
            m_log.debug("modifyUser() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * ユーザを削除します。
	 * 
	 * @param uid ユーザID
	 * @throws NamingException
	 * @throws RemoveException
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.AccessLock#lock(String)
	 */
	public void deleteUser(String uid) throws NamingException, RemoveException, FinderException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.ACCESS);
    	
		try {
			//Userを検索し取得
			UserLocal user = UserUtil.getLocalHome().findByUid(uid);
			String dn = user.getDn();
			user.remove();
			
			//アクセス権を取得し、Userに設定
			List<String> roleList = RoleConstant.getRoles();
			for(int i = 0; i < roleList.size(); i++){
				String roleName = roleList.get(i);
				releaseUser(roleName, dn);
			}
			
			//認証キャッシュ更新
			flushAuthenticationCache();

		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("ACC", "acc");
            String[] args = {uid};
            apllog.put("SYS", "007", args);
            
            m_log.debug("deleteUser() : " + e.getMessage());
			throw e;
		} catch (RemoveException e) {
            AplLogger apllog = new AplLogger("ACC", "acc");
            String[] args = {uid};
            apllog.put("SYS", "007", args);
            
            m_log.debug("deleteUser() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("ACC", "acc");
            String[] args = {uid};
            apllog.put("SYS", "007", args);
            
            m_log.debug("deleteUser() : " + e.getMessage());
			throw e;
		}
	}
	
    /**
     * ロールにユーザのDNを追加します。
     * 
     * @param roleName ロール名
     * @param dn ユーザのDN
     * @throws FinderException
     * @throws NamingException
     */
    @SuppressWarnings("unchecked")
	protected void assignUser(String roleName, String dn) throws FinderException, NamingException {
    	RoleLocal role = null;
        try {
            //Roleエントリを取得
        	role = RoleUtil.getLocalHome().findByCn(roleName);
        	
        	//UserのDNを追加
        	ArrayList list = role.getMember();
        	if(list == null){
        		list = new ArrayList();
        	}
        	list.add(dn);
        	role.setMember(list);
        	
        } catch (FinderException e) {
        	throw new FinderException("ModifyUser.assignUser() -> error : " + e.getMessage());
        } catch (NamingException e) {
            throw e;
        }
    }
    
    /**
     * ロールからユーザのDNを削除します。
     * 
     * @param roleName ロール名
     * @param dn ユーザのDN
     * @throws FinderException
     * @throws NamingException
     */
    protected void releaseUser(String roleName, String dn) throws FinderException, NamingException {
    	RoleLocal role = null;
        try {
            //Roleエントリを取得
        	role = RoleUtil.getLocalHome().findByCn(roleName);
        	
        	//UserのDNを削除
        	String userDn = dn;
        	ArrayList list = role.getMember();
        	if(list != null){
	        	for(int i = 0; i < list.size(); i++){
	        		if(userDn.equalsIgnoreCase((String)list.get(i))){
	        			list.remove(i);
	        			break;
	        		}
	        	}
	        	role.setMember(list);
        	}
        	else{
        		role.setMember(null);
        	}
        } catch (FinderException e) {
        	throw new FinderException("ModifyUser.releaseUser() -> " + e.getMessage());
        } catch (NamingException e) {
            throw e;
        }
    }
    
    /**
     * ユーザのパスワードを変更します。
     * 
     * @param uid ユーザID
     * @param password パスワード
     * @throws FinderException
     * @throws NamingException
     * 
     * @see com.clustercontrol.accesscontrol.factory.AccessLock#lock(String)
     */
    public void modifyPassword(String uid, String password) throws FinderException, NamingException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.ACCESS);
    	
    	try {
			//Userを検索し取得
			UserLocal user = UserUtil.getLocalHome().findByUid(uid);
			
			//パスワードを設定
			user.setPassword(password);
			
			//認証キャッシュ更新
			flushAuthenticationCache();

		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("ACC", "acc");
            String[] args = {uid};
            apllog.put("SYS", "009", args);
            
            m_log.debug("modifyPassword() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("ACC", "acc");
            String[] args = {uid};
            apllog.put("SYS", "008", args);
            
            m_log.debug("modifyPassword() : " + e.getMessage());
			throw e;
		}
    }
	
	/**
	 * 引数で指定されたチェック対象が、Stringのインスタンスかチェックし、<BR>
	 * Stringならばチャック対象を返します。違っていればnullを返します。
	 * 
     * @param value チェック対象
     * @return チェック結果
     */
    private String checkString(Object value){
		if(value instanceof String){
			if(((String)value).compareTo("") == 0){
				return null;
			}
			else{
				return (String)value;
			}
		}
		else{
			return null;
		}
	}
	
	/**
	 * ユーザIDとして有効な文字列かチェックします。<BR>
	 * 無効な場合、java.text.ParseExceptionをスローする 
	 * 
	 * @param uid ユーザID
	 * @throws ParseException
	 */
	private void checkValidUid(String uid) throws ParseException{
		
		char charData = '\u0000';
		
		// 有効：'A'-'Z','a'-'z','0'-'9'
		for (int i=0;i < uid.length();i++) {
			charData = uid.charAt(i);   
			if (((charData < 'A' )|| (charData > 'Z' )) &&
					((charData < 'a' ) || ( charData > 'z')) &&
					((charData < '0' ) || ( charData > '9'))) {
				throw new ParseException(uid, i);
			}
		}
	}
	
	/**
	 * JMX経由で、認証キャッシュをフラッシュします。
	 */
	public void flushAuthenticationCache() {

		try{
	        InitialContext ic = new InitialContext();
	        
	        //RMIAdaptorを取得
	        RMIAdaptor server = (RMIAdaptor) ic.lookup(LOOKUP_NAME);

	        //ObjectNameを設定
	        ObjectName name = new ObjectName(OBJECT_NAME);
	        
	        //ObjectNameのOperationNameのメソッドを実行
	        Object[] args = {"hinemos"};
	        String[] signature = {String.class.getName()};
	        server.invoke(name, OPERATION_NAME, args, signature);

		} catch (NamingException e) {
			m_log.debug("flushAuthenticationCache() : " + e.getMessage());
		} catch (MalformedObjectNameException e) {
			m_log.debug("flushAuthenticationCache() : " + e.getMessage());
		} catch (NullPointerException e) {
			m_log.debug("flushAuthenticationCache() : " + e.getMessage());
		} catch (InstanceNotFoundException e) {
			m_log.debug("flushAuthenticationCache() : " + e.getMessage());
		} catch (MBeanException e) {
			m_log.debug("flushAuthenticationCache() : " + e.getMessage());
		} catch (ReflectionException e) {
			m_log.debug("flushAuthenticationCache() : " + e.getMessage());
		} catch (IOException e) {
			m_log.debug("flushAuthenticationCache() : " + e.getMessage());
		}
	}
}
