/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.accesscontrol.ejb.entity.RoleLocal;
import com.clustercontrol.accesscontrol.ejb.entity.RoleUtil;
import com.clustercontrol.accesscontrol.ejb.entity.UserLocal;
import com.clustercontrol.accesscontrol.ejb.entity.UserUtil;
import com.clustercontrol.bean.Property;
import com.clustercontrol.util.PropertyUtil;

/**
 * ユーザを検索するクラスです。
 *
 * @version 2.0.0
 * @since 2.0.0
 */
public class SelectUser {

    /**
     * ユーザ一覧情報を取得します。
	 * <p>
	 * <ol>
	 * <li>全てのUserを取得します。</li>
	 * <li>取得したUser数、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>1Userの情報をテーブルのカラム順（{@link com.clustercontrol.accesscontrol.bean.UserListTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 *   <dl>
	 *   <dt>ユーザ一覧情報（Objectの2次元配列）</dt>
	 *   <dd>{ ユーザ情報1 {カラム1の値, カラム2の値, … }, ユーザ情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *   </dl>
	 *  </ol>
	 *  <li>ユーザ一覧情報を返します。</li>
	 * </ol>
	 * 
     * @return ユーザ一覧情報
     * @throws FinderException
     * @throws NamingException
     * 
     * @see com.clustercontrol.accesscontrol.ejb.entity.UserLocalHome#findAll()
     */
    @SuppressWarnings("unchecked")
	public ArrayList getUserList() throws FinderException, NamingException {

        ArrayList list = null;
		try {
			//Userの全エントリを取得
			Collection ct = UserUtil.getLocalHome().findAll();
			
			list = new ArrayList();
			Iterator itr = ct.iterator();
			while(itr.hasNext()){
			    UserLocal user = (UserLocal)itr.next();
			    ArrayList info = new ArrayList();
			    info = new ArrayList();
			    info.add(user.getUid());
			    info.add(user.getCn());
			    info.add(user.getDescription());
			    info.add(user.getCreatorsName());
			    info.add(user.getCreateTimestamp());
			    info.add(user.getModifiersName());
			    info.add(user.getModifyTimestamp());
			    list.add(info);
			}
		} catch (FinderException e) {
			throw new FinderException("SelectUser.getUserList() find error : " + e.getMessage());
		} catch (NamingException e) {
			throw e;
		}
        
        return list;
    }
    
    /**
     * 検索条件に一致するユーザ一覧情報を取得します。
	 * <p>
	 * <ol>
	 * <li>指定された検索条件に一致するUserを取得します。</li>
	 * <li>取得したUser数、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>1Userの情報をテーブルのカラム順（{@link com.clustercontrol.accesscontrol.bean.UserListTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 *   <dl>
	 *   <dt>ユーザ一覧情報（Objectの2次元配列）</dt>
	 *   <dd>{ ユーザ情報1 {カラム1の値, カラム2の値, … }, ユーザ情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *   </dl>
	 *  </ol>
	 *  <li>ユーザ一覧情報を返します。</li>
	 * </ol>
	 * 
	 * @param property 検索条件
     * @return ユーザ一覧情報
     * @throws FinderException
     * @throws NamingException
     * 
     * @see com.clustercontrol.accesscontrol.ejb.entity.UserLocalHome#findUser(java.lang.String, java.lang.String, java.lang.String, java.util.List)
     * @see com.clustercontrol.util.PropertyUtil#getProperty(com.clustercontrol.bean.Property, java.lang.String)
     */
    @SuppressWarnings("unchecked")
    public ArrayList getUserList(Property property) throws FinderException, NamingException {
        ArrayList list = null;
        String uid = null;
        String name = null;
        String description = null;

        try {
            ArrayList values = null;
            //UID取得
            values = PropertyUtil.getPropertyValue(property, UserFilterProperty.UID);
            uid = (String)values.get(0);
            //名前取得
            values = PropertyUtil.getPropertyValue(property, UserFilterProperty.NAME);
            name = (String)values.get(0);
            //説明取得
            values = PropertyUtil.getPropertyValue(property, UserFilterProperty.DESCRIPTION);
            description = (String)values.get(0);
            
            //アクセス権取得
            List<String> accessList = new ArrayList<String>();
			List<String> roleList = RoleConstant.getRoles();
			for(int i = 0; i < roleList.size(); i++){
				String roleName = roleList.get(i);
				
				values = PropertyUtil.getPropertyValue(property, roleName);
				if(values.size() > 0 && ((Boolean)values.get(0)).booleanValue()){
					accessList.add(roleName);
				}
			}
			
            //Userの最上位から検索する
            Collection ct = 
                UserUtil.getLocalHome().findUser(
                		uid, 
                		name, 
                        description, 
                        accessList);
            
            list = new ArrayList();
            Iterator itr = ct.iterator();
            while(itr.hasNext()){
            	UserLocal user = (UserLocal)itr.next();
                ArrayList info = new ArrayList();
                info.add(user.getUid());
                info.add(user.getCn());
                info.add(user.getDescription());
                info.add(user.getCreatorsName());
                info.add(user.getCreateTimestamp());
                info.add(user.getModifiersName());
                info.add(user.getModifyTimestamp());
                list.add(info);
            }
        } catch (FinderException e) {
        	throw new FinderException("SelectUser.getUserList() find error : " + e.getMessage());
        } catch (NamingException e) {
            throw e;
        }
        
        return list;
    }
    
    /**
     * ユーザ情報を取得します。
     * 
     * @param uid ユーザID
     * @return ユーザ情報
     * @throws FinderException
     * @throws NamingException
     * 
     * @see com.clustercontrol.accesscontrol.ejb.entity.UserLocalHome#findByUid(java.lang.String)
     */
    public UserLocal getUser(String uid) throws FinderException, NamingException {
    	UserLocal user = null;
        try {
            //Userエントリを取得
        	user = UserUtil.getLocalHome().findByUid(uid);
        } catch (FinderException e) {
        	throw new FinderException("SelectUser.getUser() find error : " + e.getMessage());
        } catch (NamingException e) {
            throw e;
        }
        
        return user;
    }
    
    /**
     * Role(権限)を取得します。
     * 
     * @param roleName Role名
     * @return Role情報
     * @throws FinderException
     * @throws NamingException
     * 
     * @see com.clustercontrol.accesscontrol.ejb.entity.RoleLocalHome#findByCn(java.lang.String)
     */
    public RoleLocal getRole(String roleName) throws FinderException, NamingException {
    	RoleLocal role = null;
        try {
            //Userエントリを取得
        	role = RoleUtil.getLocalHome().findByCn(roleName);
        } catch (FinderException e) {
        	throw new FinderException("SelectUser.getRole() find error : " + e.getMessage());
        } catch (NamingException e) {
            throw e;
        }
        
        return role;
    }
    
    /**
     * ユーザ名を取得します。
     * 
     * @param uid ユーザID
     * @return ユーザ名
     * @throws FinderException
     * @throws NamingException
     * 
     * @see com.clustercontrol.accesscontrol.ejb.entity.UserLocalHome#findByUid(java.lang.String)
     */
    public String getUserName(String uid) throws FinderException, NamingException {
    	String userName = null;
        try {
            //Userエントリを取得
        	UserLocal user = UserUtil.getLocalHome().findByUid(uid);
        	// 名前
			if(user.getCn() != null && user.getCn().compareTo("") != 0){
				userName = user.getCn();
			}
        } catch (FinderException e) {
        	throw new FinderException("SelectUser.getUserName() find error : " + e.getMessage());
        } catch (NamingException e) {
            throw e;
        }
        
        return userName;
    }
}
