/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.monitor.run.composite;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.util.Messages;

/**
 * 監視基本情報コンポジットクラスです。
 * <p>
 * <dl>
 *  <dt>コンポジット</dt>
 *  <dd>「監視項目ID」 テキストボックス</dd>
 *  <dd>「説明」 テキストボックス</dd>
 * </dl>
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class MonitorBasicComposite extends Composite {
	
	/** カラム数（タイトル）。 */
	public static final int WIDTH_TITLE = 5;
	
	/** カラム数（値）。*/
	public static final int WIDTH_VALUE = 2;
	
	/** 監視項目ID テキストボックス。 */
	private Text m_textMonitorId = null;
	
	/** 説明 テキストボックス。 */
	private Text m_textDescription = null;
	
	/**
	 * インスタンスを返します。
	 * <p>
	 * 初期処理を呼び出し、コンポジットを配置します。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public MonitorBasicComposite(Composite parent, int style) {
		super(parent, style);
		
		this.initialize();
	}
	
	/**
	 * コンポジットを配置します。
	 */
	private void initialize() {
		
		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;
		
		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.numColumns = 15;
		this.setLayout(layout);
		
		/*
		 * 監視項目ID
		 */
		// ラベル
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("monitor.id") + " : ");
		// テキスト
		this.m_textMonitorId = new Text(this, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 8;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textMonitorId.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_64));
        this.m_textMonitorId.setLayoutData(gridData);
		
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		/*
		 * 説明
		 */
		// ラベル
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("description") + " : ");
		// テキスト
		this.m_textDescription = new Text(this, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 8;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textDescription.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_256));
        this.m_textDescription.setLayoutData(gridData);
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
	}
	
	/**
	 * 引数で指定された監視情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる監視情報
	 */
	public void setInputData(MonitorInfo info) {
		
		if(info != null){
			if (info.getMonitorId() != null) {
				this.m_textMonitorId.setText(info.getMonitorId());
				this.m_textMonitorId.setEnabled(false);
			}
			if (info.getDescription() != null) {
				this.m_textDescription.setText(info.getDescription());
			}
		}
	}
	
	/**
	 * 引数で指定された監視情報に、入力値を設定します。
	 * <p>
	 * 入力値チェックを行い、不正な場合は認証結果を返します。
	 * 不正ではない場合は、<code>null</code>を返します。
	 * 
	 * @param info 入力値を設定する監視情報
	 * @return 検証結果
	 * 
	 * @see #setValidateResult(String, String)
	 */
	public ValidateResult createInputData(MonitorInfo info) {
		
		if(info != null){
			if (this.m_textMonitorId.getText() != null
					&& !"".equals((this.m_textMonitorId.getText()).trim())) {
				info.setMonitorId(this.m_textMonitorId.getText());
			} else {
				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.1"));
			}
			if (this.m_textDescription.getText() != null
					&& !"".equals((this.m_textDescription.getText()).trim())) {
				info.setDescription(this.m_textDescription.getText());
			}
		}
		return null;
	}
	
	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		this.m_textMonitorId.setEnabled(enabled);
		this.m_textDescription.setEnabled(enabled);
	}
	
	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 * @return 認証結果
	 */
	protected ValidateResult setValidateResult(String id, String message) {
		
		ValidateResult validateResult = new ValidateResult();
		validateResult.setValid(false);
		validateResult.setID(id);
		validateResult.setMessage(message);
		
		return validateResult;
	}
}