/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.dialog;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.notify.action.DeleteNotify;
import com.clustercontrol.notify.bean.NotifyTabelDefine;
import com.clustercontrol.notify.composite.NotifyListComposite;
import com.clustercontrol.util.Messages;

/**
 * 通知[一覧]ダイアログクラスです。
 * 
 * @version 2.2.0
 * @since 2.0.0
 */
public class NotifyListDialog extends CommonDialog {

    /** 通知一覧 コンポジット。 */
    private NotifyListComposite notifyList = null;

    /** 追加 ボタン。 */
    private Button buttonAdd = null;

    /** 変更 ボタン。 */
    private Button buttonModify = null;

    /** 削除 ボタン。 */
    private Button buttonDelete = null;

    
    /**
     * ダイアログのインスタンスを返します。
     * 
     * @param parent 親のシェルオブジェクト
     */
    public NotifyListDialog(Shell parent) {
        super(parent);
        setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
    }

    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     */
    protected Point getInitialSize() {
        return new Point(800, 350);
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent 親のコンポジット
     * 
     * @see com.clustercontrol.notify.composite.NotifyListComposite
     */
    protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.notify.list"));

        // レイアウト
        GridLayout layout = new GridLayout(8, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 8;
        parent.setLayout(layout);

        /*
         * ログ一覧
         */

        this.notifyList = new NotifyListComposite(parent, SWT.BORDER);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 7;
        this.notifyList.setLayoutData(gridData);

        /*
         * 操作ボタン
         */

        Composite composite = new Composite(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.numColumns = 1;
        composite.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        composite.setLayoutData(gridData);

        // 追加ボタン
        this.buttonAdd = this
                .createButton(composite, Messages.getString("add"));
        this.buttonAdd.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	NotifyCreateDialog dialog = new NotifyCreateDialog(getParentShell());
            	if (dialog.open() == IDialogConstants.OK_ID) {
            		notifyList.update();
            	}
            }
        });

        // 変更ボタン
        @SuppressWarnings("unused") Label dummy = new Label(composite, SWT.NONE);
        this.buttonModify = this.createButton(composite, Messages.getString("modify"));
        this.buttonModify.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	
            	String notifyId = null;
            	
            	Table table = notifyList.getTableViewer().getTable();
            	TableItem[] item = table.getSelection();
            	
            	if(item != null && item.length>0){
            		try{
            			notifyId = (String)((ArrayList)item[0].getData()).get(NotifyTabelDefine.NOTIFY_ID);
            		}
            		catch(Exception ex){
            		}
            	}

                if (notifyId != null) {
                	NotifyCreateDialog dialog = new NotifyCreateDialog(getParentShell(), notifyId, false);
                	if (dialog.open() == IDialogConstants.OK_ID) {
                		int selectIndex = table.getSelectionIndex();
                		notifyList.update();
                		table.setSelection(selectIndex);
                	}
                }
                else{
                	MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.notify.8"));
                }
            }
        });

        // 削除ボタン
        dummy = new Label(composite, SWT.NONE);
        this.buttonDelete = this.createButton(composite, Messages
                .getString("delete"));
        this.buttonDelete.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	
            	String notifyId = null;
            	
            	Table table = notifyList.getTableViewer().getTable();
            	TableItem[] item = table.getSelection();
            	
            	if(item != null && item.length>0){
            		try{
            			notifyId = (String)((ArrayList)item[0].getData()).get(NotifyTabelDefine.NOTIFY_ID);
            		}
            		catch(Exception ex){
            		}
            	}

            	if (notifyId != null) {

            		String[] args = { notifyId };
            		if (MessageDialog.openConfirm(
                			null, 
                			Messages.getString("confirmed"), 
                			Messages.getString("message.notify.7", args))) {
            			
            			boolean result = new DeleteNotify().delete(notifyId);
            			if(result){
                        	MessageDialog.openInformation(
                        			null, 
                        			Messages.getString("successful"), 
                        			Messages.getString("message.notify.5", args));
            				
            				notifyList.update();
            			}
            			else{
                        	MessageDialog.openError(
                        			null, 
                        			Messages.getString("failed"), 
                        			Messages.getString("message.notify.6", args));
            			}
            		}
            	}
                else{
                	MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.notify.9"));
                }
            }
        });

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 8;
        line.setLayoutData(gridData);

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    protected ValidateResult validate() {
        return super.validate();
    }

    /**
     * ボタンを返します。
     * 
     * @param parent 親のコンポジット
     * @param label ラベル文字列
     * @return 生成されたボタン
     */
    private Button createButton(Composite parent, String label) {
        Button button = new Button(parent, SWT.NONE);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);

        button.setText(label);

        return button;
    }
    
    /**
     * ボタンを生成します。<BR>
     * 閉じるボタンを生成します。
     * 
     * @param parent ボタンバーコンポジット
     */
    protected void createButtonsForButtonBar(Composite parent) {
    	
        // 閉じるボタン
        this.createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CLOSE_LABEL, false);
    }

}