/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.collectiverun.bean;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * 一括制御情報をツリー構造化するためのクラス
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class TreeItem implements Serializable, Cloneable {
	private static final long serialVersionUID = -8195124905953255470L;

	/** 親 */
	protected TreeItem parent = null;

    /** 情報オブジェクト */
    protected ItemInfo info = null;

    /** 子の格納リスト */
    protected List<TreeItem> children = null;

    /**
     * 引数の情報を保持したインスタンスを返します。
     * 
     * @param parent
     *            親のオブジェクト
     * @param data
     *            スコープ情報オブジェクト
     */
    public TreeItem(TreeItem parent, ItemInfo info) {

        this.setParent(parent);
        this.setInfo(info);

        if (parent != null) {
            parent.addChildren(this);
        }

        this.children = new ArrayList<TreeItem>();
    }

    // ----- instance フィールド ----- //

    /**
     * 親を返します。
     * 
     * @return 親
     */
    public TreeItem getParent() {
        return this.parent;
    }

    /**
     * 親を設定します。
     * <p>
     * 
     * インスタンス化の際に親へ関係付けているため、子を削除するメソッドを実装した 後に可視性(スコープ)を拡大して下さい。 <br>
     * また、新しい親への関係付けも行うように実装して下さい。
     * 
     * @return 親
     */
    public void setParent(TreeItem parent) {
        this.parent = parent;
    }

    /**
     * 一括制御情報を返します。
     * 
     * @return 一括制御情報
     */
    public ItemInfo getInfo() {
        return this.info;
    }

    /**
     * 一括制御情報を設定します。
     * 
     * @param data 一括制御情報
     */
    public void setInfo(ItemInfo info) {
        this.info = info;
    }

    /**
     * 子を追加します。
     * <p>
     * 
     * この際、childeの親はこのオブジェクトとして設定されます。
     * 
     * @param child 子
     */
    public void addChildren(TreeItem child) {
        child.setParent(this);
        children.add(child);
    }
    
    /**
     * 子を削除します。
     * 
     * @param child
     */
    public void removeChildren(TreeItem child) {
        for (int i = 0; i < children.size(); i++) {
            if (child.equals(children.get(i))) {
                children.remove(i);
                break;
            }
        }
    }

    /**
     * 子の数を返します。
     * 
     * @return 子の数
     */
    public int size() {
        return children.size();
    }

    /**
     * 全ての子を返します。
     * <p>
     * 
     * 並び順は、追加された順となっています。
     * 
     * @return 全ての子。
     */
    public TreeItem[] getChildren() {
        TreeItem[] result = new TreeItem[this.size()];
        return (TreeItem[]) children.toArray(result);
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#clone()
     */
    public Object clone() {
        ItemInfo cloneInfo = (ItemInfo)this.getInfo().clone();
        if(cloneInfo == null)
        	return null;
        
        //FIXME super.clone()を呼び出し、clone()メソッドの規約を満たすようにインスタンスを生成する
//        TreeItem clone = new TreeItem(null, cloneInfo);
        TreeItem clone;
		try {
			clone = (TreeItem) super.clone();
			clone.setParent(null);
			clone.setInfo(cloneInfo);
		} catch (CloneNotSupportedException e) {
			return null;
		}
        
        return clone;
    }
}