/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterPK;

/**
 * ジョブコマンドマスタのDAOインターフェースを実装するクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterBean
 * @see com.clustercontrol.jobmanagement.dao.JobCommandMasterDAO
 */
public class JobCommandMasterDAOImpl implements JobCommandMasterDAO {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobCommandMasterDAOImpl.class);
	
	public void init() {
	}
	
	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobCommandMasterDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterPK, com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterBean)
	 */
	public void load(JobCommandMasterPK pk, JobCommandMasterBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_command_mst WHERE job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setArgument(res.getString("argument"));
				ejb.setArgument_job_id(res.getString("argument_job_id"));
				ejb.setEffective_user(res.getString("effective_user"));
				ejb.setError_end_flg(res.getInt("error_end_flg"));
				ejb.setError_end_value(res.getInt("error_end_value"));
				ejb.setFacility_id(res.getString("facility_id"));
				ejb.setJob_id(res.getString("job_id"));
				ejb.setProcess_mode(res.getInt("process_mode"));
				ejb.setStart_command(res.getString("start_command"));
				ejb.setStop_command(res.getString("stop_command"));
			} else {
				String msg = "JobCommandMaster data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}
	
	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobCommandMasterDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterBean)
	 */
	public void store(JobCommandMasterBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getJob_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_job_command_mst SET ");
			sql.append("facility_id = ?, ");
			sql.append("process_mode = ?, ");
			sql.append("start_command = ?, ");
			sql.append("stop_command = ?, ");
			sql.append("effective_user = ?, ");
			sql.append("error_end_flg = ?, ");
			sql.append("error_end_value = ?, ");
			sql.append("argument_job_id = ?, ");
			sql.append("argument = ? ");
			sql.append("WHERE job_id = ?");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getFacility_id());
			stmt.setInt(2, ejb.getProcess_mode());
			stmt.setString(3, ejb.getStart_command());
			stmt.setString(4, ejb.getStop_command());
			stmt.setString(5, ejb.getEffective_user());
			stmt.setInt(6, ejb.getError_end_flg());
			stmt.setInt(7, ejb.getError_end_value());
			stmt.setString(8, ejb.getArgument_job_id());
			stmt.setString(9, ejb.getArgument());
			stmt.setString(10, ejb.getJob_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getJob_id() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getJob_id() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getJob_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : " + ejb.getJob_id());
	}
	
	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobCommandMasterDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterPK)
	 */
	public void remove(JobCommandMasterPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_command_mst WHERE job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}
	
	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobCommandMasterDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterBean)
	 */
	public JobCommandMasterPK create(JobCommandMasterBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : " + ejb.getJob_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		JobCommandMasterPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_command_mst (");
			sql.append("job_id, ");
			sql.append("facility_id, ");
			sql.append("process_mode, ");
			sql.append("start_command, ");
			sql.append("stop_command, ");
			sql.append("effective_user, ");
			sql.append("error_end_flg, ");
			sql.append("error_end_value, ");
			sql.append("argument_job_id, ");
			sql.append("argument) ");
			sql.append("VALUES (?,?,?,?,?,?,?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getJob_id());
			stmt.setString(2, ejb.getFacility_id());
			stmt.setInt(3, ejb.getProcess_mode());
			stmt.setString(4, ejb.getStart_command());
			stmt.setString(5, ejb.getStop_command());
			stmt.setString(6, ejb.getEffective_user());
			stmt.setInt(7, ejb.getError_end_flg());
			stmt.setInt(8, ejb.getError_end_value());
			stmt.setString(9, ejb.getArgument_job_id());
			stmt.setString(10, ejb.getArgument());
			
			int row =	stmt.executeUpdate();
			pk = new JobCommandMasterPK(
					ejb.getJob_id());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getJob_id() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getJob_id() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getJob_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getJob_id());
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobCommandMasterDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobCommandMasterPK> ret = new ArrayList<JobCommandMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_command_mst";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobCommandMasterPK pk= new JobCommandMasterPK(res.getString("job_id"));
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}
	
	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobCommandMasterDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterPK)
	 */
	public JobCommandMasterPK findByPrimaryKey(JobCommandMasterPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_command_mst WHERE job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたファシリティIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterBean#ejbFindByFacilityId(String)
	 * @see com.clustercontrol.jobmanagement.dao.JobCommandMasterDAO#findByFacilityId(java.lang.String)
	 */
	public Collection findByFacilityId(String facilityId) throws FinderException {
		m_log.debug("findByFacilityId() start : " + facilityId);
		
		ArrayList<JobCommandMasterPK> ret = new ArrayList<JobCommandMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_command_mst WHERE facility_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, facilityId);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobCommandMasterPK pk= new JobCommandMasterPK(res.getString("job_id"));
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findByFacilityId() error : " + facilityId + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByFacilityId() error : " + facilityId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByFacilityId() end : " + facilityId);
		return ret;
	}
}
