/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoPK;

/**
 * ジョブ情報のDAOインターフェースを実装するクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean
 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO
 */
public class JobInfoDAOImpl implements JobInfoDAO {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobInfoDAOImpl.class);

	public void init() {
	}
	
	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobInfoPK, com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean)
	 */
	public void load(JobInfoPK pk, JobInfoBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_info WHERE session_id = ? AND job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setDescription(res.getString("description"));
				ejb.setJob_id(res.getString("job_id"));
				ejb.setJob_name(res.getString("job_name"));
				ejb.setJob_type(res.getInt("job_type"));
				if(res.getTimestamp("reg_date") != null)
					ejb.setReg_date(new Date(res.getTimestamp("reg_date").getTime()));
				else
					ejb.setReg_date(null);
				ejb.setReg_user(res.getString("reg_user"));
				ejb.setSession_id(res.getString("session_id"));
				if(res.getTimestamp("update_date") != null)
					ejb.setUpdate_date(new Date(res.getTimestamp("update_date").getTime()));
				else
					ejb.setUpdate_date(null);
				ejb.setUpdate_user(res.getString("update_user"));
			} else {
				String msg = "JobInfo data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}
	
	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean)
	 */
	public void store(JobInfoBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getSession_id() + ", " + ejb.getJob_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_job_info SET ");
			sql.append("job_name = ?, ");
			sql.append("description = ?, ");
			sql.append("job_type = ?, ");
			sql.append("reg_date = ?, ");
			sql.append("update_date = ?, ");
			sql.append("reg_user = ?, ");
			sql.append("update_user = ? ");
			sql.append("WHERE session_id = ? AND ");
			sql.append("job_id = ?");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getJob_name());
			stmt.setString(2, ejb.getDescription());
			stmt.setInt(3, ejb.getJob_type());
			if(ejb.getReg_date() instanceof java.util.Date)
				stmt.setTimestamp(4, new Timestamp(ejb.getReg_date().getTime()));
			else
				stmt.setTimestamp(4, null);
			if(ejb.getUpdate_date() instanceof java.util.Date)
				stmt.setTimestamp(5, new Timestamp(ejb.getUpdate_date().getTime()));
			else
				stmt.setTimestamp(5, null);
			stmt.setString(6, ejb.getReg_user());
			stmt.setString(7, ejb.getUpdate_user());
			stmt.setString(8, ejb.getSession_id());
			stmt.setString(9, ejb.getJob_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getSession_id() + ", "
					+ ejb.getJob_id() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : " + ejb.getSession_id() + ", " + ejb.getJob_id());
	}
	
	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobInfoPK)
	 */
	public void remove(JobInfoPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_info WHERE session_id = ? AND job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}
	
	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean)
	 */
	public JobInfoPK create(JobInfoBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : " + ejb.getSession_id() + ", " + ejb.getJob_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		JobInfoPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_info (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("job_name, ");
			sql.append("description, ");
			sql.append("job_type, ");
			sql.append("reg_date, ");
			sql.append("update_date, ");
			sql.append("reg_user, ");
			sql.append("update_user) ");
			sql.append("VALUES (?,?,?,?,?,?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getSession_id());
			stmt.setString(2, ejb.getJob_id());
			stmt.setString(3, ejb.getJob_name());
			stmt.setString(4, ejb.getDescription());
			stmt.setInt(5, ejb.getJob_type());
			if(ejb.getReg_date() instanceof java.util.Date)
				stmt.setTimestamp(6, new Timestamp(ejb.getReg_date().getTime()));
			else
				stmt.setTimestamp(6, null);
			if(ejb.getUpdate_date() instanceof java.util.Date)
				stmt.setTimestamp(7, new Timestamp(ejb.getUpdate_date().getTime()));
			else
				stmt.setTimestamp(7, null);
			stmt.setString(8, ejb.getReg_user());
			stmt.setString(9, ejb.getUpdate_user());
			
			int row =	stmt.executeUpdate();
			pk = new JobInfoPK(
					ejb.getSession_id(),
					ejb.getJob_id());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getSession_id() + ", " + ejb.getJob_id());
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobInfoPK> ret = new ArrayList<JobInfoPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_info";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobInfoPK pk= new JobInfoPK(
						res.getString("session_id"),
						res.getString("job_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}
	
	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobInfoPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobInfoPK)
	 */
	public JobInfoPK findByPrimaryKey(JobInfoPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_info WHERE session_id = ? AND job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
	
	/**
	 * ジョブ情報をジョブマスタから作成します。
	 * 
	 * @param sessionId セッションID
	 * @param jobs ジョブIDのコレクション
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean#ejbHomeMakeSession(String, Collection)
	 */
	public void makeSession(String sessionId, Collection jobs) throws CreateException{
		m_log.debug("makeSession() start : " + sessionId );
		
		Connection conn = null;
		Connection connMaster = null;
		PreparedStatement stmt = null;
		PreparedStatement stmtMaster = null;
		String jobId = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			connMaster = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義(INSERT)
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_info (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("job_name, ");
			sql.append("description, ");
			sql.append("job_type, ");
			sql.append("reg_date, ");
			sql.append("update_date, ");
			sql.append("reg_user, ");
			sql.append("update_user) ");
			sql.append("VALUES (?,?,?,?,?,?,?,?,?)");

			//SQL文のセット(INSERT)
			stmt = conn.prepareStatement(sql.toString());
			stmt.setString(1, sessionId);
			
			//SQL文の定義(マスタ検索)
			StringBuilder sqlMaster = new StringBuilder();
			sqlMaster.append("SELECT * FROM cc_job_mst ");
			sqlMaster.append("WHERE cc_job_mst.job_id = ?");
			
			//SQL文のセット(マスタ検索)
			stmtMaster = connMaster.prepareStatement(sqlMaster.toString());
			
			Iterator it = jobs.iterator();
			while(it.hasNext()){
				//ジョブID取得
				Hashtable ht = (Hashtable)it.next();
				jobId = (String)ht.get("jobId");
				
				//SQL実行(マスタ検索)
				stmtMaster.setString(1, jobId);
				ResultSet resMaster = stmtMaster.executeQuery();
				
				if (resMaster.next()) {
					//ジョブの場合、INSERTを実行
					stmt.setString(2, resMaster.getString("job_id"));
					stmt.setString(3, resMaster.getString("job_name"));
					stmt.setString(4, resMaster.getString("description"));
					stmt.setInt(5, resMaster.getInt("job_type"));
					stmt.setTimestamp(6, resMaster.getTimestamp("reg_date"));
					stmt.setTimestamp(7, resMaster.getTimestamp("update_date"));
					stmt.setString(8, resMaster.getString("reg_user"));
					stmt.setString(9, resMaster.getString("update_user"));

					int row =	stmt.executeUpdate();
					if (row != 1) {
						String msg = "result row is not 1";
						m_log.error("makeSession() error : " + sessionId + ", " + resMaster.getString("job_id") + " SQLException");
						throw new EJBException(msg);
					}
				}
				resMaster.close();
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
				if(stmtMaster != null){
					stmtMaster.close();
				}
				if(connMaster != null){
					connMaster.close();
				}
			} catch (SQLException e1) {
				m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("makeSession() end : " + sessionId );
	}
}
