/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterPK;

/**
 * ジョブマスタのDAOインターフェースを実装するクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobMasterBean
 * @see com.clustercontrol.jobmanagement.dao.JobMasterDAO
 */
public class JobMasterDAOImpl implements JobMasterDAO {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobMasterDAOImpl.class);
	
	public void init() {
	}
	
	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobMasterDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobMasterPK, com.clustercontrol.jobmanagement.ejb.entity.JobMasterBean)
	 */
	public void load(JobMasterPK pk, JobMasterBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_mst WHERE job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setDescription(res.getString("description"));
				ejb.setJob_id(res.getString("job_id"));
				ejb.setJob_name(res.getString("job_name"));
				ejb.setJob_type(res.getInt("job_type"));
				if(res.getTimestamp("reg_date") != null)
					ejb.setReg_date(new Date(res.getTimestamp("reg_date").getTime()));
				else
					ejb.setReg_date(null);
				ejb.setReg_user(res.getString("reg_user"));
				if(res.getTimestamp("update_date") != null)
					ejb.setUpdate_date(new Date(res.getTimestamp("update_date").getTime()));
				else
					ejb.setUpdate_date(null);
				ejb.setUpdate_user(res.getString("update_user"));
			} else {
				String msg = "JobMaster data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}
	
	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobMasterDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobMasterBean)
	 */
	public void store(JobMasterBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getJob_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_job_mst SET ");
			sql.append("job_name = ?, ");
			sql.append("description = ?, ");
			sql.append("job_type = ?, ");
			sql.append("reg_date = ?, ");
			sql.append("update_date = ?, ");
			sql.append("reg_user = ?, ");
			sql.append("update_user = ? ");
			sql.append("WHERE job_id = ?");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getJob_name());
			stmt.setString(2, ejb.getDescription());
			stmt.setInt(3, ejb.getJob_type());
			if(ejb.getReg_date() instanceof java.util.Date)
				stmt.setTimestamp(4, new Timestamp(ejb.getReg_date().getTime()));
			else
				stmt.setTimestamp(4, null);
			if(ejb.getUpdate_date() instanceof java.util.Date)
				stmt.setTimestamp(5, new Timestamp(ejb.getUpdate_date().getTime()));
			else
				stmt.setTimestamp(5, null);
			stmt.setString(6, ejb.getReg_user());
			stmt.setString(7, ejb.getUpdate_user());
			stmt.setString(8, ejb.getJob_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getJob_id() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getJob_id() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getJob_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : " + ejb.getJob_id());
	}
	
	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobMasterDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobMasterPK)
	 */
	public void remove(JobMasterPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_mst WHERE job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}
	
	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobMasterDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobMasterBean)
	 */
	public JobMasterPK create(JobMasterBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : " + ejb.getJob_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		JobMasterPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_mst (");
			sql.append("job_id, ");
			sql.append("job_name, ");
			sql.append("description, ");
			sql.append("job_type, ");
			sql.append("reg_date, ");
			sql.append("update_date, ");
			sql.append("reg_user, ");
			sql.append("update_user) ");
			sql.append("VALUES (?,?,?,?,?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getJob_id());
			stmt.setString(2, ejb.getJob_name());
			stmt.setString(3, ejb.getDescription());
			stmt.setInt(4, ejb.getJob_type());
			if(ejb.getReg_date() instanceof java.util.Date)
				stmt.setTimestamp(5, new Timestamp(ejb.getReg_date().getTime()));
			else
				stmt.setTimestamp(5, null);
			if(ejb.getUpdate_date() instanceof java.util.Date)
				stmt.setTimestamp(6, new Timestamp(ejb.getUpdate_date().getTime()));
			else
				stmt.setTimestamp(6, null);
			stmt.setString(7, ejb.getReg_user());
			stmt.setString(8, ejb.getUpdate_user());
			
			int row =	stmt.executeUpdate();
			pk = new JobMasterPK(
					ejb.getJob_id());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getJob_id() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getJob_id() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getJob_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getJob_id());
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobMasterBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobMasterDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobMasterPK> ret = new ArrayList<JobMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_mst";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobMasterPK pk= new JobMasterPK(
						res.getString("job_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}
	
	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobMasterBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobMasterPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobMasterDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobMasterPK)
	 */
	public JobMasterPK findByPrimaryKey(JobMasterPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_mst WHERE job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
}
