/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.ejb.session;

import java.rmi.RemoteException;
import java.text.ParseException;
import java.util.ArrayList;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.SchedulerException;
import org.quartz.jobs.ee.ejb.EJBInvokerJob;

import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.jobmanagement.bean.OperationConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodePK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeUtil;
import com.clustercontrol.jobmanagement.factory.JobOperationJudgment;
import com.clustercontrol.jobmanagement.factory.JobOperationProperty;
import com.clustercontrol.jobmanagement.factory.OperateMaintenanceOfJob;
import com.clustercontrol.jobmanagement.factory.OperateSkipOfJob;
import com.clustercontrol.jobmanagement.factory.OperateStartOfJob;
import com.clustercontrol.jobmanagement.factory.OperateStopOfJob;
import com.clustercontrol.jobmanagement.factory.OperateSuspendOfJob;
import com.clustercontrol.jobmanagement.factory.OperateWaitOfJob;
import com.clustercontrol.jobmanagement.factory.OperationJob;
import com.clustercontrol.jobmanagement.message.RunResultInfo;
import com.clustercontrol.quartzmanager.ejb.session.QuartzManager;
import com.clustercontrol.quartzmanager.util.QuartzUtil;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.util.apllog.AplLogger;


/**
 * ジョブ管理機能の実行管理を行う Session Bean です。
 * 
 * @ejb.bean name="JobRunManagement"	
 *           jndi-name="JobRunManagement"
 *           type="Stateless" 
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=RepositoryController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=QuartzManager"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CalendarController"
 * 
 * @ejb.permission
 *     unchecked="true"
 */
public abstract class JobRunManagementBean implements javax.ejb.SessionBean {
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( JobRunManagementBean.class );
    
	/** Quartzに設定するグループ名 */
	public static final String GROUP_NAME = "JOB_MANAGEMENT";
    
    /**
     * Quartzへジョブ管理の定周期処理を登録します。<BR>
	 * Quartzからは、{@link com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#run()} が呼び出されます。
	 * 
     * @ejb.interface-method
     * 
     * @param cronString cronの定義文字列
     * @throws NamingException
     * @throws ParseException
     * @throws SchedulerException
     * 
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#addSchedule(org.quartz.JobDetail, org.quartz.Trigger)
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#deleteSchedule(java.lang.String, java.lang.String)
     * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementHome#JNDI_NAME
     * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#GROUP_NAME
     */
    public void addQuartz(String cronString) throws NamingException, ParseException, SchedulerException {
        //QuartzのSchedulerをルックアップ
		QuartzManager manager = QuartzUtil.getQuartzManager();
		
        try {
            //ジョブ削除
        	manager.deleteSchedule(JobRunManagementHome.JNDI_NAME, 
                    GROUP_NAME);
        } catch (SchedulerException e) {
            m_log.debug("addQuartz() : deleteJob error = " + e.getMessage());
        } catch (RemoteException e) {
        	//TODO RemoteException未実装
		}
        
        //JobDetail作成
        JobDetail job = new JobDetail(
        		JobRunManagementHome.JNDI_NAME, 
                GROUP_NAME, 
                EJBInvokerJob.class);
        
        String serverName = System.getProperty("jboss.server.name");
	    if(serverName.equals("all")){
	        job.getJobDataMap().put(
	                EJBInvokerJob.PROVIDER_URL, 
	                "jnp://localhost:1100");
	    }
	    
        //ジョブ完了時に削除されないようにする。
        job.setDurability(true);
	    //ジョブ実行失敗時に再実行するようにする。
	    job.setRequestsRecovery(true);
	    
        //JobDetailに呼び出すクラスとメソッドを設定
        job.getJobDataMap().put(
                EJBInvokerJob.EJB_JNDI_NAME_KEY, 
                JobRunManagementHome.JNDI_NAME);
        job.getJobDataMap().put(
                EJBInvokerJob.EJB_METHOD_KEY, 
                "run");
        
        //JobDetailに呼び出すメソッドの引数を設定
        Object[] jdArgs = new Object[0];
        job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
        
        //CronTriggerを作成
        CronTrigger cronTrigger = new CronTrigger(JobRunManagementHome.JNDI_NAME, GROUP_NAME);
        
        //スケジュールを設定
        cronTrigger.setCronExpression(cronString);
        
        //Schedulerにジョブを登録する
        try {
			manager.addSchedule(job, cronTrigger);
		} catch (RemoteException e) {
			//TODO RemoteException未実装
		}
    }
    
    /**
     * Quartzからジョブ管理の定周期処理を削除します。
     * 
     * @ejb.interface-method
     * 
     * @throws NamingException
     * @throws ParseException
     * @throws SchedulerException
     * 
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#deleteSchedule(java.lang.String, java.lang.String)
     * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementHome#JNDI_NAME
     * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#GROUP_NAME
     */
    public void deleteQuartz() throws NamingException, ParseException, SchedulerException {
        //QuartzのSchedulerをルックアップ
    	QuartzManager manager = QuartzUtil.getQuartzManager();
		
        try {
            //ジョブ削除
        	manager.deleteSchedule(JobRunManagementHome.JNDI_NAME, 
                    GROUP_NAME);
        } catch (SchedulerException e) {
            m_log.debug("deleteQuartz() : deleteJob error = " + e.getMessage());
        } catch (RemoteException e) {
			//TODO RemoteException未実装
		}
    }
    
    /**
     * Quartzからのコールバックメソッド
     * <P>
     * Quartzから定周期で呼び出されます。<BR>
     * <BR>
     * 実行状態が実行中のセッションをチェックし、実行可能なジョブを開始する。<BR>
     * 実行状態が待機のセッションをチェックし、ジョブを開始する。
     * 
     * @ejb.interface-method
     * 
     * @throws FinderException
     * @throws NamingException
     * 
     * @see com.clustercontrol.jobmanagement.factory.OperationJob#runJob()
     * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#runWaitJob()
     */
    public void run() throws FinderException, NamingException {
    	m_log.debug("run()");
    	
    	OperationJob operation = new OperationJob();
    	operation.runJob();
    	
    	//実行状態が待機のジョブユニットを実行する
    	try {
    		JobRunManagementLocal jobRunManagement = JobRunManagementUtil.getLocalHome().create();
			jobRunManagement.runWaitJob();
        } catch (FinderException e) {
            m_log.debug("run() : " + e.getMessage());
            throw e;
        } catch (NamingException e) {
            m_log.debug("run() : " + e.getMessage());
            throw e;
        } catch (CreateException e) {
            m_log.debug("run() : " + e.getMessage());
		}
    }
    
    /**
     * 実行状態が待機のセッションのジョブを実行します。<P>
     * 
     * 実行状態の待機のセッションをチェックし、ジョブを開始する。
     * 
     * @ejb.interface-method
     * 
     * @ejb.transaction
	 *     type="RequiresNew"
     * 
     * @throws NamingException
     * @throws FinderException
     * 
     * @see com.clustercontrol.jobmanagement.factory.OperationJob#runWaitJob()
     */
    public void runWaitJob() throws NamingException, FinderException {
        m_log.debug("runWaitJob()");
        
        OperationJob operation = new OperationJob();
        operation.runWaitJob();
    }
    
    /**
     * 実行状態が停止処理中のセッションのジョブをチェックします。<P>
     * 
     * 実行状態の停止処理中のセッションをチェックし、ジョブを停止する。
     * 
     * @ejb.interface-method
     * 
     * @ejb.transaction
	 *     type="RequiresNew"
     * 
     * @throws NamingException
     * @throws FinderException
     * 
     * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkStoppingJob()
     */
    public void checkStoppingJob() throws NamingException, FinderException {
        m_log.debug("checkStoppingJob()");
        
        OperationJob operation = new OperationJob();
        operation.checkStoppingJob();
    }
    
    /**
     * ジョブを実行します。
     * 
     * @ejb.interface-method
     * 
     * @ejb.transaction
	 *     type="RequiresNew"
     * 
     * @param sessionId セッションID
     * @param jobId ジョブID
     * @throws FinderException
     * @throws NamingException
     * 
     * @see com.clustercontrol.jobmanagement.factory.OperationJob#runJob(String, String)
     */
    public void runJob(String sessionId, String jobId) throws FinderException, NamingException {
        m_log.debug("runJob() : sessionId=" + sessionId + ", jobId=" + jobId);
        
        OperationJob operation = new OperationJob();
        operation.runJob(sessionId, jobId);
    }

	/**
	 * ジョブの操作を行います。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param property ジョブ操作用プロパティ
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.OperateStartOfJob#startJob(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateSuspendOfJob#suspendJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateSuspendOfJob#releaseSuspendJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateWaitOfJob#waitJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateWaitOfJob#releaseWaitJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateSkipOfJob#skipJob(String, String, Integer)
	 * @see com.clustercontrol.jobmanagement.factory.OperateSkipOfJob#releaseSkipJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateStopOfJob#stopJob(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateMaintenanceOfJob#maintenanceJob(String, String, String, Integer)
	 */
	public void operationJob(Property property) throws FinderException, NamingException {
	    m_log.debug("operationJob()");
	    
		//セッションID取得
		ArrayList values = PropertyUtil.getPropertyValue(property, JobOperationProperty.SESSION);
		String sessionId = null;
		if(values.get(0) instanceof String && ((String)values.get(0)).length() > 0)
		    sessionId = (String)values.get(0);
		
		//ジョブID取得
		values = PropertyUtil.getPropertyValue(property, JobOperationProperty.JOB);
		String jobId = null;
		if(values.get(0) instanceof String && ((String)values.get(0)).length() > 0)
		    jobId = (String)values.get(0);
		
		//ファシリティID取得
		values = PropertyUtil.getPropertyValue(property, JobOperationProperty.FACILITY);
		String facilityId = null;
		if(values.size() > 0 && values.get(0) instanceof String && ((String)values.get(0)).length() > 0)
		    facilityId = (String)values.get(0);
		
		//制御取得
		values = PropertyUtil.getPropertyValue(property, JobOperationProperty.CONTROL);
		Integer control = null;
		if(values.get(0) instanceof String){
		    String controlString = (String)values.get(0);
		    control = Integer.valueOf(OperationConstant.stringToType(controlString));
		}
		
		//終了値取得
		values = PropertyUtil.getPropertyValue(property, JobOperationProperty.END_VALUE);
		Integer endValue = null;
		if(values.size() > 0 && values.get(0) instanceof Integer)
		    endValue = (Integer)values.get(0);

	    int status = 0;
	    int jobType = 0;
	    
	    try {
            if(facilityId != null && facilityId.length() > 0){
                //セッションIDとジョブIDから、セッションジョブを取得
                JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
                            new JobSessionJobPK(sessionId, jobId));
                
                //ジョブタイプを取得
                if(sessionJob.getJobInfo().getJob_type().intValue() == JobConstant.TYPE_FILEJOB){
                	jobId = jobId + "_" + facilityId;
                	facilityId = null;
                	
                	//セッションIDとジョブIDから、セッションジョブを取得
                    JobSessionJobLocal childSessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
                            new JobSessionJobPK(sessionId, jobId));
	                
	                //実行状態を取得
	                status = childSessionJob.getStatus().intValue();
	                
	                //ジョブタイプを取得
	                jobType = JobOperationJudgment.TYPE_JOBNET;
                }
                else{
	                //セッションIDとジョブIDから、セッションジョブを取得
	                JobSessionNodeLocal sessionNode = 
	                    JobSessionNodeUtil.getLocalHome().findByPrimaryKey(
	                            new JobSessionNodePK(sessionId, jobId, facilityId));
	                
	                //実行状態を取得
	                status = sessionNode.getStatus().intValue();
	                
	                //ジョブタイプを取得
	                jobType = JobOperationJudgment.TYPE_NODE;
                }
            }
            else{
                //セッションIDとジョブIDから、セッションジョブを取得
                JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
                            new JobSessionJobPK(sessionId, jobId));
                
                //実行状態を取得
                status = sessionJob.getStatus().intValue();
                
                //ジョブタイプを取得
                if(sessionJob.getJobInfo().getJob_type().intValue() == JobConstant.TYPE_JOB){
                    jobType = JobOperationJudgment.TYPE_JOB;
                }
                else{
                    jobType = JobOperationJudgment.TYPE_JOBNET;
                }
            }
        } catch (Exception e) {

        }
        
        //ジョブタイプとジョブの実行状態から操作可能かチェック
        if(JobOperationJudgment.judgment(control.intValue(), jobType, status)){
            if(control.intValue() == OperationConstant.TYPE_START_AT_ONCE){
                //開始[即時]
        	    try {
                    new OperateStartOfJob().startJob(sessionId, jobId, facilityId);
                } catch (FinderException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "007", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "007", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (EJBException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "007", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
				} catch (RemoveException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "007", args);
                    m_log.debug("operationJob() : " + e.getMessage());
				}
            }
            else if(control.intValue() == OperationConstant.TYPE_START_SUSPEND){
                //開始[中断解除]
                try {
                    new OperateSuspendOfJob().releaseSuspendJob(sessionId, jobId);
                } catch (FinderException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "008", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "008", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_START_WAIT){
                //開始[保留解除]
                try {
                    new OperateWaitOfJob().releaseWaitJob(sessionId, jobId);
                } catch (FinderException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "009", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "009", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_START_SKIP){
                //開始[スキップ解除]
                try {
                    new OperateSkipOfJob().releaseSkipJob(sessionId, jobId);
                } catch (FinderException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "010", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "010", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_STOP_AT_ONCE){
                try {
                    //停止[コマンド]
                    new OperateStopOfJob().stopJob(sessionId, jobId, facilityId);
                } catch (FinderException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "011", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "011", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_STOP_SUSPEND){
                try {
                    //停止[中断]
                    new OperateSuspendOfJob().suspendJob(sessionId, jobId);
                } catch (FinderException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "012", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "012", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_STOP_WAIT){
                try {
                    //停止[保留]
                    new OperateWaitOfJob().waitJob(sessionId, jobId);
                } catch (FinderException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "012", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "012", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_STOP_SKIP){
                //停止[スキップ]
                if(endValue == null){
                    throw new NullPointerException();
                }
                try {
                    new OperateSkipOfJob().skipJob(sessionId, jobId, endValue);
                } catch (FinderException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "013", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "013", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_STOP_MAINTENANCE){
                //停止[終了値の変更]
                if(endValue == null){
                    throw new NullPointerException();
                }
                try {
                    new OperateMaintenanceOfJob().maintenanceJob(sessionId, jobId, facilityId, endValue);
                } catch (FinderException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "014", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "014", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                }
            }
        }
        else{
            throw new IllegalStateException();
        }
	}
	
	/**
	 * ノード終了処理を行います。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param info 実行結果情報
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#endNode(RunResultInfo)
	 */
	public void endNode(RunResultInfo info) throws FinderException, NamingException {
	    m_log.debug("endNode() : sessionId=" + info.getSessionId() + ", jobId=" + info.getJobId() + ", facilityId=" + info.getFacilityId());
	    
	    OperationJob operation = new OperationJob();
	    operation.endNode(info);
	}
}
