/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.logagent;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;
import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.logtransfer.bean.TopicConstant;




/**
 * Topic受信クラス
 * Topicへの接続と、メッセージの受信を行う。
 * @version 2.0.0
 * @since 2.0.0
 * 
 */
public class ReceiveTopic implements MessageListener, ExceptionListener {

	private static long RETRY_INTERVAL = 10000;

	private static final String TOPIC_CON_FACTORY = "ConnectionFactory";
	private static final String TOPIC_USER_NAME = "topic.user.name";
	private static final String TOPIC_USER_PASSWORD = "topic.user.password";

	protected ArrayList m_facilityIdList;
    protected SendQueue m_sendQueue;

	private Properties m_props;

	private TopicConnectionFactory m_factory;

    protected TopicConnection m_con;
    protected Topic m_topic;
    protected TopicSession m_session;
    protected TopicSubscriber m_subscriber;
    
    protected TransferLogManager m_logManager;
	
	private boolean m_errFlg = false;

	/** メッセージサービス再接続用タイマー * */
	protected static Timer m_timer = new Timer(true);

	//ロガー
	static private Log log = LogFactory.getLog(ReceiveTopic.class);

	
	/**
	 * コンストラクタ
	 * @param facilityIdList ファシリティIDのリスト　
	 * @param sendQueue　メッセージ送信用クラス
	 * @param props　プロパティファイル情報
	 */
	@SuppressWarnings("unchecked")
	public ReceiveTopic( Collection facilityIdList, TransferLogManager logManager, SendQueue sendQueue, Properties props) {
		super();
		m_logManager = logManager;
		m_facilityIdList = new ArrayList(facilityIdList);
		m_sendQueue = sendQueue;
		m_props = props;

		//接続処理
        initial();
	}
    
    
    /**
     * ファシリティIDリスト設定.
     * 設定されたファイシリティIDのリストが、前回と違う場合は、トピック受信の再接続を行う
     * @param facilityIdList　
     * @since
     */
	@SuppressWarnings("unchecked")
	synchronized public void setFacilityIdList(Collection facilityIdList) {
	
		//取得した情報が変化しているかどうか確認
		if(facilityIdList.size() == m_facilityIdList.size()){
			if( m_facilityIdList.containsAll(facilityIdList) ){
				return;
			}
		}

		//追加分リスト
		ArrayList<String> addFacilityList = new ArrayList<String>(facilityIdList);
		addFacilityList.removeAll(m_facilityIdList);
		
		//削除分リスト
		ArrayList<String> removefacilityList = new ArrayList<String>(m_facilityIdList);
		removefacilityList.removeAll(facilityIdList);
		
		
		//削除分をスレッド停止
		for (Iterator iter = removefacilityList.iterator(); iter.hasNext();) {
			String facility = (String) iter.next();
			//スレッドに削除依頼
			m_logManager.removeTransferLogInfo(facility);
			
		}
		
		
		//ファシリティリスト更新
		m_facilityIdList = new ArrayList(facilityIdList);

		//Topic再接続
		if( !isErrFlg() ){
		    if( initialTopic() ){

				//追加分のログ情報を要求送信
				for (Iterator iter = m_facilityIdList.iterator(); iter.hasNext();) {
					m_sendQueue.put(iter.next());	
				}
			}
		
		}

		
	}

	
	/* 
     * トピック受信処理
     * (non-Javadoc)
     * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
     */
    @SuppressWarnings("unchecked")
	public void onMessage(Message message) {
        
        
        log.debug("onMessage start");
        String facilityId;
        try {
			facilityId = message.getStringProperty("FacilityId");
		} catch (JMSException e1) {
			log.error("受信メッセージが不正", e1);
			return;
		}
        
        if(message instanceof ObjectMessage){
            ObjectMessage objectMessage = (ObjectMessage)message;
            Object obj;
			try {
				obj = objectMessage.getObject();
			} catch (JMSException e) {
				log.error("受信メッセージが不正", e);
				return;
			}

			if(obj instanceof ArrayList){
            	log.debug("onMessage get ArrayList");
                
            	ArrayList list = (ArrayList)obj;
				// ログ転送
            	m_logManager.setTransferLogInfo(facilityId, list);
                
            }else{
				log.error("受信メッセージが不正" + obj.toString());
				return;
            }
        }else{
			log.error("受信メッセージが不正" + message.getClass());
			return;
        }
        log.debug("onMessage end");
    }
    
    /* 通信エラーハンドラ
     * (non-Javadoc)
     * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
     */
    public void onException(JMSException arg0) {

    	log.error(arg0);
    	
        setErrFlg(true);
    }

    /**
	 * @param errFlg
	 *            errFlg を設定。
	 */
	synchronized private void setErrFlg(boolean errFlg) {
		if (m_errFlg == false && errFlg == true) {

			m_timer.schedule(new ReSetupTask(), RETRY_INTERVAL,
					RETRY_INTERVAL);

		}
		m_errFlg = errFlg;
	}

	/**
	 * @return errFlg を戻します。
	 */
	synchronized private boolean isErrFlg() {
		return m_errFlg;
	}
	/**
	 *  
	 */
	synchronized private boolean reInitial() {

		boolean ret = false;

		log.info("再接続処理!");

		terminate();

		if (initial()) {

			ret = true;

			log.info("再接続処理:成功!");

			
			//エラーフラグ解除
			setErrFlg(false);


		} else {
			log.info("再接続処理:失敗!");
		}

		return ret;
	}
	/**
	 * サーバ接続の終了処理
	 *  
	 */
	public void terminate() {

		terminateSumscriber();

		
        try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
		}

		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e1) {
		}
	}
	/**
	 * トピック受信処理の終了 
	 */
	private void terminateSumscriber() {
		try {
			if (m_subscriber != null)
		        m_subscriber.close();
		} catch (JMSException e) {
		}
	}



	/**
	 * 初期化処理
	 * JMSへの接続、トピック受信設定を行う
	 * @return
	 */
	private boolean initial() {

		log.info("EJB接続初期化");

		InitialContext con = null;

		try {
			//InitialContextの生成
			con = new InitialContext(m_props);

			//コネクションファクトリ生成
			m_factory = (TopicConnectionFactory) con.lookup(TOPIC_CON_FACTORY);

			//コネクション生成
			if (m_props.getProperty(TOPIC_USER_NAME) != null) {
				//ユーザ認証
				m_con = m_factory.createTopicConnection(m_props
						.getProperty(TOPIC_USER_NAME), m_props
						.getProperty(TOPIC_USER_PASSWORD));
			} else {
				//ユーザ認証なし
				m_con = m_factory.createTopicConnection();
			}

			
			//セッション生成
			m_session = m_con.createTopicSession(false,
					Session.AUTO_ACKNOWLEDGE);

			//メッセージTopic取得
			m_topic = (Topic) con.lookup(TopicConstant.TOPIC_NAME_UPDATE);

			
			//エラーハンドらセット
	        m_con.setExceptionListener(this);

	        m_con.start();

	        //トピック接続開始
	        initialTopic();

			//ファシリティIDをマネージャに通知（メッセージ送信）
			for (Iterator iter = m_facilityIdList.iterator(); iter.hasNext();) {
				m_sendQueue.put(iter.next());	
			}


		} catch (Exception e) {
			log.error("Init", e);
			setErrFlg(true);
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e1) {
			}
		}
		return true;

	}

	/**
	 * トピック受信設定
	 * @return
	 * @since
	 */
	private boolean initialTopic() {


		//現在のTopic受信を終了
		terminateSumscriber();

		if(isErrFlg()){
			//エラー中の再接続ならば
			//再接続できたら一旦ログの読み込みスレッドを停止させる
			m_logManager.stopTransfer();
		}
		
		//新しいファシリティIDでトピック受信開始
		try {
	        
	        //ファシリティIDでメッセージセレクターを作成
	        StringBuffer msgSelector = new StringBuffer();
	        for (Iterator iter = m_facilityIdList.iterator(); iter.hasNext();) {
				String facilityId = (String) iter.next();
				msgSelector.append("FacilityId='");
				msgSelector.append(facilityId);
				msgSelector.append("'");
				if( iter.hasNext() ){
					msgSelector.append(" OR ");
				}
			}
	        
	        if(msgSelector.length() != 0){
	        
		        m_subscriber = m_session.createSubscriber(m_topic, msgSelector.toString(), false);
		        
		        //コールバックを登録する
		        m_subscriber.setMessageListener(this);

		        
	        }else{
	        	log.debug("ファシリティ情報がないので、何もしない");
	        }
			

		} catch (Exception e) {
			log.error("TopicInit", e);
			setErrFlg(true);
			return false;
		} finally {

		}
		return true;

	}
	/**
	 * EJB再接続タイマータスク
	 * 通信エラーとなった場合に定周期で呼ばれ再接続を行う 
	 */
	protected class ReSetupTask extends TimerTask {

		/**
		 * コネクションクローズ
		 */
		public void run() {
			if (reInitial()) {
				//このタスクをタイマーから解除
				cancel();
			}
		}

	}
	
	/**
	 * ファシリティID再取得タイマータスク
	 * ファシリティIDが取得できない場合、取得できるまで再取得を行う 
	 */
	protected class RreacquisitionTask extends TimerTask {

		/**
		 * 再取得
		 */
		public void run() {
			if (reInitial()) {
				//このタスクをタイマーから解除
				cancel();
			}
		}

	}

}
