/*
 
Copyright (C) 2007 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.maintenance.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.SchedulerException;

import sun.util.calendar.CalendarUtils;

import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerUtil;
import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.maintenance.bean.MaintenanceKeepMstConstant;
import com.clustercontrol.maintenance.bean.ScheduleInfo;
import com.clustercontrol.maintenance.ejb.entity.MaintenanceInfoData;
import com.clustercontrol.maintenance.ejb.entity.MaintenanceKeepMstLocal;
import com.clustercontrol.maintenance.factory.AddMaintenance;
import com.clustercontrol.maintenance.factory.DeleteMaintenance;
import com.clustercontrol.maintenance.factory.ModifyMaintenance;
import com.clustercontrol.maintenance.factory.ModifySchedule;
import com.clustercontrol.maintenance.factory.OperationMaintenance;
import com.clustercontrol.maintenance.factory.SelectMaintenanceInfo;
import com.clustercontrol.maintenance.factory.SelectMaintenanceKeepMst;
import com.clustercontrol.maintenance.factory.SelectMaintenanceTypeMst;
import com.clustercontrol.maintenance.factory.SelectSchedule;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 
 * メンテナンス機能の管理を行う Session Bean です。<BR>
 * 
 * @ejb.bean name="MaintenanceController"
 *           display-name="MaintenanceController"
 *           description="MaintenanceController"
 *           jndi-name="MaintenanceController"
 *           type="Stateless"
 *           view-type="remote"
 *           
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 *           
 */
public abstract class MaintenanceControllerBean implements SessionBean {

   protected static Log m_log = LogFactory.getLog( MaintenanceControllerBean.class );
   
	/** コンテキスト情報 */
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public MaintenanceControllerBean() {
		super();
	}
	
	/**
	 * メンテナンス情報を追加します。
	 * 
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws SystemException 
	 * @throws IllegalStateException 
	 * @throws NotSupportedException 
	 * @throws RollbackException 
	 * @throws HeuristicMixedException 
	 * @throws HeuristicRollbackException 
	 * @throws SchedulerException 
	 * @throws ParseException 
	 * @throws RemoteException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceWrite"
     *     method-intf="Remote"
	 * 
	 */
	public void addMaintenance(MaintenanceInfoData data, ScheduleInfo schedule) throws CreateException, NamingException, IllegalStateException, SystemException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SchedulerException, ParseException, RemoteException {
		m_log.debug("addMaintenance");
		
		Principal pri = m_context.getCallerPrincipal();
		
		TransactionManager tm = null;
		Transaction tx = null;
		Transaction oldTx = null;
		boolean result = false;
		
		try {
			
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			
			try {
				// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
				if(tm.getTransaction() != null){
					oldTx = tm.suspend();
				}
				
					// トランザクション開始
					if(tm.getTransaction() == null){
						tm.begin();	
						tx = tm.getTransaction();
					}
					
					// メンテナンス情報を登録
					AddMaintenance add = new AddMaintenance();
					add.addMaintenance(data, pri.getName());
					// Quartzへ登録
					addSchedule(schedule);
					
					result = true;
					
					
					if(result){
						// コミット
						tx.commit();
					}
			} catch (SecurityException e) {
				m_log.debug("addMaintenance() : " + e.getMessage());
				throw e;
			} catch (SystemException e) {
				m_log.debug("addMaintenance() : " + e.getMessage());
				throw e;
			} catch (NotSupportedException e) {
				m_log.debug("addMaintenance() : " + e.getMessage());
				throw e;
			} catch (RollbackException e) {
				m_log.debug("addMaintenance() : " + e.getMessage());
				throw e;
			} catch (HeuristicMixedException e) {
				m_log.debug("addMaintenance() : " + e.getMessage());
				throw e;
			} catch (HeuristicRollbackException e) {
				m_log.debug("addMaintenance() : " + e.getMessage());
				throw e;
			} finally {
				// トランザクション関連の例外が発生した場合は、ロールバック
				if(tx != null){
					if(!result){
						tx.rollback();
					}
					tx = null;
				}
			}
				
		} catch (CreateException e) {
			m_log.debug("addMaintenance() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.debug("addMaintenance() : " + e.getMessage());
			throw e;
		} catch (SchedulerException e) {
			m_log.debug("addMaintenance() : " + e.getMessage());
			throw e;
		} catch (ParseException e) {
			m_log.debug("addMaintenance() : " + e.getMessage());
			throw e;
		} catch (RemoteException e) {
			m_log.debug("addMaintenance() : " + e.getMessage());
			throw e;
		} finally {
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
				
				} catch(InvalidTransactionException e){
					throw e;
				} catch(IllegalStateException e){
					throw e;
				} catch(SystemException e){
					throw e;
				}
			}
		}
		
	}
	
	/**
	 * メンテナンス情報を変更します。
	 * 
	 * @throws FinderException 
	 * @throws NamingException 
	 * @throws SystemException 
	 * @throws NotSupportedException 
	 * @throws RollbackException 
	 * @throws HeuristicMixedException 
	 * @throws HeuristicRollbackException 
	 * @throws SchedulerException 
	 * @throws ParseException 
	 * @throws RemoteException 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceWrite"
     *     method-intf="Remote"
	 * 
	 */
	public void modifyMaintenance(MaintenanceInfoData data, ScheduleInfo schedule) throws FinderException, NamingException, SystemException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SchedulerException, ParseException, RemoteException {
		m_log.debug("modifyMaintenance");
		
		Principal pri = m_context.getCallerPrincipal();
		
		TransactionManager tm = null;
		Transaction tx = null;
		Transaction oldTx = null;
		boolean result = false;
		
		try {
			
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			
			try {
				// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
				if(tm.getTransaction() != null){
					oldTx = tm.suspend();
				}
				
					// トランザクション開始
					if(tm.getTransaction() == null){
						tm.begin();	
						tx = tm.getTransaction();
					}
					
					// メンテナンス情報を登録
					ModifyMaintenance modify = new ModifyMaintenance();
					modify.modifyMaintenance(data, pri.getName());
					// Quartzへ登録
					modifySchedule(schedule);
					
					result = true;
					
					
					if(result){
						// コミット
						tx.commit();
					}
			} catch (SecurityException e) {
				m_log.debug("modifyMaintenance() : " + e.getMessage());
				throw e;
			} catch (SystemException e) {
				m_log.debug("modifyMaintenance() : " + e.getMessage());
				throw e;
			} catch (NotSupportedException e) {
				m_log.debug("modifyMaintenance() : " + e.getMessage());
				throw e;
			} catch (RollbackException e) {
				m_log.debug("modifyMaintenance() : " + e.getMessage());
				throw e;
			} catch (HeuristicMixedException e) {
				m_log.debug("modifyMaintenance() : " + e.getMessage());
				throw e;
			} catch (HeuristicRollbackException e) {
				m_log.debug("modifyMaintenance() : " + e.getMessage());
				throw e;
			} finally {
				// トランザクション関連の例外が発生した場合は、ロールバック
				if(tx != null){
					if(!result){
						tx.rollback();
					}
					tx = null;
				}
			}
				
		} catch (NamingException e) {
			m_log.debug("modifyMaintenance() : " + e.getMessage());
			throw e;
		} catch (SchedulerException e) {
			m_log.debug("modifyMaintenance() : " + e.getMessage());
			throw e;
		} catch (ParseException e) {
			m_log.debug("modifyMaintenance() : " + e.getMessage());
			throw e;
		} catch (RemoteException e) {
			m_log.debug("modifyMaintenance() : " + e.getMessage());
			throw e;
		} finally {
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
				
				} catch(InvalidTransactionException e){
					throw e;
				} catch(IllegalStateException e){
					throw e;
				} catch(SystemException e){
					throw e;
				}
			}
		}
		
	}
	
	/**
	 * メンテナンス情報を削除します。
	 * 
	 * @throws RemoveException 
	 * @throws NamingException 
	 * @throws SystemException 
	 * @throws NotSupportedException 
	 * @throws RollbackException 
	 * @throws HeuristicMixedException 
	 * @throws HeuristicRollbackException 
	 * @throws SchedulerException 
	 * @throws RemoteException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceWrite"
     *     method-intf="Remote"
	 * 
	 */
	public void deleteMaintenance(String maintenanceId) throws RemoveException, NamingException, SystemException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SchedulerException, RemoteException {
		m_log.debug("deleteMaintenance");
		
		TransactionManager tm = null;
		Transaction tx = null;
		Transaction oldTx = null;
		boolean result = false;
		
		try {
			
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			
			try {
				// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
				if(tm.getTransaction() != null){
					oldTx = tm.suspend();
				}
				
					// トランザクション開始
					if(tm.getTransaction() == null){
						tm.begin();	
						tx = tm.getTransaction();
					}
					
					// メンテナンス情報を削除
					DeleteMaintenance delete = new DeleteMaintenance();
					delete.deleteMaintenance(maintenanceId);
					
					deleteSchedule(maintenanceId);

					result = true;
					
					
					if(result){
						// コミット
						tx.commit();
					}
			} catch (SecurityException e) {
				m_log.debug("deleteMaintenance() : " + e.getMessage());
				throw e;
			} catch (SystemException e) {
				m_log.debug("deleteMaintenance() : " + e.getMessage());
				throw e;
			} catch (NotSupportedException e) {
				m_log.debug("deleteMaintenance() : " + e.getMessage());
				throw e;
			} catch (RollbackException e) {
				m_log.debug("deleteMaintenance() : " + e.getMessage());
				throw e;
			} catch (HeuristicMixedException e) {
				m_log.debug("deleteMaintenance() : " + e.getMessage());
				throw e;
			} catch (HeuristicRollbackException e) {
				m_log.debug("deleteMaintenance() : " + e.getMessage());
				throw e;
			} finally {
				// トランザクション関連の例外が発生した場合は、ロールバック
				if(tx != null){
					if(!result){
						tx.rollback();
					}
					tx = null;
				}
			}
				
		} catch (NamingException e) {
			m_log.debug("deleteMaintenance() : " + e.getMessage());
			throw e;
		} catch (SchedulerException e) {
			m_log.debug("deleteMaintenance() : " + e.getMessage());
			throw e;
		} catch (RemoteException e) {
			m_log.debug("deleteMaintenance() : " + e.getMessage());
			throw e;
		} finally {
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
				
				} catch(InvalidTransactionException e){
					throw e;
				} catch(IllegalStateException e){
					throw e;
				} catch(SystemException e){
					throw e;
				}
			}
		}
		
		
	}
	
	
	
	/**
	 * 
	 * 確認済みのイベントログを削除します。
	 * 
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
     *     
	 */
	public void deleteEventLog(String keep_id) throws FinderException, NamingException, CreateException {
		m_log.debug("deleteEventLog()");
		
		Connection conn = null;
		Statement stmt = null;
		
		Timestamp timestamp = getTimestamp(keep_id);
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			StringBuffer sql = new StringBuffer();
			sql.append("DELETE FROM cc_event_multi_info");
			sql.append(" WHERE (monitor_id, plugin_id, output_date, facility_id)");
			sql.append(" IN (SELECT monitor_id, plugin_id, output_date, facility_id FROM cc_event_log");
			sql.append(" WHERE output_date < '" + timestamp.toString() + "'");
			sql.append(" AND confirm_flg='1'); ");
			sql.append("DELETE FROM cc_event_log WHERE output_date < '" + timestamp.toString() + "' AND confirm_flg='1';");
			
			//SQL文のセット
			stmt = conn.createStatement();
			
			m_log.debug("execute : " + sql.toString());
			stmt.executeUpdate(sql.toString());
				
		} catch (SQLException e) {
			m_log.error("removeAll() error : SQLException");
			e.printStackTrace();

		} finally {
			try {
				stmt.close();
				conn.close();
			} catch (SQLException e) {
				
			}
		}
	}

	
	/**
	 * 
	 * 完了したジョブ実行履歴を削除します。
	 * 
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
     *     
	 */
	public void deleteJobHistory(String keep_id) throws FinderException, NamingException, CreateException {
		m_log.debug("deleteJobHistory()");
		
		Connection conn = null;
		Statement stmt = null;
		
		Timestamp timestamp = getTimestamp(keep_id);
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			StringBuffer sql = new StringBuffer();
			
			//正常終了したsession_idを取得
			sql.append("SELECT session_id  INTO TEMPORARY cc_job_completed_sessions");
			sql.append(" FROM cc_job_session_job");
			sql.append(" WHERE end_date < '" + timestamp.toString() + "' AND status='300' AND (job_id, session_id ) IN (select job_id,session_id");
			sql.append(" FROM cc_job_session);");
			
			//TOPのジョブが変更済みとなっているsession_idを取得
			sql.append("SELECT a.session_id  INTO TEMPORARY cc_job_changed_sessions");
			sql.append(" FROM cc_job_session AS a join cc_job_session_job AS b on a.session_id=b.session_id");
			sql.append(" AND a.job_id=b.job_id");
			sql.append(" AND b.end_date < '" + timestamp.toString() + "'");
			sql.append(" AND b.status='301';");

			sql.append("SELECT a.session_id INTO TEMPORARY cc_job_delete_target_sessions FROM (SELECT session_id FROM cc_job_completed_sessions) AS a;");
			sql.append("INSERT INTO cc_job_delete_target_sessions (SELECT session_id FROM cc_job_changed_sessions);");
			
			String st =  " WHERE (session_id) IN (select session_id from cc_job_delete_target_sessions);";
			
			//削除用
			sql.append(" DELETE FROM cc_job_param_info" + st);
			sql.append(" DELETE FROM cc_job_end_info" + st);
			sql.append(" DELETE FROM cc_job_notice_info" + st);
			sql.append(" DELETE FROM cc_job_command_info" + st);
			sql.append(" DELETE FROM cc_job_start_info" + st);
			sql.append(" DELETE FROM cc_job_start_job_info" + st);
			sql.append(" DELETE FROM cc_job_start_time_info" + st);
			sql.append(" DELETE FROM cc_job_file_info" + st);
			sql.append(" DELETE FROM cc_job_info" + st);
			sql.append(" DELETE FROM cc_job_relation_info" + st);
			sql.append(" DELETE FROM cc_job_session_node" + st);
			sql.append(" DELETE FROM cc_job_session_job" + st);
			sql.append(" DELETE FROM cc_job_session" + st);
			
			//SQL文のセット
			stmt = conn.createStatement();
			
			m_log.debug("execute : " + sql.toString());
			stmt.executeUpdate(sql.toString());
			
		} catch (SQLException e) {
			m_log.error("removeAll() error : SQLException");
			e.printStackTrace();

		} finally {
			try {
				stmt.close();
				conn.close();
			} catch (SQLException e) {
				
			}
		}
	}
	
	
	/**
	 * 
	 * 完了した一括制御実行履歴を削除します。
	 * 
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
     *     
	 */
	public void deleteCollectiveRunHistory(String keep_id) throws FinderException, NamingException, CreateException {
		m_log.debug("deleteCollectiveRunHistory()");
		
		Connection conn = null;
		Statement stmt = null;

		Timestamp timestamp = getTimestamp(keep_id);
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			StringBuffer sql = new StringBuffer();
			
			sql.append("SELECT session_id INTO TEMPORARY cc_crun_completed_sessions");
			sql.append(" FROM cc_crun_session WHERE end_date < '" + timestamp.toString() + "' AND status='300';");
			
			String st =  " WHERE (session_id) IN (select session_id from cc_crun_completed_sessions); ";
			
			sql.append("DELETE FROM cc_crun_session_detail " + st);
			sql.append("DELETE FROM cc_crun_session_param " + st);
			sql.append("DELETE FROM cc_crun_session " + st);

			
			//SQL文のセット
			stmt = conn.createStatement();
			
			m_log.debug("execute : " + sql.toString());
			stmt.executeUpdate(sql.toString());
			
		} catch (SQLException e) {
			m_log.error("removeAll() error : SQLException");
			e.printStackTrace();

		} finally {
			try {
				stmt.close();
				conn.close();
			} catch (SQLException e) {
				
			}
		}
	}
	
	/**
	 * 
	 * スケジュール情報を登録します。
	 * 
	 * @throws RemoteException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceWrite"
     *     method-intf="Remote"
	 * 
	 */
	public void addSchedule(ScheduleInfo info)  throws NamingException, SchedulerException, ParseException, RemoteException {
	    m_log.debug("addSchedule() : scheduleId=" + info.getId());
	    
	    Principal pri = m_context.getCallerPrincipal();
	    
	    ModifySchedule modify = new ModifySchedule();
	    try {
            modify.addSchedule(info, pri.getName());
        } catch (ParseException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "004", args);
            m_log.debug("addSchedule() : " + e.getMessage());
            throw e;
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "004", args);
            m_log.debug("addSchedule() : " + e.getMessage());
            throw e;
        } catch (SchedulerException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "004", args);
            m_log.debug("addSchedule() : " + e.getMessage());
            throw e;
        }
	}
	
	/**
	 * 
	 * スケジュール情報を修正します。
	 * @throws RemoteException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceWrite"
     *     method-intf="Remote"
	 * 
	 */
	public void modifySchedule(ScheduleInfo info) throws NamingException, SchedulerException, ParseException, RemoteException {
	    m_log.debug("modifySchedule() : scheduleId=" + info.getId());
	    
	    Principal pri = m_context.getCallerPrincipal();
	    
	    ModifySchedule modify = new ModifySchedule();
	    try {
            modify.modifySchedule(info, pri.getName());
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
            m_log.debug("modifySchedule() : " + e.getMessage());
            throw e;
        } catch (SchedulerException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
            m_log.debug("modifySchedule() : " + e.getMessage());
            throw e;
        } catch (ParseException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
            m_log.debug("modifySchedule() : " + e.getMessage());
            throw e;
        }
		
	}
	
	/**
	 * 
	 * スケジュール情報を削除します。
	 * @throws RemoteException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceWrite"
     *     method-intf="Remote"
	 * 
	 */
	public void deleteSchedule(String scheduleId) throws NamingException, SchedulerException, RemoteException {
		m_log.debug("deleteSchedule() : scheduleId=" + scheduleId);
	    
	    ModifySchedule modify = new ModifySchedule();
	    try {
            modify.deleteSchedule(scheduleId);
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {scheduleId};
            apllog.put("SYS", "005", args);
            m_log.debug("deleteSchedule() : " + e.getMessage());
            throw e;
        } catch (SchedulerException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {scheduleId};
            apllog.put("SYS", "005", args);
            m_log.debug("deleteSchedule() : " + e.getMessage());
            throw e;
        }
	}
	
	/**
	 *
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *    role-name="MaintenanceRead"
     *    method-intf="Remote"
	 * 
	 * @return
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 */
	public MaintenanceInfoData getMaintenanceInfo(String maintenanceId) throws FinderException, NamingException, CreateException {
		m_log.debug("getMaintenanceInfoData()");
		
		SelectMaintenanceInfo select = new SelectMaintenanceInfo();
		MaintenanceInfoData info = select.getMaintenanceInfo(maintenanceId);
		
		return info;
		
	}
	
	/**
	 *
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceRead"
     *     method-intf="Remote"
	 * 
	 * @return
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 */
	public ArrayList getMaintenanceList() throws FinderException, NamingException, CreateException {
		m_log.debug("getMaintenanceList()");
		
		SelectMaintenanceInfo select = new SelectMaintenanceInfo();
		ArrayList list = select.getMaintenanceList();
		
		return list;
		
	}
	
	/**
	 *
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="MaintenanceRead"
	 *     method-intf="Remote"
	 * 
	 * @return
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 */
	public ArrayList getMaintenanceTypeList() throws FinderException, NamingException, CreateException {
		m_log.debug("getMaintenanceTypeList()");
		
		SelectMaintenanceTypeMst select = new SelectMaintenanceTypeMst();
		ArrayList list = select.getMaintenanceTypeMst();
		
		return list;
		
	}
	
	/**
	 *
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="MaintenanceRead"
	 *     method-intf="Remote"
	 * 
	 * @return
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 */
	public ArrayList getMaintenanceTypeNameIdList() throws FinderException, NamingException, CreateException {
		m_log.debug("getMaintenanceTypeNameIdList()");
		
		SelectMaintenanceTypeMst select = new SelectMaintenanceTypeMst();
		ArrayList list = select.getMaintenanceTypeNameIdList();
		
		return list;
		
	}
	
	/**
	 *
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="MaintenanceRead"
	 *     method-intf="Remote"
	 * 
	 * @return
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 */
	public ArrayList getMaintenanceKeepNameIdList() throws FinderException, NamingException, CreateException {
		m_log.debug("getMaintenanceKeepList()");
		
		SelectMaintenanceKeepMst select = new SelectMaintenanceKeepMst();
		ArrayList list = select.getMaintenanceKeepNameIdList();
		
		return list;
		
	}
	
	
	
	/**
	 * メンテナンス一覧のテーブル定義情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceRead"
     *     method-intf="Remote"
	 * 
	 * @param local ロケール
	 * @return TableColumnInfoの配列
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getMaintenanceListTableDefine(Locale local) {
		
		/** テーブル情報定義配列 */
		SelectMaintenanceInfo maintenance = new SelectMaintenanceInfo();
		ArrayList tableDefine = maintenance.getMaintenanceListTableDefine(local);

		return tableDefine;
	}
	
	/**
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="MaintenanceRead"
     *     method-intf="Remote"
	 * 
	 * @return
	 * @throws NamingException
	 * @throws SchedulerException
	 */
	public ArrayList getSchedule(String maintenanceId) throws NamingException, SchedulerException {
	    m_log.debug("getSchedule()");
	    
	    SelectSchedule select = new SelectSchedule();
	    //FIXME FinderException以外はそのままthrow
//	    Schedule schedule = select.getSchedule(maintenanceId);
	    ArrayList info = select.getSchedule(maintenanceId);

	    return info;
	}
	
	/**
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="MaintenanceRead"
     *     method-intf="Remote"
	 * 
	 * @return
	 * @throws NamingException
	 * @throws SchedulerException
	 */
	public ArrayList getScheduleList() throws NamingException, SchedulerException {
	    m_log.debug("getScheduleList()");
	    
	    SelectSchedule select = new SelectSchedule();
	    //FIXME FinderException以外はそのままthrow
	    ArrayList list = select.getScheduleList();

	    return list;
	}
	
	
	/**
	 * 
	 * メンテナンス機能をスケジュール実行します。<BR>
	 * Quartzからスケジュール実行時に呼び出されます。
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 */
	public void scheduleRunMaintenance(String maintenanceId, String calendarId) {
	    m_log.debug("scheduleRunMaintenance() : maintenanceId=" + maintenanceId + ", calendarId=" + calendarId);
	    
    	//カレンダをチェック
	    try {
			boolean check = false;
			if(calendarId != null && calendarId.length() > 0){
				CalendarControllerLocal calendar = 
					CalendarControllerUtil.getLocalHome().create();
					
				//カレンダによる実行可/不可のチェック
				if(calendar.isRun(calendarId, new Date()).booleanValue()){
					check = true;
				}
			}
			else{
				check = true;
			}
			
			if(!check)
				return;
			
			//メンテナンス実行
			runMaintenance(maintenanceId);
		} catch (CreateException e) {
//			throw e;
		} catch (NamingException e) {
//			throw e;
		} catch (FinderException e) {
			m_log.debug("scheduleRunJob() -> " + e.getMessage());
//			throw e;
		}
		
		
	}
	
	/**
	 * 
	 * メンテナンス機能を実行するメソッドです。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param maintenanceId
	 */
	public void runMaintenance(String maintenanceId) {
		m_log.debug("runMaintenance() : maintenanceId=" + maintenanceId);
		
		OperationMaintenance operation = new OperationMaintenance();
		operation.runMaintenance(maintenanceId);
	}
	
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean がインスタンスプールに格納される際に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	private Timestamp getTimestamp(String keep_id) throws FinderException, NamingException, CreateException {
		
		int order_no = 0;
		
		//削除対象外期間が設定されていた場合
		if (keep_id != null && !keep_id.equals("")) {
			SelectMaintenanceKeepMst select = new SelectMaintenanceKeepMst();
			MaintenanceKeepMstLocal mst = select.getMaintenanceKeepMst(keep_id);
			order_no = mst.getOrder_no();
		}

		
		Calendar calendar = GregorianCalendar.getInstance();
		
		//リスト表示順は空白が0番目となるため
		switch (order_no) {
		
		case MaintenanceKeepMstConstant.ONE_WEEK_INDEX:
			calendar.add(Calendar.WEEK_OF_MONTH, -1);
			break;
			
		case MaintenanceKeepMstConstant.TWO_WEEKS_INDEX:
			calendar.add(Calendar.WEEK_OF_MONTH, -2);
			break;
			
		case MaintenanceKeepMstConstant.ONE_MONTH_INDEX:
			calendar.add(Calendar.MONTH, -1);
			break;
			
		case MaintenanceKeepMstConstant.THREE_MONTHS_INDEX:
			calendar.add(Calendar.MONTH, -3);
			break;
			
		case MaintenanceKeepMstConstant.SIX_MONTHS_INDEX:
			calendar.add(Calendar.MONTH, -6);
			break;
			
		case MaintenanceKeepMstConstant.ONE_YEAR_INDEX:
			calendar.add(Calendar.YEAR, -1);
			break;
		}
		
		return new Timestamp(calendar.getTimeInMillis());
		
	}
	
	public void ejbRemove() throws EJBException, RemoteException {}

	public void ejbActivate() throws EJBException, RemoteException {}

	public void ejbPassivate() throws EJBException, RemoteException {}

}
