/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.factory;

import java.io.IOException;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.jmx.adaptor.rmi.RMIAdaptor;

import com.clustercontrol.bean.ConfirmConstant;
import com.clustercontrol.bean.FacilityTargetConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.monitor.bean.EventTabelDefine;
import com.clustercontrol.monitor.ejb.entity.EventLogLocal;
import com.clustercontrol.monitor.ejb.entity.EventLogPK;
import com.clustercontrol.monitor.ejb.entity.EventLogUtil;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.util.apllog.AplLogger;


/**
 * イベント情報の確認を更新するクラスです。
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class ModifyEventConfirm {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( ModifyEventConfirm.class );
	
	/** 
	 * RMIAdaptorのルックアップ名。<BR>
	 * イベント情報のキャッシュをフラッシュするために使用します。
	 */
	public static final String LOOKUP_NAME = "jmx/invoker/RMIAdaptor";
	/**
	 * イベント情報のオブジェクト名。<BR>
	 * キャッシュをフラッシュするオブジェクト名を指定します。
	 */
	public static final String OBJECT_NAME = "jboss.j2ee:jndiName=EventLog,service=EJB";
	/**
	 * 操作名。<BR>
	 * キャッシュをフラッシュする操作名を指定します。
	 */
	public static final String OPERATION_NAME = "flushCache";
	
	/**
	 * 引数で指定されたイベント情報一覧の確認を更新します。
	 * <p>指定されたリストには、各イベント情報のリスト（{@link ArrayList}）が格納されています。
	 * また、１イベント情報の各値は、テーブルのカラム順（{@link com.clustercontrol.monitor.bean.EventTabelDefine}）にリストに格納されています。
	 * <p>
	 * <dl>
	 *  <dt>イベント情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ イベント情報1 {カラム1の値, カラム2の値, … }, イベント情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 * </dl>
	 * <p>取得したイベント情報の確認フラグを更新します。確認フラグが済の場合は、確認済み日時も更新します。
	 * 
	 * @param list 更新対象のイベント情報一覧（Objectの2次元配列）
	 * @param confirmType 確認フラグ（未／済）（更新値）
	 * @throws NamingException
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.monitor.bean.EventTabelDefine
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see #modifyConfirm(String, String, String, Date, Date, int)
	 */
	public void modifyConfirm(List list,
			int confirmType
			) throws NamingException, FinderException  {
		
		if (list != null && list.size()>0) {
			
			// 確認済み日時
			Date confirmDate = null;
			if(confirmType == ConfirmConstant.TYPE_CONFIRMED){
				confirmDate = new Date();
			}
			
			ArrayList info = null;
			for(int index=0; index<list.size(); index++){
				
				info = (ArrayList)list.get(index);
				if (info != null && info.size()>0) {
					this.modifyConfirm((String) info.get(EventTabelDefine.MONITOR_ID),
							(String) info.get(EventTabelDefine.PLUGIN_ID),
							(String) info.get(EventTabelDefine.FACILITY_ID),
							(Date) info.get(EventTabelDefine.RECEIVE_TIME),
							confirmDate,
							confirmType
							);
				}
			}
		}
	}
	
	/**
	 * 引数で指定されたイベント情報の確認を更新します。<BR>
	 * 取得したイベント情報の確認フラグを更新します。確認フラグが済の場合は、確認済み日時も更新します。
	 * 
	 * @param monitorId 更新対象の監視項目ID
	 * @param pluginId 更新対象のプラグインID
	 * @param facilityId 更新対象のファシリティID
	 * @param outputDate 更新対象の受信日時
	 * @param confirmDate 更新対象の確認済み日時
	 * @param confirmType 確認フラグ（未／済）（更新値）
	 * @throws NamingException
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindByPrimaryKey(EventLogPK)
	 */
	public void modifyConfirm(String monitorId,
			String pluginId,
			String facilityId,
			Date outputDate,
			Date confirmDate,
			int confirmType
			) throws NamingException, FinderException  {
		 
		try
		{
			// イベントログ情報を取得
			EventLogPK pk = new EventLogPK(monitorId,
					pluginId,
					facilityId,
					new Timestamp(outputDate.getTime()));
			EventLogLocal event = EventLogUtil.getLocalHome().findByPrimaryKey(pk);
			
			// 確認有無を変更
			event.setConfirmFlg(new Integer(confirmType));
			
			if(confirmType == ConfirmConstant.TYPE_CONFIRMED){
				
				if(confirmDate == null){
					confirmDate = new Date();
				}
				event.setConfirmDate(new Timestamp(confirmDate.getTime()));	
			}
			
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("MON", "mon");
            String[] args = {facilityId, monitorId, pluginId, ConfirmConstant.typeToString(confirmType)};
            apllog.put("SYS", "004", args);
			m_log.debug("modifyConfirm():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("MON", "mon");
            String[] args = {facilityId, monitorId, pluginId, ConfirmConstant.typeToString(confirmType)};
            apllog.put("SYS", "004", args);
			m_log.debug("modifyConfirm():" + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * 引数で指定された条件に一致するイベント情報の確認を一括更新します。
	 * <p>
	 * <ol>
	 * <li>引数で指定されたプロパティに格納された更新条件を、プロパティユーティリティ（{@link com.clustercontrol.util.PropertyUtil}）を使用して取得します。</li>
	 * <li>引数で指定されたファシリティ配下のファシリティと更新条件を基に、イベント情報を取得します。</li>
	 * <li>取得したイベント情報の確認フラグを更新します。確認フラグが済の場合は、確認済み日時も更新します。</li>
	 * <li>イベント情報Entity Beanのキャッシュをフラッシュします。</li>
	 * 
	 * @param confirmType 確認フラグ（未／済）（更新値）
	 * @param facilityId 更新対象の親ファシリティID
	 * @param property 更新条件
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws SQLException
	 * 
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.util.PropertyUtil#getPropertyValue(com.clustercontrol.bean.Property, java.lang.String)
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityIdList(String, int)
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbHomeBatchConfirm(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer, Integer)
	 */
	public void modifyBatchConfirm(int confirmType, String facilityId, Property property)
	throws CreateException, NamingException, FinderException, SQLException  {
		
		ArrayList values = null;
		
		Integer priority = null;
		Timestamp outputFromDate = null;
		Timestamp outputToDate = null;
		Timestamp generationFromDate = null;
		Timestamp generationToDate = null;
		String facilityType = null;
		String application = null;
		String message = null;
		
		try 
		{
			//重要度取得
			values = PropertyUtil.getPropertyValue(property, StatusFilterProperty.PRIORITY);
			if(!"".equals((String)values.get(0))){
				int priorityType = PriorityConstant.stringToType((String)values.get(0));
				if(priorityType != -1){
					priority = new Integer(priorityType);
				}
			}
			
			//更新日時（自）取得			
			values = PropertyUtil.getPropertyValue(property, EventFilterProperty.OUTPUT_FROM_DATE);
			if(values.get(0) instanceof Date){
				outputFromDate = new Timestamp(((Date)values.get(0)).getTime());
				outputFromDate.setNanos(0);
			}
			
			//更新日時（至）取得			
			values = PropertyUtil.getPropertyValue(property, EventFilterProperty.OUTPUT_TO_DATE);
			if(values.get(0) instanceof Date){
				outputToDate = new Timestamp(((Date)values.get(0)).getTime());
				outputToDate.setNanos(999999999);
			}
			
			//出力日時（自）取得
			values = PropertyUtil.getPropertyValue(property, EventFilterProperty.GENERATION_FROM_DATE);
			if(values.get(0) instanceof Date){
				generationFromDate = new Timestamp(((Date)values.get(0)).getTime());
				generationFromDate.setNanos(0);
			}
			
			//出力日時（至）取得
			values = PropertyUtil.getPropertyValue(property, EventFilterProperty.GENERATION_TO_DATE);
			if(values.get(0) instanceof Date){
				generationToDate = new Timestamp(((Date)values.get(0)).getTime());
				generationToDate.setNanos(999999999);
			}
			
			//対象ファシリティ種別取得
			values = PropertyUtil.getPropertyValue(property, StatusFilterProperty.FACILITY_TYPE);
			if(!"".equals((String)values.get(0))){
				facilityType = (String)values.get(0);
			}
			
			//アプリケーション取得
			values = PropertyUtil.getPropertyValue(property, StatusFilterProperty.APPLICATION);
			if(!"".equals((String)values.get(0))){
				application = (String)values.get(0);
			}
			
			//メッセージ取得
			values = PropertyUtil.getPropertyValue(property, StatusFilterProperty.MESSAGE);
			if(!"".equals((String)values.get(0))){
				message = (String)values.get(0);
			}
			
			// 対象ファシリティのファシリティIDを取得
			// ファシリティが指定されない（最上位）場合は、ファシリティIDを指定せずに検索を行う
			String[] facilityIds = null;
			if(facilityId != null && !"".equals(facilityId)){
				
				int level = RepositoryControllerBean.ALL;
				if(FacilityTargetConstant.STRING_BENEATH.equals(facilityType)){
					level = RepositoryControllerBean.ONE_LEVEL;
				}
	
				ArrayList facilityIdList = RepositoryControllerUtil.getLocalHome().create().getFacilityIdList(facilityId, level);
				
				// スコープの場合
				if(facilityIdList != null && facilityIdList.size() > 0){
					facilityIds = new String[facilityIdList.size()];
					facilityIdList.toArray(facilityIds);
				}
				// ノードの場合
				else{
					facilityIds = new String[1];
					facilityIds[0] = facilityId;
				}
			}
			
			// 引数と反対の確認状態を取得する
			Integer selectConfirmFlg = null;
			if(confirmType == ConfirmConstant.TYPE_CONFIRMED){
				selectConfirmFlg = new Integer(ConfirmConstant.TYPE_UNCONFIRMED);
			}
			else if(confirmType == ConfirmConstant.TYPE_UNCONFIRMED){
				selectConfirmFlg = new Integer(ConfirmConstant.TYPE_CONFIRMED);
			}

    	    // アップデートする設定フラグ
			Integer confirmFlg = new Integer(confirmType);
			
   			EventLogUtil.getLocalHome().batchConfirm(
   					facilityIds,
   					priority,
   					outputFromDate,
   					outputToDate,
   					generationFromDate,
   					generationToDate,
   					application,
   					message,
   					selectConfirmFlg,
   					confirmFlg
   			);
		} catch (CreateException e) {
			this.setAplLogForBatchConfirmFailure(facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmType);
			m_log.debug("modifyBatchConfirm():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			this.setAplLogForBatchConfirmFailure(facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmType);
			m_log.debug("modifyBatchConfirm():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			this.setAplLogForBatchConfirmFailure(facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmType);
			m_log.debug("modifyBatchConfirm():" + e.getMessage());
			throw e;
		}
		
		//Flush Entity Cache
		try{
	        InitialContext ic = new InitialContext();
	        
	        //RMIAdaptorを取得
	        RMIAdaptor server = (RMIAdaptor) ic.lookup(LOOKUP_NAME);

	        //ObjectNameを設定
	        ObjectName name = new ObjectName(OBJECT_NAME);
	        
	        //ObjectNameのOperationNameのメソッドを実行
	        server.invoke(name, OPERATION_NAME, null, null);

		} catch (NamingException e) {
			m_log.debug("modifyBatchConfirm() : " + e.getMessage());
		} catch (MalformedObjectNameException e) {
			m_log.debug("modifyBatchConfirm() : " + e.getMessage());
		} catch (NullPointerException e) {
			m_log.debug("modifyBatchConfirm() : " + e.getMessage());
		} catch (InstanceNotFoundException e) {
			m_log.debug("modifyBatchConfirm() : " + e.getMessage());
		} catch (MBeanException e) {
			m_log.debug("modifyBatchConfirm() : " + e.getMessage());
		} catch (ReflectionException e) {
			m_log.debug("modifyBatchConfirm() : " + e.getMessage());
		} catch (IOException e) {
			m_log.debug("modifyBatchConfirm() : " + e.getMessage());
		}
	}
	
	/**
	 * アプリケーションログに、一括更新に失敗した旨のログを出力します。
	 * 
	 * @param facilityId
	 * @param priority
	 * @param outputFromDate
	 * @param outputToDate
	 * @param generationFromDate
	 * @param generationToDate
	 * @param facilityType
	 * @param application
	 * @param message
	 * @param confirmType
	 */
	private void setAplLogForBatchConfirmFailure(String facilityId, 
										Integer priority, 
										Timestamp outputFromDate, 
										Timestamp outputToDate, 
										Timestamp generationFromDate, 
										Timestamp generationToDate,
										String facilityType,
										String application,
										String message,
										int confirmType) {
		
		int priorityInt = -1;
		String outputFromDateString = null;
		String outputToDateString = null;
		String generationFromDateString = null;
		String generationToDateString = null;
		
		if(priority != null)
			priorityInt = priority.intValue();
		if(outputFromDate != null)
			outputFromDateString = outputFromDate.toString();
		if(outputToDate != null)
			outputToDateString = outputToDate.toString();
		if(generationFromDate != null)
			generationFromDateString = generationFromDate.toString();
		if(generationToDate != null)
			generationToDateString = generationToDate.toString();
		
        AplLogger apllog = new AplLogger("MON", "mon");
        String[] args = {facilityId,PriorityConstant.typeToString(priorityInt),outputFromDateString,outputToDateString,generationFromDateString,generationToDateString,facilityType,application,message,ConfirmConstant.typeToString(confirmType)};
        apllog.put("SYS", "005", args);
	}
}

