/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.action;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.rmi.RemoteException;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import com.clustercontrol.performance.util.CollectorItemCodeFactory;
import com.clustercontrol.performance.util.Messages;
import com.clustercontrol.performance.util.OutputFormat;
import com.clustercontrol.performanceMGR.bean.CollectedDataInfo;
import com.clustercontrol.performanceMGR.bean.CollectedDataSet;
import com.clustercontrol.performanceMGR.bean.CollectorItemInfo;
import com.clustercontrol.performanceMGR.bean.CollectorProperty;
import com.clustercontrol.performanceMGR.ejb.bmp.RecordCollectorData;

/**
 * 実績データエクスポートアクションクラス
 * 
 * @version 1.0
 * @since 1.0
 *  
 */
public class RecordDataWriter implements Runnable {

    private CollectorProperty property;

    private String facilityID;

    private boolean headerFlag;

    private File file;

    private FileOutputStream fos;

    private OutputStreamWriter osw;

    private BufferedWriter bw;

    private Date lastDate = null;

    private int getSize = 100; // マネージャから一度に取得するデータ数（時間軸方向）

    private long collectStartTime;

    private long collectStopTime;

    private int progress;

    private boolean canceled;

    /**
     * コンストラクタ
     */
    public RecordDataWriter(String filename, CollectorProperty property,
            String facilityID, boolean headerFlag) {
        this.file = new File(filename);
        this.property = property;
        this.facilityID = facilityID;
        this.headerFlag = headerFlag;
    }

    /**
     * 新規にファイルを生成します。
     * 
     * @return 指定されたファイルが存在せず、ファイルの生成に成功した場合は true、示されたファイルがすでに存在する場合は false
     */
    public boolean createNewFile() throws IOException {
        return this.file.createNewFile();
    }

    /**
     * ファイルへのエクスポートを実行します。
     */
    public void run() {
        try {
            write(property, facilityID, headerFlag);
        } catch (IOException e) {
        }
    }

    /**
     * 現在までに性能値データの何％のエクスポートが完了したのかを取得します。
     * 
     * @return 進捗
     */
    public int getProgress() {
        return progress;
    }

    /**
     * 上書きモードでファイルを開きます。
     * 
     * @throws FileNotFoundException
     * @throws UnsupportedEncodingException
     */
    private void openFile() throws FileNotFoundException {
        try {
            fos = new FileOutputStream(this.file); // 上書きモードで開く
            osw = new OutputStreamWriter(fos);
            bw = new BufferedWriter(osw);
        } catch (FileNotFoundException e) {
            throw e;
        } finally {
        }
    }

    /**
     * ファイルを閉じます。
     * 
     * @throws IOException
     */
    private void closeFile() throws IOException {
        try {
            bw.close();
            osw.close();
            fos.close();
        } catch (IOException e) {
            throw e;
        } finally {
        }
    }

    /**
     * 指定の収集データのリストをファイルに出力します。
     */
    public void write(CollectorProperty property, String facilityID,
            boolean headerFlag) throws IOException {
        try {
            // ファイルを開く
            openFile();

            if (headerFlag) {
                // ヘッダ部を出力
                writeHeader(property.getCollectorData(), facilityID);
            }

            // 収集のプロパティを取得
            RecordCollectorData collectorData = property.getCollectorData();

            // 収集項目のリストを取得
            List itemList = property.getItemList();
            int[] itemIDs = new int[itemList.size()];

            for (int i = 0; i < itemIDs.length; i++) {
                itemIDs[i] = ((CollectorItemInfo) itemList.get(i))
                        .getCollectorItemID();
            }

            // 順列に並び代えます。
            Arrays.sort(itemIDs);

            String[] facilityIDs = new String[1];
            facilityIDs[0] = facilityID;

            collectStartTime = collectorData.getStartDate().getTime();
            collectStopTime = collectorData.getStopDate().getTime();
            int interval = collectorData.getInterval();
            int timeBlock = interval * 1000 * getSize; // 1度に取得する時間の間隔(ミリ秒単位)

            // 重複部分
            long overlap = interval * 3 * 1000; // ミリ秒

            long stratTime = collectStartTime;
            long endTime = 0;

            // ループを回す回数を計算(進捗計算のため)
            int loopMax = ((int) (collectStopTime - collectStartTime)
                    / timeBlock + 1);
            // ループの回数を保持する(進捗計算のため)
            int loopCount = 0;

            while (this.canceled == false && stratTime < collectStopTime) {
                endTime = stratTime + timeBlock; // 今回マネージャから取得するデータの最大の時刻

                // 指定期間の収集回数より1つ少ない要素しか取得できないため少し重複して取得
                Date startDate = new Date(stratTime - overlap); // 今回マネージャから取得するデータの最小日時
                Date endDate = new Date(endTime); // 今回マネージャから取得するデータの最大日時

                RecordController controller = RecordController.getInstance();
                // マネージャとの接続に失敗した場合はエラー
                if (controller == null) {
                    throw new RemoteException("Can't connect to manager.");
                }

                // 取得した性能データを一時的に保存するためのバッファ
                CollectedDataSet dataSetBuffer = new CollectedDataSet();

                // 項目を分割して数回に分けてマネージャから性能値を取得する
                for (int i = 0; i < itemIDs.length; i++) {
                    int[] itemIDBuffer = new int[1];
                    itemIDBuffer[0] = itemIDs[i];

                    CollectedDataSet dataSet = controller
                            .getRecordCollectedData(collectorData
                                    .getCollectorID(), facilityIDs,
                                    itemIDBuffer, startDate, endDate);

                    // 性能データの取得に失敗した場合はエラー
                    if (dataSet == null) {
                        throw new RemoteException(
                                "Get performance data failed.");
                    }

                    dataSetBuffer.addCollectedDataList(dataSet);

                    // 進捗を計算
                    progress = (int) (((loopCount + ((double) i / itemIDs.length)) / loopMax) * 100);
                }

                // 性能値を出力
                white(facilityID, property.getItemList(), itemIDs,
                        dataSetBuffer);

                stratTime = endTime;
                loopCount++;
            }
            // 処理が終了したので進捗を100%とする
            progress = 100;
        } catch (IOException e) {
            throw e;
        } finally {
            // ファイルを閉じる
            closeFile();
        }
    }

    /**
     * ヘッダ部を出力します。
     * 
     * @param profile
     */
    private void writeHeader(RecordCollectorData profile, String facilityID)
            throws IOException {
        try {
            // 収集設定を出力

            // 収集ID
            bw.write(Messages.getString("EXPORT_HEADER_COLLECTORID") + ", "
                    + profile.getCollectorID());
            bw.newLine(); // 改行

            // 収集ラベル
            bw.write(Messages.getString("EXPORT_HEADER_LABEL") + ", "
                    + profile.getLabel());
            bw.newLine(); // 改行

            // 収集対象スコープ
            bw.write(Messages.getString("EXPORT_HEADER_FACILITY") + ", "
                    + facilityID);
            bw.newLine(); // 改行

            // 収集開始時刻
            bw.write(Messages.getString("EXPORT_HEADER_STARTDATE") + ", "
                    + OutputFormat.dateToString(profile.getStartDate()));
            bw.newLine(); // 改行
            bw.newLine(); // 改行
        } catch (IOException e) {
            throw e;
        } finally {
        }
    }

    /**
     * 性能値をファイルに出力します。
     * 
     * @param facilityID
     * @param itemInfoList
     * @param targetItemIDs
     * @param dataSet
     * @throws IOException
     */
    private void white(String facilityID, List itemInfoList,
            int[] targetItemIDs, CollectedDataSet dataSet) throws IOException {
        try {
            // 初めての書き込みの場合は項目を出力
            if (lastDate == null) {
                // 収集項目名を出力
                String msg = Messages.getString("EXPORT_COLUMN_TIME");
                for (int j = 0; j < targetItemIDs.length; j++) {
                    // 収集項目リストから収集IDに一致するものを探しその収集項目名を取得
                    for (int i = 0; i < itemInfoList.size(); i++) {
                        CollectorItemInfo itemInfo = (CollectorItemInfo) itemInfoList
                                .get(i);
                        if (itemInfo.getCollectorItemID() == targetItemIDs[j]) {
                            String itemName = CollectorItemCodeFactory
                                    .getFullItemName(itemInfo
                                            .getCollectorItemCode(), itemInfo
                                            .getDeviceName());

                            msg = msg + ", " + itemName;
                            i = itemInfoList.size(); // IDは重複することはないためループを抜ける
                        }
                    }
                }
                bw.write(msg);
                bw.newLine(); // 改行
            }

            List[] collectedDataListArray = new List[dataSet
                    .getDataListNum(facilityID)];

            if (collectedDataListArray.length != targetItemIDs.length) {
                // エラー処理
                bw.write(Messages.getString("INVALID_DATA_IN_DB"));
                bw.newLine(); // 改行
            	
            	return;
            }

            for (int i = 0; i < targetItemIDs.length; i++) {
                // 収集項目IDの順番に収集済みデータリストを配列に置き換える
                collectedDataListArray[i] = dataSet.getCollectedDataList(
                        facilityID, targetItemIDs[i]);
            }

            // 全ての配列で時間軸方向の要素の数は同じであるため，
            // 適当なものから収集データリストの時間軸方向のサイズを取得
            int timeSize = collectedDataListArray[0].size();

            // 全ての配列で時間軸方向の要素の数は同じであるかチェックする。
            for (int i = 1; i < targetItemIDs.length; i++) {
                if(timeSize != collectedDataListArray[i].size()){
                    // エラー処理
                    bw.write(Messages.getString("INVALID_DATA_IN_DB"));
                    bw.newLine(); // 改行
                    
                	return;
                }
            }
            
            CollectedDataInfo data = null;
            for (int t = 0; t < timeSize; t++) {
                // 最初の項目を取得
                CollectedDataInfo firstData = (CollectedDataInfo) collectedDataListArray[0]
                        .get(t);

                // 重複する部分は出力しない
                if (lastDate == null
                        || firstData.getDate().getTime() > lastDate.getTime()) {

                    String msg = null;
                    for (int i = 0; i < targetItemIDs.length; i++) {
                        data = (CollectedDataInfo) collectedDataListArray[i]
                                .get(t);

                        if (i == 0) { // 最初の項目だけ時刻も出力
                            // 出力文字列を設定
                            msg = OutputFormat.dateToString(data.getDate())
                                    + ", "
                                    + OutputFormat.doubleToString(data
                                            .getValue());
                        } else {
                            // 出力文字列を設定
                            msg = msg
                                    + ", "
                                    + OutputFormat.doubleToString(data
                                            .getValue());
                        }
                    }
                    bw.write(msg);
                    bw.newLine(); // 改行
                }
            }

            // 出力された最後の日時を記録
            if (data != null) {
                lastDate = data.getDate();
            }
        } catch (IOException e) {
            throw e;
        } finally {
        }
    }

    /**
     * エクスポート中に処理を中止します。
     * 
     * @param b
     *            処理を中止したい場合にはtrueを設定します。
     */
    public void setCanceled(boolean b) {
        this.canceled = b;
    }
}