/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performanceMGR.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityInfo;
import com.clustercontrol.bean.FacilityPath;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;

/**
 * リポジトリからのデータ取得を提供するクラス
 * @version 1.0
 * @since 1.0
 */
public class Repository {
	/**
	 * リポジトリから現在のファシリティツリー情報を取得します。
	 * @param facilityID 取得したいツリーの最上位のファシリティID
	 * @return ファシリティツリーの最上位の要素
	 */
	public FacilityTreeItem getFacilityTree(String facilityID) {
		// ツリー情報を取得
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();
			FacilityTreeItem facilityTree = bean.getFacilityTree(Locale.getDefault());
			bean.remove();
			
			// 指定のファシリティID以下のツリーを取得する
			FacilityTreeItem subFacilityTree = selectFacilityTreeItem(facilityTree, facilityID);
			
			if(subFacilityTree == null){
				// エラー処理
				String message= "Facility not found : " + facilityID;
				return null;
			}
			
			//FacilityTreeの最上位インスタンスを作成
			FacilityInfo info = new FacilityInfo();
			info.setFacilityName(FacilityConstant.STRING_COMPOSITE);
			info.setType(FacilityConstant.TYPE_COMPOSITE);
			FacilityTreeItem top = new FacilityTreeItem(null, info);
			
			// 取得したファシリティツリーをコンポジットアイテムに繋ぐ
			subFacilityTree.setParent(top);
			top.addChildren(subFacilityTree);
			
			return top;
		} catch (CreateException e) {
			e.printStackTrace();
			throw new EJBException(e);
		} catch (NamingException e) {
			e.printStackTrace();
			throw new EJBException(e);
		} catch (FinderException e) {
			// リポジトリの内容が変更になっているなどの理由で取得できなかった場合
			// EJBException は throw しない
			// 収集時以外にも収集用EntityBeanの参照時(収集停止時など)に参照されるため
			// EJBException を throw すると処理が終了してしまい，収集停止ができなくなる
			e.printStackTrace();
		} catch (RemoveException e) {
			e.printStackTrace();
			throw new EJBException(e);
		}
		return null;
	}
	
	/**
	 * 指定のファシリティIDのパスを取得します。
	 * @param facilityID パスを取得したいファシリティのファシリティID
	 * @return ファシリティパス
	 */
	public String getFacilityPath(String facilityID) {
		// ツリー情報を取得
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();
			FacilityTreeItem facilityTree = bean.getFacilityTree(Locale.getDefault());
			bean.remove();

			FacilityTreeItem treeItem = selectFacilityTreeItem(facilityTree, facilityID);
			
			if(treeItem == null){
				// エラー処理
				String message= "Facility not found : " + facilityID;
				throw new EJBException(message);
			}
			
			FacilityPath fpath = new FacilityPath(">");
			
			return fpath.getPath(treeItem);
		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (RemoveException e) {
			throw new EJBException(e);
		}
	}

	/**
	 * リポジトリから指定のファシリティID以下の階層のノードのデバイス情報を取得します。
	 * 
	 * @param collectorID 収集ID
	 * @param facilityID 対象ファシリティID
	 * @return　デバイス情報のリスト
	 */
	public List getDeviceList(String facilityID) {
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();

			ArrayList deviceList = new ArrayList();
			
			if(facilityID == null){  // ファシリティIDがnullの場合
				// スコープにふくまれる全てのノードのIDのリストを取得
				List nodeList = bean.getNodeFacilityIdList();
				
				Iterator itr = nodeList.iterator();
				while(itr.hasNext()){
					String nodeID = (String)itr.next();

					// デバイス情報を取得しリストに追加
					deviceList.addAll(bean.getDeviceList(nodeID));
				}
			} else if(bean.isNode(facilityID)){	// 対象ファシリティがノードの場合
				// 指定のファシリティIDのデバイスを取得
				 deviceList.addAll(bean.getDeviceList(facilityID));
			} else {	// 対象ファシリティがスコープの場合
				// スコープにふくまれるノードのIDのリストを取得
				List nodeList = bean.getNodeFacilityIdList(facilityID, RepositoryControllerBean.ALL);
				
				Iterator itr = nodeList.iterator();
				while(itr.hasNext()){
					String nodeID = (String)itr.next();

					// デバイス情報を取得しリストに追加
					deviceList.addAll(bean.getDeviceList(nodeID));
				}
			}
			bean.remove();

			return deviceList;
		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (RemoveException e) {
			throw new EJBException(e);
		}
	}
	
	/**
	 * ファシリティツリーの中で指定のファシリティIDを持つファシリティを再帰的に探します
	 * 
	 * @param facilityTree 対象のファシリティツリー
	 * @param facilityID パスを取得したいファシリティのファシリティID
	 * @return ファシリティ情報
	 */
	private FacilityTreeItem selectFacilityTreeItem(FacilityTreeItem facilityTree, String facilityID){
		if(facilityTree.getData().getFacilityId().equals(facilityID)){
			return facilityTree;
		} else {
			for(int i=0; i<facilityTree.getChildren().length; i++){
				FacilityTreeItem target = facilityTree.getChildren()[i];
				FacilityTreeItem temp = selectFacilityTreeItem(target, facilityID);  // 再帰的
				if(temp != null){
					return temp;  
				}
			}
		}
		return null;
	}
	
	/**
	 * 指定ファシリティIDのノードのIPアドレスを取得します。
	 * 
	 * @param facilityID 対象のファシリティID
	 */
	public String getIPAddress(String facilityID) {
		HashMap nodeInfo = new HashMap();
		
		// IPアドレスを取得
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();
			ArrayList attributes = new ArrayList();
			attributes.add(FacilityAttributeConstant.IPNETWORKNUMBER);
			nodeInfo = bean.getNodeDetail(facilityID, attributes);
			bean.remove();
		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (FinderException e) {
			throw new EJBException(e);
		} catch (RemoveException e) {
			throw new EJBException(e);
		}

		if(nodeInfo != null){
			return (String)nodeInfo.get(FacilityAttributeConstant.IPNETWORKNUMBER);
		} else {
			// エラー処理
			String message= "Couldn't get an IP Address : " + facilityID;
			throw new EJBException(message);
		}
	}
	
	/**
	 * 指定のファシリティIDがノードであるか否かを判定します。
	 * 
	 * @param facilityID ファシリティID
	 * @return ノードの場合は true
	 */
	public boolean isNode(String facilityID){
		RepositoryControllerLocal bean = null;
		try {
			bean = RepositoryControllerUtil.getLocalHome().create();
			return bean.isNode(facilityID);
		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (FinderException e) {
			throw new EJBException(e);
		} finally {
			try {
				bean.remove();
			} catch (EJBException e) {
				throw new EJBException(e);
			} catch (RemoveException e) {
				throw new EJBException(e);
			}
		}
	}
}
