/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmptrap.factory;

import java.sql.Timestamp;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.jms.JMSException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapOidInfo;
import com.clustercontrol.snmptrap.bean.TopicConstant;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapInfoLocal;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapInfoPK;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapInfoUtil;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMasterLocal;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMasterPK;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMasterUtil;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapOidInfoLocal;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapOidInfoUtil;
import com.clustercontrol.snmptrap.message.UpdateSnmpTrapInfo;
import com.clustercontrol.snmptrap.util.SendTopic;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 監視情報更新クラス
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class ModifyMonitorInfo {
	
	protected static Log m_log = LogFactory.getLog( ModifyMonitorInfo.class );
	
	/** 監視情報ビーン */
	protected MonitorSnmpTrapInfo m_monitorInfo;
	
	/**
	 * 監視情報を更新
	 * 
	 * @param info 監視情報
	 * @param user ユーザ
	 * @return 更新に成功した場合、true
	 * @throws CreateException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws InvalidTransactionException
	 * @throws NamingException
	 * @throws FinderException 
	 * @throws RemoveException 
	 */
	public boolean modify(MonitorSnmpTrapInfo info, String user) throws CreateException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, InvalidTransactionException, NamingException, FinderException, RemoveException {
		
    	//アクセスロック
    	AccessLock.lock(AccessLock.MONITOR_SNMPTRAP);

		m_monitorInfo = info;
		
		TransactionManager tm = null;
		Transaction tx = null;
		Transaction oldTx = null;
		boolean result = false;
		
		try 
		{
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			
			// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
			if(tm.getTransaction() != null){
				oldTx = tm.suspend();
			}
			
			try{
				// トランザクション開始
				if(tm.getTransaction() == null){
					tm.begin();	
					tx = tm.getTransaction();
				}
				
				// 監視情報を更新
				result = modifyMonitorInfo(user);
				if(result){
					// コミット
					tx.commit();
				}
				
				try {
					// 更新通知
					UpdateSnmpTrapInfo updateInfo = new UpdateSnmpTrapInfo(
							UpdateSnmpTrapInfo.TYPE_UPDATE,
							info.getMonitorId());
					
					SendTopic sendTopic = new SendTopic(TopicConstant.TOPIC_NAME_UPDATE);
					sendTopic.put(updateInfo);
					
				} catch (NamingException e) {
					m_log.error("modify() : 送信エラー。" + e.getMessage() );
				} catch (JMSException e) {
					m_log.error("modify() : 送信エラー。" + e.getMessage() );
				}
			
			} catch(NotSupportedException e){
				outputLog(e, "modify()");
				throw e;
			} catch (RollbackException e) {
				outputLog(e, "modify()");
				throw e;
			} catch (HeuristicMixedException e) {
				outputLog(e, "modify()");
				throw e;
			} catch(HeuristicRollbackException e){
				outputLog(e, "modify()");
				throw e;
			} catch(SystemException e){
				outputLog(e, "modify()");
				throw e;
			}
			finally{
				// トランザクション関連の例外が発生した場合は、ロールバック
				if(tx != null){
					if(!result){
						tx.rollback();
					}
					tx = null;
				}
			}
		}
		finally{
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
				
				} catch(InvalidTransactionException e){
					outputLog(e, "modify()");
					throw e;
				} catch(IllegalStateException e){
					outputLog(e, "modify()");
					throw e;
				} catch(SystemException e){
					outputLog(e, "modify()");
					throw e;
				}
			}
		}
		return result;
	}

	/**
	 * 監視情報を更新
	 * 
	 * @param user ユーザ
	 * @return 更新に成功した場合、true
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FinderException 
	 * @throws RemoveException 
	 */
	public boolean modifyMonitorInfo(String user) throws CreateException, NamingException, FinderException, RemoveException {
		boolean result = false;
		
		try 
		{
			Timestamp now = new Timestamp(new Date().getTime());
			
			// 監視情報を取得し更新
			SnmpTrapInfoLocal info = 
				SnmpTrapInfoUtil.getLocalHome().findByPrimaryKey(
					new SnmpTrapInfoPK(m_monitorInfo.getMonitorId()));
			
			info.setDescription(m_monitorInfo.getDescription());
			info.setFacilityId(m_monitorInfo.getFacilityId());
			info.setCommunityName(m_monitorInfo.getCommunityName());
			info.setNotifyId(m_monitorInfo.getNotifyId());
			info.setApplication(m_monitorInfo.getApplication());
			info.setValidFlg(m_monitorInfo.getValid());
			info.setCalendarId(m_monitorInfo.getCalendarId());
			info.setUpdateDate(now);
			info.setUpdateUser(user);
			
			// 監視OID情報を削除
			Collection ct = info.getSnmpTrapOidInfo();
		    if(ct != null && ct.size() > 0){
		        Object[] infoArray = ct.toArray();
		        for(int j = 0; j < infoArray.length; j++){
		        	SnmpTrapOidInfoLocal oidInfo = (SnmpTrapOidInfoLocal)infoArray[j];
		            if(oidInfo != null){
		            	oidInfo.remove();
		            }
		        }
		    }	
			
			if(m_monitorInfo.getOidInfos() != null){
				
				for(int i = 0; i < m_monitorInfo.getOidInfos().size(); i++){
					MonitorSnmpTrapOidInfo oidInfo = m_monitorInfo.getOidInfos().get(i);
					
					SnmpTrapMasterLocal master = 
						SnmpTrapMasterUtil.getLocalHome().findByPrimaryKey(
							new SnmpTrapMasterPK(
									oidInfo.getMib(), 
									oidInfo.getTrapOid(), 
									oidInfo.getGenericId(), 
									oidInfo.getSpecificId()));
					
					//詳細メッセージをマスタの詳細メッセージと比較
					String descr = new String();
					if(oidInfo.getDescr() != null && master.getDescr() != null && 
							!oidInfo.getDescr().equalsIgnoreCase(master.getDescr())){
						if(!oidInfo.getDescr().equalsIgnoreCase(master.getDescr())){
							descr = oidInfo.getDescr();
						}
					}
					else{
						descr = oidInfo.getDescr();
					}
					
					// 監視OID情報を挿入
					SnmpTrapOidInfoUtil.getLocalHome().create(
							oidInfo.getMonitorId(),
							oidInfo.getMib(),
							oidInfo.getTrapOid(),
							new Integer(oidInfo.getGenericId()),
							new Integer(oidInfo.getSpecificId()),
							new Integer(oidInfo.getValidFlg()),
							new Integer(oidInfo.getPriority()),
							oidInfo.getLogmsg(),
							descr
					);
				}
			}

			result = true;
			
		} catch (CreateException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		} catch (FinderException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		} catch (EJBException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		} catch (RemoveException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		}
		
		return result;
	}
	
	/**
     * アプリケーションログにログを出力
     * 
     * @param e 例外
     */
	private void outputLog(Exception e, String method){
		AplLogger apllog = new AplLogger("TRAP", "trap");
		String[] args = {m_monitorInfo.getMonitorId() };
        apllog.put("SYS", "002", args);
        m_log.debug(method + ":" + e.getMessage());
	}
}
