/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.snmptrap.util;

import java.util.ArrayList;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;
import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.snmptrap.bean.TopicConstant;
import com.clustercontrol.snmptrap.factory.TrapSnmpManager;
import com.clustercontrol.snmptrap.message.UpdateSnmpTrapInfo;

/**
 * Topic受信クラス
 * Topicへの接続と、メッセージの受信を行う。
 * @version 2.1.0
 * @since 2.1.0
 * 
 */
public class ReceiveSnmpTrapTopic implements MessageListener, ExceptionListener {
	
	private static long RETRY_INTERVAL = 10000;
	
	private static final String TOPIC_CON_FACTORY = "ConnectionFactory";
	
	protected ArrayList m_facilityIdList;
	protected SendQueue m_sendQueue;
	
	private TopicConnectionFactory m_factory;
	
	protected TopicConnection m_con;
	protected Topic m_topic;
	protected TopicSession m_session;
	protected TopicSubscriber m_subscriber;
	
	protected TrapSnmpManager m_manager;
	
	private boolean m_errFlg = false;
	
	/** メッセージサービス再接続用タイマー * */
	protected static Timer m_timer = new Timer(true);
	
	//ロガー
	static private Log log = LogFactory.getLog(ReceiveSnmpTrapTopic.class);
	
	
	/**
	 * コンストラクタ
	 * @param trapSnmpManager
	 */
	@SuppressWarnings("unchecked")
	public ReceiveSnmpTrapTopic(TrapSnmpManager trapSnmpManager) {
		super();
		m_manager = trapSnmpManager;
		
		//接続処理
		initial();
	}
	
	/* 
	 * トピック受信処理
	 * (non-Javadoc)
	 * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
	 */
	@SuppressWarnings("unchecked")
	public void onMessage(Message message) {
		
		if(message instanceof ObjectMessage){
			ObjectMessage objectMessage = (ObjectMessage)message;
			Object obj;
			try {
				obj = objectMessage.getObject();
			} catch (JMSException e) {
				log.error("受信メッセージが不正", e);
				return;
			}
			
			if(obj instanceof UpdateSnmpTrapInfo){
				log.debug("onMessage get UpdateRepositoryInfo");
				
				UpdateSnmpTrapInfo info = (UpdateSnmpTrapInfo)obj;
				
				// キャッシュ更新
				m_manager.updateCache(info);
				
			}else{
				log.error("受信メッセージが不正" + obj.toString());
				return;
			}
		}else{
			log.error("受信メッセージが不正" + message.getClass());
			return;
		}
		log.debug("onMessage end");
	}
	
	/* 通信エラーハンドラ
	 * (non-Javadoc)
	 * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
	 */
	public void onException(JMSException arg0) {
		
		log.error(arg0);
		
		setErrFlg(true);
	}
	
	/**
	 * errFlg を設定。
	 * 
	 * @param errFlg
	 */
	synchronized private void setErrFlg(boolean errFlg) {
		if (m_errFlg == false && errFlg == true) {
			
			m_timer.schedule(new ReSetupTask(), RETRY_INTERVAL, RETRY_INTERVAL);
			
		}
		m_errFlg = errFlg;
	}
	
	/**
	 *  
	 */
	synchronized private boolean reInitial() {
		
		boolean ret = false;
		
		log.info("再接続処理!");
		
		terminate();
		
		if (initial()) {
			
			ret = true;
			
			log.info("再接続処理:成功!");
			
			//エラーフラグ解除
			setErrFlg(false);
			
			// キャッシュ更新
			m_manager.updateCache();
			
		} else {
			log.info("再接続処理:失敗!");
		}
		
		return ret;
	}
	/**
	 * サーバ接続の終了処理
	 *  
	 */
	public void terminate() {
		
		terminateSumscriber();
		
		
		try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
		}
		
		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e1) {
		}
	}
	/**
	 * トピック受信処理の終了 
	 */
	private void terminateSumscriber() {
		try {
			if (m_subscriber != null)
				m_subscriber.close();
		} catch (JMSException e) {
		}
	}
	
	/**
	 * 初期化処理
	 * JMSへの接続、トピック受信設定を行う
	 * @return
	 */
	private boolean initial() {
		
		log.info("EJB接続初期化");
		
		InitialContext context = null;
		
		try {
			//InitialContextの生成
			String serverName = System.getProperty("jboss.server.name");
			if(serverName.equals("all")){
				Properties properties = new Properties(); 
				properties.put(InitialContext.INITIAL_CONTEXT_FACTORY,"org.jnp.interfaces.NamingContextFactory");
				properties.put(InitialContext.URL_PKG_PREFIXES,"org.jboss.naming:org.jnp.interfaces");
				properties.put(InitialContext.PROVIDER_URL,"jnp://localhost:1100");
				
				context = new InitialContext(properties);
			}
			else{
				context = new InitialContext();
			}
			
			//コネクションファクトリ生成
			m_factory = (TopicConnectionFactory) context.lookup(TOPIC_CON_FACTORY);
			
			//コネクション生成
			m_con = m_factory.createTopicConnection();
			
			//セッション生成
			m_session = m_con.createTopicSession(false,
					Session.AUTO_ACKNOWLEDGE);
			
			//メッセージTopic取得
			m_topic = (Topic) context.lookup(TopicConstant.TOPIC_NAME_UPDATE);
			
			//エラーハンドラセット
			m_con.setExceptionListener(this);
			
			m_con.start();
			
			//トピック接続開始
			initialTopic();
			
		} catch (Exception e) {
			log.error("initial", e);
			setErrFlg(true);
			return false;
		} finally {
			try {
				if (context != null)
					context.close();
			} catch (Exception e1) {
			}
		}
		return true;
		
	}
	
	/**
	 * トピック受信設定
	 * @return
	 * @since
	 */
	private boolean initialTopic() {
		
		// 現在のTopic受信を終了
		terminateSumscriber();
		
		// トピック受信開始
		try {
			m_subscriber = m_session.createSubscriber(m_topic);
			
			//コールバックを登録する
			m_subscriber.setMessageListener(this);
			
		} catch (Exception e) {
			log.error("initialTopic", e);
			setErrFlg(true);
			return false;
		} finally {
			
		}
		return true;
		
	}
	/**
	 * EJB再接続タイマータスク
	 * 通信エラーとなった場合に定周期で呼ばれ再接続を行う 
	 */
	protected class ReSetupTask extends TimerTask {
		
		/**
		 * コネクションクローズ
		 */
		public void run() {
			if (reInitial()) {
				//このタスクをタイマーから解除
				cancel();
			}
		}
	}
}
