/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmptrap.util;

import java.io.Serializable;
import java.util.Properties;

import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicPublisher;
import javax.jms.TopicSession;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Topic送信クラス
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class SendTopic {
    protected static Log m_log = LogFactory.getLog( SendTopic.class );
    
    protected String m_topicName;
    protected TopicConnectionFactory m_connectionFactory;
    protected TopicConnection m_connection;
    protected Topic m_topic;
    protected TopicSession m_session;
    
    /**
     * コンストラクタ
     * 
     * @param topicName
     * @throws NamingException
     * @throws JMSException
     */
    public SendTopic(String topicName) throws NamingException, JMSException{
        m_topicName = topicName;
        
        //初期化
        connect();
    }
    
    /**
     * 初期化処理
     * 
     * @throws NamingException
     * @throws JMSException
     */
    protected void connect() throws NamingException, JMSException {
    	InitialContext context = null;
    	try{   
    		String serverName = System.getProperty("jboss.server.name");
    		if(serverName.equals("all")){
    			Properties properties = new Properties(); 
    			properties.put(InitialContext.INITIAL_CONTEXT_FACTORY,"org.jnp.interfaces.NamingContextFactory");
    			properties.put(InitialContext.URL_PKG_PREFIXES,"org.jboss.naming:org.jnp.interfaces");
    			properties.put(InitialContext.PROVIDER_URL,"jnp://localhost:1100");
    			
    			context = new InitialContext(properties);
    		}
    		else{
    			context = new InitialContext();
    		}
    		m_connectionFactory = (TopicConnectionFactory)context.lookup("ConnectionFactory");
    		m_connection = m_connectionFactory.createTopicConnection();
    		m_session = m_connection.createTopicSession( false, Session.AUTO_ACKNOWLEDGE );
    		m_connection.start();
    		m_topic = (Topic)context.lookup(m_topicName);
    	}
    	catch(NamingException e){
    		terminate();
    		throw e;
    	}
    	catch(JMSException e){
    		terminate();
    		throw e;
    	}
    	finally{
    		try {
    			if (context != null)
    				context.close();
    		}
    		catch (NamingException e) {
    		}
    	}
    }
    
    /**
     * メッセージ送信処理
     * 
     * @param info
     * @throws JMSException
     */
    public void put(Serializable orgMessage) throws JMSException {
    	TopicPublisher publisher = null;
    	try{
    		publisher = m_session.createPublisher(m_topic);
    		
    		//送信メッセージ作成
    		ObjectMessage message = m_session.createObjectMessage(orgMessage);
    		
    		//メッセージ送信
    		publisher.publish(message);
    		m_log.debug("put() topic ");
    	}
    	catch(JMSException e){
    		throw e;
    	}
    	finally{
    		try {
    			if (publisher != null)
    				publisher.close();
    		}
    		catch (JMSException e) {
    		}
    		
    		terminate();
    	}
    }
    
	/**
	 * メッセージ送信後処理
	 * 
	 * @throws JMSException
	 */
	public void terminate() {
		try {
			if (m_connection != null){
				m_connection.stop();
			}
		}
		catch (JMSException e){
			m_log.debug("terminate:" + e.getMessage());
		}
		
		try {
			if (m_session != null){
				m_session.close();
				m_session = null;
			}
		}
		catch (JMSException e) {
			m_log.debug("terminate:" + e.getMessage());
		}
		
		try {
			if (m_connection != null){
				m_connection.close();
				m_connection = null;
			}
		}
		catch (JMSException e){
			m_log.debug("terminate:" + e.getMessage());
		}
	}
}