/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.forward;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * syslog転送メインクラスです。<BR>
 * 標準入力でsyslogメッセージを受け取り、メッセージ管理クラスへ渡します。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class LogForward {

	/** メッセージ管理のインスタンス。 */
	private LogManager m_manageLog;

	/** ログ出力のインスタンス。 */
	static Log log = LogFactory.getLog(LogForward.class);

	/**
	 * メイン処理です。<BR>
	 * syslog転送を開始します。
	 * 
	 * @param args プロパティファイル名（配列の１個目）
	 * @throws Exception
	 * 
	 * @see #forward()
	 */
	public static void main(String[] args) throws Exception {

		log.debug("Start");

		LogForward logForward = new LogForward(args[0]);

		try {
			logForward.forward();
		} catch (Exception e) {
			log.error(e);
		}

		System.exit(0);

		log.debug("End");
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param propFileName プロパティファイル名
	 * @throws Exception
	 * 
	 * @see com.clustercontrol.syslogng.forward.LogManager
	 */
	LogForward(String propFileName) throws Exception {

	
		//プロパティファイル読み込み
		Properties props = new Properties();
		props.load(new FileInputStream(propFileName));

		m_manageLog = new LogManager(props);

		//ログ処理スレッドを開始
		m_manageLog.start();
		//スレッド待機待ち
		while (true) {
			if (m_manageLog.isWaiting())
				break;
		}
		
		//終了呼び出し設定
    	Runtime.getRuntime().addShutdownHook(new Thread() {
    	    public void run() { m_manageLog.interrupt(); }
    	});
		
	}

	/**
	 * メッセージを送信します。<BR>
	 * 引数で渡されたメッセージを、メッセージ管理へ渡します。
	 * 
	 * @param msg メッセージ
	 */
	private void put(String msg) {

		if (msg == null || msg.equals(""))
			return;

		log.debug("put Message:" + msg);

		m_manageLog.add(msg);

	}

	/**
	 * メッセージを転送します。<BR>
	 * 標準入力でメッセージを受け取り、メッセージ管理へ渡します。
	 * 
	 * @throws Exception
	 * 
	 * @see #put(String)
	 */
	public void forward() throws Exception {

		log.debug("Read InputStaream");

		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		while (true) { /* 標準入力待ち */
			try {
				//1行受信したらメッセージ送信
				put(br.readLine());
			} catch (IOException e) {
				log.error(e);
				break;
			}
		}
		m_manageLog.interrupt();
	}

}
