/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.factory;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.syslogng.bean.TopicConstant;
import com.clustercontrol.syslogng.message.UpdateMonitorRuleInfo;
import com.clustercontrol.syslogng.util.SendTopic;
import com.clustercontrol.syslogng.bean.LogFilterInfo;
import com.clustercontrol.syslogng.ejb.entity.SyslogMonitorRuleLocal;
import com.clustercontrol.syslogng.ejb.entity.SyslogMonitorRuleLocalHome;
import com.clustercontrol.syslogng.ejb.entity.SyslogMonitorRuleUtil;
import com.clustercontrol.util.apllog.AplLogger;


/**
 * フィルタ情報を登録するクラスです。
 *
 * @version 2.1.0
 * @since 1.0.0
 */
public class ModifyMonitorRule {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( ModifyMonitorRule.class );
	
	/**
	 * フィルタ情報を作成します。
	 * <p>
	 * <ol>
	 *  <li>フィルタ情報を作成します。</li>
	 *  <li>リポジトリ更新用トピックへ、フィルタ情報更新情報を送信します。
	 *      SyslogForwardで、SyslogForwardで保持しているフィルタ情報キャッシュが更新されます。</li>
	 * </ol>
	 * 
	 * @param list フィルタ情報一覧（{@link com.clustercontrol.syslogng.bean.LogFilterInfo}のリスト）
	 * @param user 作成ユーザ
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws CreateException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.syslogng.ejb.entity.SyslogMonitorRuleBean
	 * @see com.clustercontrol.syslogng.util.SendTopic
	 * @see com.clustercontrol.syslogng.bean.TopicConstant#TOPIC_NAME_EXECUTE
	 * @see com.clustercontrol.syslogng.message.UpdateMonitorRuleInfo
	 */
	public boolean createMonitorRule(ArrayList list, String user) throws CreateException, NamingException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.MONITOR_SYSLOG);
    	
		if(list == null){
			return true;
		}
		
		try
		{
			SyslogMonitorRuleLocalHome monitorRuleHome = SyslogMonitorRuleUtil.getLocalHome();
			SyslogMonitorRuleLocal monitorRule = null;
			
			Timestamp regDate = null;
			Timestamp updateDate = new Timestamp(new Date().getTime());
			
			String regUser = null;

			for(int index=0; index<list.size(); index++){
				
				if(list.get(index) instanceof LogFilterInfo){
					
					LogFilterInfo filterInfo = (LogFilterInfo)list.get(index);
					
					if(filterInfo.getRegDate() != null){
						regDate = new Timestamp(filterInfo.getRegDate().getTime());
					}
					else{
						regDate = updateDate;
					}
					
					if(filterInfo.getRegUser() != null && !"".equals(filterInfo.getRegUser().trim())){
						regUser = filterInfo.getRegUser();
					}
					else{
						regUser = user;
					}
					
					monitorRule = monitorRuleHome.create(
							new Integer(filterInfo.getOrderNo()),
							filterInfo.getDescription(),
							filterInfo.getFacilityId(),
							filterInfo.getPattern(),
							new Integer(filterInfo.getProcessType()),
							new Integer(filterInfo.getPriority()),
							filterInfo.getMonitorId(),
							filterInfo.getApplication(),
							filterInfo.getMessageId(),
							filterInfo.getMessage(),
							filterInfo.getNotifyId(),
							filterInfo.getCalendarId(),
							Integer.valueOf(filterInfo.getJobRun()),
							filterInfo.getJobId(),
							Integer.valueOf(filterInfo.getJobInhibitionFlg()),
							Integer.valueOf(filterInfo.getJobFailurePriority()),
							new Integer(ValidConstant.booleanToType(filterInfo.isValidFlg())),
							regDate,
							updateDate,
							regUser,
							user
							);
				}
			}
			
			try {
				//Topicに送信
				SendTopic send = new SendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
				send.put(new UpdateMonitorRuleInfo());
			} catch (Exception e) {
				m_log.debug("createMonitorRule() : 更新情報送信エラー : " + e.getMessage());
			}
			
			return true;
			
		} catch (CreateException e) {
            AplLogger apllog = new AplLogger("SLOGNG", "slogng");
            String[] args = {"insert"};
            apllog.put("SYS", "002", args);
			m_log.debug("createMonitorRule():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("SLOGNG", "slogng");
            String[] args = {"insert"};
            apllog.put("SYS", "002", args);
			m_log.debug("createMonitorRule():" + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * フィルタ情報を削除します。
	 * <p>
	 * <ol>
	 *  <li>全てのフィルタ情報を取得します。</li>
	 *  <li>フィルタ情報を削除します。</li>
	 * </ol>
	 * 
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws RemoveException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.syslogng.ejb.entity.SyslogMonitorRuleBean
	 */
	public boolean deleteMonitorRule() throws RemoveException, FinderException, NamingException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.MONITOR_SYSLOG);
    	
		try 
		{
			SyslogMonitorRuleLocalHome moniterRuleHome = SyslogMonitorRuleUtil.getLocalHome();
			
			// 全監視条件を取得
			Collection ct = moniterRuleHome.findAll();
			
			Iterator itr = ct.iterator();
			while(itr.hasNext())
			{
				SyslogMonitorRuleLocal moniterRule = (SyslogMonitorRuleLocal)itr.next();
				// 監視条件を削除
				moniterRuleHome.remove(moniterRule.getPrimaryKey());
			}
			return true;
			
		} catch (RemoveException e) {
            AplLogger apllog = new AplLogger("SLOGNG", "slogng");
            String[] args = {"delete"};
            apllog.put("SYS", "002", args);
			m_log.debug("createMonitorRule():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("SLOGNG", "slogng");
            String[] args = {"delete"};
            apllog.put("SYS", "002", args);
			m_log.debug("createMonitorRule():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("SLOGNG", "slogng");
            String[] args = {"delete"};
            apllog.put("SYS", "002", args);
			m_log.debug("createMonitorRule():" + e.getMessage());
			throw e;
		}
	}
}
