/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.troubledetection.util;

import java.util.LinkedList;
import java.util.NoSuchElementException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.monitor.message.ParallelLogOutputInfo;
import com.clustercontrol.troubledetection.bean.TroubleDetectionInfo;

/**
 * メール送信マネージャクラス
 * メッセージを受け取り、メールを送信する
 *  
 * @version 2.1.0
 * @since 2.1.0
 */
public class MailManager extends Thread {
	
	/** メール送信 */
	private static SendMail sender = new SendMail();
	
	/** ログ出力情報キュー */
	private LinkedList<Object> m_infoList = new LinkedList<Object>();
	
	/** 待機状態フラグ */
	private boolean m_waiting = false;
	
	/** 障害検知情報 */
	private TroubleDetectionInfo troubleDetectionInfo;
	
	/** ロガー */
	private Log log = LogFactory.getLog(this.getClass());
	
	/** シングルトン用インスタンス */
	private static MailManager instance = null;
	
	/**
	 * このクラスのインスタンスを取得します。
	 * 
	 * @return　MailManager メール送信マネージャ
	 */
	public synchronized static MailManager getMailManager() {
		if (instance == null) {
			instance = new MailManager();
		}
		return instance;
	}
	
	/**
	 * コンストラクタ
	 */
	private MailManager() {
		
	}
	
	/**
	 * 障害検知情報設定
	 * 
	 * @param info
	 */
	public void setTroubleDetectionInfo(TroubleDetectionInfo info) {
		troubleDetectionInfo = info;
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Runnable#run()
	 */
	public void run() {
		
		log.debug("Thread Start");
		
		while (true) {
			
			Object info = null;
			
			//処理要求待ち
			synchronized (this) {
				
				info = getInfo();
				if (info == null) {
					//ログ出力情報がないので処理待ち
					log.debug("run() : Waiting for processing...");
					m_waiting = true;
					try {
						wait();
					} catch (InterruptedException e) {
						break;
					}
					
					m_waiting = false;
					continue;
				}
				
			}
			
			
			//ログ出力情報を受信
			if (info instanceof ParallelLogOutputInfo) {
				//ログ出力情報を処理
				proccess((ParallelLogOutputInfo) info);
			}
		}
		
		//終了処理
		terminate();
		
		log.debug("Thread End");
		
	}
	
	/**
	 * メールを送信する。
	 * 
	 * @param info ログ出力情報
	 */
	protected void proccess(ParallelLogOutputInfo info) {
		//メール通知が成功するまでリトライ
		for(int i = 0; i < this.troubleDetectionInfo.getRetryMail().intValue(); i++){
			log.debug("proccess() :" + info);
			//メール送信
			if(sender.sendMail(
					info.getLogOutputInfo().getAddress(), 
					info.getLogOutputInfo(), 
					info.getOutputDate()))
				break;
		}
	}
	
	/**
	 * ログ出力情報を追加する
	 * 
	 * @param obj ログ出力情報
	 */
	public void add(Object obj) {
		
		boolean notifyFlg = false;
		
		synchronized (m_infoList) {
			if (m_infoList.size() == 0) {
				notifyFlg = true;
			}
			
			
			//ログ出力情報を処理
			if (obj instanceof ParallelLogOutputInfo) {
				//ログ出力情報をキューに追加
				m_infoList.add(obj);
			}
			
		}
		
		if ( notifyFlg && isWaiting() ) {
			//スレッドに処理開始指示
			synchronized (this) {
				this.notify();
			}
		}
		
	}
	
	/**
	 * ログ出力情報取り出し<BR>
	 * ログ出力情報用のキューから1つ取り出し ログ出力情報がないときはnullを返す
	 * 
	 * @return
	 */
	protected Object getInfo() {
		synchronized (m_infoList) {
			try {
				return m_infoList.removeFirst();
				
			} catch (NoSuchElementException e) {
				return null;
			}
			
		}
	}
	
	/**
	 * @return waiting を戻します。
	 */
	synchronized public boolean isWaiting() {
		return m_waiting;
	}
	
	/**
	 * サーバ接続の終了処理
	 *  
	 */
	private void terminate() {
		
	}
}
