/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.factory;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.FacilityTargetConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.monitor.ejb.entity.StatusInfoLocal;
import com.clustercontrol.monitor.ejb.entity.StatusInfoUtil;
import com.clustercontrol.monitor.ejb.session.MonitorControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ステータス情報を検索するクラスです。
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class SelectStatus {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SelectStatus.class );

	/**
	 * ステータス情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定されたファシリティ配下のファシリティを、指定されたファシリティのターゲットで取得します。</li>
	 * <li>取得したファシリティに属するステータス情報を取得します。</li>
	 * <li>１ステータス情報をテーブルのカラム順（{@link com.clustercontrol.monitor.bean.StatusTabelDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１ステータス情報を保持するリストを、ステータス情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>ステータス情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ ステータス情報1 {カラム1の値, カラム2の値, … }, ステータス情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param level 取得対象のファシリティのターゲット（配下全て／直下のみ）
	 * @return ステータス情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#ALL
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#ONE_LEVEL
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityIdList(String, int)
	 * @see com.clustercontrol.monitor.ejb.entity.StatusInfoBean#ejbFindStatus(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String)
	 * @see com.clustercontrol.monitor.bean.StatusTabelDefine
	 */
	public ArrayList getStatusList(String facilityId, int level) throws CreateException, FinderException, NamingException {

		ArrayList list = null;
		try 
		{
			String[] facilityIds = null;
			if(level == MonitorControllerBean.ONLY){
				if(facilityId != null && !"".equals(facilityId)){
					facilityIds = new String[1];
					facilityIds[0] = facilityId;	
				}
				else{
					return null;
				}
			}
			else{
				if(facilityId == null || facilityId.equals("")){
					facilityIds = new String[0];
				}
				else{
					// 直下 または 配下すべてのファシリティIDを取得
					ArrayList facilityIdList = RepositoryControllerUtil.getLocalHome().create().getFacilityIdList(facilityId, level);
					
					if(facilityIdList != null && facilityIdList.size() > 0){
						// スコープの場合
						facilityIds = new String[facilityIdList.size()];
						facilityIdList.toArray(facilityIds);
					}
					else{
						if(facilityId != null){
							// ノードの場合
							facilityIds = new String[1];
							facilityIds[0] = facilityId;
						}
						else{
							// リポジトリが1件も登録されていない場合
							return null;
						}
					}
				}
			}
	
			// ステータス情報一覧を取得
			Collection ct = StatusInfoUtil.getLocalHome().findStatus(
					facilityIds,
					null,
					null,
					null,
					null,
					null,
					null,
					null
			);
			
			// 2次元配列に変換
			list = this.collectionToArray(facilityId, ct);
			
		} catch (CreateException e) {
            AplLogger apllog = new AplLogger("MON", "mon");
            String[] args = {facilityId, null, null, null, null, null, null, null, null};
            apllog.put("SYS", "002", args);
            m_log.debug("getStatusList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("MON", "mon");
            String[] args = {facilityId, null, null, null, null, null, null, null, null};
            apllog.put("SYS", "002", args);
            m_log.debug("getStatusList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("MON", "mon");
            String[] args = {facilityId, null, null, null, null, null, null, null, null};
            apllog.put("SYS", "002", args);
            m_log.debug("getStatusList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * 引数で指定された条件に一致するステータス一覧情報を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定されたプロパティに格納された検索条件を、プロパティユーティリティ（{@link com.clustercontrol.util.PropertyUtil}）を使用して取得します。</li>
	 * <li>引数で指定されたファシリティ配下のファシリティと検索条件を基に、ステータス情報を取得します。</li>
	 * <li>１ステータス情報をテーブルのカラム順（{@link com.clustercontrol.monitor.bean.StatusTabelDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１ステータス情報を保持するリストを、ステータス情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>ステータス情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ ステータス情報1{カラム1の値, カラム2の値, … }, ステータス情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param property 検索条件
	 * @return ステータス情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.util.PropertyUtil#getPropertyValue(com.clustercontrol.bean.Property, java.lang.String)
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityIdList(String, int)
	 * @see com.clustercontrol.monitor.ejb.entity.StatusInfoBean#ejbFindStatus(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String)
	 * @see com.clustercontrol.monitor.bean.StatusTabelDefine
	 */
	public ArrayList getStatusList(String facilityId, Property property) throws CreateException, FinderException, NamingException {

		ArrayList list = null;
		
		Integer priority = null;
		Timestamp outputFromDate = null;
		Timestamp outputToDate = null;
		Timestamp generationFromDate = null;
		Timestamp generationToDate = null;
		String facilityType = null;
		String application = null;
		String message = null;;

		try 
		{
			ArrayList values = null;
			
			//重要度取得
			values = PropertyUtil.getPropertyValue(property, StatusFilterProperty.PRIORITY);
			if(!"".equals((String)values.get(0))){
				int priorityType = PriorityConstant.stringToType((String)values.get(0));
				if(priorityType != -1){
					priority = new Integer(priorityType);
				}
			}
			
			//更新日時（自）取得
			values = PropertyUtil.getPropertyValue(property, StatusFilterProperty.OUTPUT_FROM_DATE);
			if(values.get(0) instanceof Date){
				outputFromDate = new Timestamp(((Date)values.get(0)).getTime());
				outputFromDate.setNanos(0);
			}
			
			//更新日時（至）取得
			values = PropertyUtil.getPropertyValue(property, StatusFilterProperty.OUTPUT_TO_DATE);
			if(values.get(0) instanceof Date){
				outputToDate = new Timestamp(((Date)values.get(0)).getTime());
				outputToDate.setNanos(999999999);
			}
			
			//出力日時（自）取得
			values = PropertyUtil.getPropertyValue(property, StatusFilterProperty.GENERATION_FROM_DATE);
			if(values.get(0) instanceof Date){
				generationFromDate = new Timestamp(((Date)values.get(0)).getTime());
				generationFromDate.setNanos(0);
			}
			
			//出力日時（至）取得
			values = PropertyUtil.getPropertyValue(property, StatusFilterProperty.GENERATION_TO_DATE);
			if(values.get(0) instanceof Date){
				generationToDate = new Timestamp(((Date)values.get(0)).getTime());
				generationToDate.setNanos(999999999);
			}
			
			//対象ファシリティ種別取得
			values = PropertyUtil.getPropertyValue(property, StatusFilterProperty.FACILITY_TYPE);
			if(!"".equals((String)values.get(0))){
				facilityType = (String)values.get(0);
			}
			
			//アプリケーション取得
			values = PropertyUtil.getPropertyValue(property, StatusFilterProperty.APPLICATION);
			if(!"".equals((String)values.get(0))){
				application = (String)values.get(0);
			}
			
			//メッセージ取得
			values = PropertyUtil.getPropertyValue(property, StatusFilterProperty.MESSAGE);
			if(!"".equals((String)values.get(0))){
				message = (String)values.get(0);
			}

			String[] facilityIds = null;
			if(facilityId == null || facilityId.equals("")){
				facilityIds = new String[0];
			}
			else{
				// 対象ファシリティのファシリティIDを取得
				int level = RepositoryControllerBean.ALL;
				if(FacilityTargetConstant.STRING_BENEATH.equals(facilityType)){
					level = RepositoryControllerBean.ONE_LEVEL;
				}
				ArrayList facilityIdList = RepositoryControllerUtil.getLocalHome().create().getFacilityIdList(facilityId, level);

				if(facilityIdList != null && facilityIdList.size() > 0){
					// スコープの場合
					facilityIds = new String[facilityIdList.size()];
					facilityIdList.toArray(facilityIds);
				}
				else{
					if(facilityId != null){
						// ノードの場合
						facilityIds = new String[1];
						facilityIds[0] = facilityId;
					}
					else{
						// リポジトリが1件も登録されていない場合
						return null;
					}
				}
			}
			
			// ステータス情報一覧を、検索条件を指定して取得
			Collection ct = StatusInfoUtil.getLocalHome().findStatus(
					facilityIds,
					priority,
					outputFromDate,
					outputToDate,
					generationFromDate,
					generationToDate,
					application,
					message
			);

			// 2次元配列に変換
			list = this.collectionToArray(facilityId, ct);

		} catch (CreateException e) {
			int priorityInt = -1;
			String outputFromDateString = null;
			String outputToDateString = null;
			String generationFromDateString = null;
			String generationToDateString = null;
			
			if(priority != null)
				priorityInt = priority.intValue();
			if(outputFromDate != null)
				outputFromDateString = outputFromDate.toString();
			if(outputToDate != null)
				outputToDateString = outputToDate.toString();
			if(generationFromDate != null)
				generationFromDateString = generationFromDate.toString();
			if(generationToDate != null)
				generationToDateString = generationToDate.toString();
			
            AplLogger apllog = new AplLogger("MON", "mon");
            String[] args = {facilityId,PriorityConstant.typeToString(priorityInt),outputFromDateString,outputToDateString,generationFromDateString,generationToDateString,facilityType,application,message};
            apllog.put("SYS", "002", args);
            m_log.debug("getStatusList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			int priorityInt = -1;
			String outputFromDateString = null;
			String outputToDateString = null;
			String generationFromDateString = null;
			String generationToDateString = null;
			
			if(priority != null)
				priorityInt = priority.intValue();
			if(outputFromDate != null)
				outputFromDateString = outputFromDate.toString();
			if(outputToDate != null)
				outputToDateString = outputToDate.toString();
			if(generationFromDate != null)
				generationFromDateString = generationFromDate.toString();
			if(generationToDate != null)
				generationToDateString = generationToDate.toString();
			
            AplLogger apllog = new AplLogger("MON", "mon");
            String[] args = {facilityId,PriorityConstant.typeToString(priorityInt),outputFromDateString,outputToDateString,generationFromDateString,generationToDateString,facilityType,application,message};
            apllog.put("SYS", "002", args);
            m_log.debug("getStatusList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			int priorityInt = -1;
			String outputFromDateString = null;
			String outputToDateString = null;
			String generationFromDateString = null;
			String generationToDateString = null;
			
			if(priority != null)
				priorityInt = priority.intValue();
			if(outputFromDate != null)
				outputFromDateString = outputFromDate.toString();
			if(outputToDate != null)
				outputToDateString = outputToDate.toString();
			if(generationFromDate != null)
				generationFromDateString = generationFromDate.toString();
			if(generationToDate != null)
				generationToDateString = generationToDate.toString();
			
            AplLogger apllog = new AplLogger("MON", "mon");
            String[] args = {facilityId,PriorityConstant.typeToString(priorityInt),outputFromDateString,outputToDateString,generationFromDateString,generationToDateString,facilityType,application,message};
            apllog.put("SYS", "002", args);
            m_log.debug("getStatusList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * 重要度が最高で受信日時が最新のステータス情報を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定されたファシリティ配下のファシリティを、指定されたファシリティのターゲットで取得します。</li>
	 * <li>取得したファシリティに属する重要度が最高 かつ 受信日時が最新のステータス情報を取得し返します。</li>
	 * </ol>
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param level 取得対象のファシリティのターゲット（配下全て／直下のみ）
	 * @return ステータス情報のローカルコンポーネントインターフェース
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityIdList(String, int)
	 * @see com.clustercontrol.monitor.ejb.entity.StatusInfoBean#ejbFindHighPriorityStatus(String[], Timestamp, Timestamp, Timestamp, Timestamp, String, String)
	 */
	public StatusInfoLocal getHighPriorityStatus(String facilityId, int level) throws CreateException, FinderException, NamingException {

		StatusInfoLocal status = null;
		try 
		{
			String[] facilityIds = null;
			if(level == MonitorControllerBean.ONLY){
				if(facilityId != null && !"".equals(facilityId)){
					facilityIds = new String[1];
					facilityIds[0] = facilityId;	
				}
				else{
					return null;
				}
			}
			else{
				// 直下 または 配下すべてのファシリティIDを取得
				ArrayList facilityIdList = RepositoryControllerUtil.getLocalHome().create().getFacilityIdList(facilityId, level);
				
				if(facilityIdList != null && facilityIdList.size() > 0){
					// スコープの場合
					facilityIds = new String[facilityIdList.size()];
					facilityIdList.toArray(facilityIds);
				}
				else{
					if(facilityId != null){
						// ノードの場合
						facilityIds = new String[1];
						facilityIds[0] = facilityId;
					}
					else{
						// リポジトリが1件も登録されていない場合
						return null;
					}
				}
			}
	
			// ステータス情報一覧を取得
			Collection ct = StatusInfoUtil.getLocalHome().findHighPriorityStatus(
					facilityIds,
					null,
					null,
					null,
					null,
					null,
					null);
			
			Iterator itr = ct.iterator();
			if(itr.hasNext()){
				status = (StatusInfoLocal)itr.next();
			}
			
		} catch (CreateException e) {
			m_log.debug("getHighPriorityStatus():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			m_log.debug("getHighPriorityStatus():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.debug("getHighPriorityStatus():" + e.getMessage());
			throw e;
		}
		return status;
	}
	
	/**
	 * DBより取得したステータス情報のObjectの2次元配列を返します。
	 * <p>
	 * <li>１イベント情報をテーブルのカラム順（{@link com.clustercontrol.monitor.bean.StatusTabelDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１イベント情報を保持するリストを、イベント情報一覧を保持するリスト（{@link ArrayList}）に格納します。<BR>
	 *  <dl>
	 *  <dt>ステータス情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ ステータス情報1 {カラム1の値, カラム2の値, … }, ステータス情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @param parentFacilityId ルートファシリティID
	 * @param ct ステータス情報取得結果
	 * @return ステータス情報一覧（Objectの2次元配列）
	 * 
	 * @see com.clustercontrol.monitor.bean.StatusTabelDefine
	 */
	@SuppressWarnings("unchecked")
	private ArrayList collectionToArray(String parentFacilityId, Collection ct) throws CreateException, FinderException, NamingException{
		
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();

		ArrayList list = new ArrayList();
		Iterator itr = ct.iterator();
		while(itr.hasNext())
		{
			StatusInfoLocal status = (StatusInfoLocal)itr.next();
			
			// スコープの取得
			String facilityPath = repository.getFacilityPath(status.getFacilityId(), parentFacilityId);
			
			ArrayList info = new ArrayList();
			info.add(status.getPriority());
			info.add(status.getPluginId());
			info.add(status.getMonitorId());
			info.add(status.getFacilityId());
			info.add(facilityPath);
			info.add(status.getApplication());
			info.add(new Date(status.getOutputDate().getTime()));
			info.add(new Date(status.getGenerationDate().getTime()));
			info.add(status.getMessageId());
			info.add(status.getMessage());
			list.add(info);
		}
		return list;
	}
}
