/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 */

package com.clustercontrol.performance.monitor.composite;

import java.util.Iterator;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.performance.util.CollectorItemCodeFactory;
import com.clustercontrol.performanceMGR.bean.CollectorItemInfo;
import com.clustercontrol.performanceMGR.monitor.bean.MonitorPerfCheckInfo;
import com.clustercontrol.util.Messages;

/**
 * 収集項目コードを選択するコンポジットクラスです。
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class CollectorItemComboComposite extends Composite {
	
	public static final int WIDTH_TITLE = 5;
	public static final int WIDTH_VALUE = 8;
	
    /** 収集項目 */
    private Combo m_comboCollectorItem = null;
	
	public CollectorItemComboComposite(Composite parent, int style) {
		super(parent, style);
        this.initialize();
    }
    
    /**
     * コンポジットを生成・構築します。
     */
    private void initialize() {
    	
    	// 変数として利用されるラベル
        Label label = null;
        // 変数として利用されるグリッドデータ
        GridData gridData = null;
        
        GridLayout layout = new GridLayout(1, true);
        layout.marginHeight = 0;
        layout.marginWidth = 0;
    	layout.numColumns = 15;	
        this.setLayout(layout);
        
        /*
         * 収集項目
         */
        // ラベル
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("monitor.item") + " : ");
        // コンボボックス
        this.m_comboCollectorItem = new Combo(this, SWT.DROP_DOWN | SWT.READ_ONLY);
        
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_comboCollectorItem.setLayoutData(gridData);
    }
    
    /**
     * 収集項目を選択するコンボボックスを生成します。
     * 
     * @param facilityId ファシリティID
     */
    public void setCollectorItemCombo(String facilityId){
		// 収集項目の一覧を生成
		List itemInfoList =
			CollectorItemCodeFactory.getAvailableCollectorItemList(facilityId);
		
		// 現在設定されている項目を全て消去
		m_comboCollectorItem.removeAll();
		
		Iterator itr = itemInfoList.iterator();
		while(itr.hasNext()){
			CollectorItemInfo itemInfo = (CollectorItemInfo)itr.next();
			
			String itemName = CollectorItemCodeFactory.getFullItemName(itemInfo);
			m_comboCollectorItem.add(itemName);
			m_comboCollectorItem.setData(itemName, itemInfo);
		}
    }
    
    public CollectorItemInfo getCollectorItem(){
        String itemName = m_comboCollectorItem.getText();
        CollectorItemInfo itemInfo = (CollectorItemInfo)m_comboCollectorItem.getData(itemName);
        return itemInfo;
    }
    
    public void select(MonitorInfo monitor){
        // 性能監視情報
        MonitorPerfCheckInfo perfCheckInfo = (MonitorPerfCheckInfo)monitor.getCheckInfo();

        // 監視対象の情報が設定されていない場合はなにもしない
        if(perfCheckInfo == null){
        	return;
        }
        
    	String facilityId = monitor.getFacilityId();

    	setCollectorItemCombo(facilityId);
    	
    	int index = 0;
    	
    	List itemInfoList = 
    		CollectorItemCodeFactory.getAvailableCollectorItemList(facilityId);
    	
    	Iterator itr = itemInfoList.iterator();
    	int i=0;
    	while(itr.hasNext()){
    		CollectorItemInfo itemInfo = (CollectorItemInfo)itr.next();
    		
    		if(itemInfo.getCollectorItemCode().equals(perfCheckInfo.getItemCode()) &&
    				itemInfo.getDeviceName().equals(perfCheckInfo.getDeviceName())){
    			index = i;
    		}
    		i++;
    	}
    	
        this.m_comboCollectorItem.select(index);
    }
    
	/**
     * 入力値を用いて監視情報を生成します。
     * 
     * @return 検証結果
     */
	public ValidateResult createInputData(MonitorInfo info) {
		if(info != null){
	        // リソース監視情報
	        MonitorPerfCheckInfo perfCheckInfo = (MonitorPerfCheckInfo)info.getCheckInfo();
			
	        if (this.m_comboCollectorItem.getText() != null
	                && !"".equals((this.m_comboCollectorItem.getText()).trim())) {
	        	String itemName = this.m_comboCollectorItem.getText();
	        	CollectorItemInfo itemInfo = (CollectorItemInfo)m_comboCollectorItem.getData(itemName);
	        	
	            perfCheckInfo.setMonitorTypeId("PERF");
	            perfCheckInfo.setMonitorId(info.getMonitorId());
	            
	            perfCheckInfo.setItemCode(itemInfo.getCollectorItemCode());
	            perfCheckInfo.setDeviceIndex(itemInfo.getDeviceIndex());
	            perfCheckInfo.setDeviceName(itemInfo.getDeviceName());
	            
	            info.setCheckInfo(perfCheckInfo);
	        } else {
	        	return setValidateResult(Messages.getString("message.hinemos.1"),
	        			Messages.getString("message.monitor.1"));
	        }
		}
		return null;
    }
	
	/**
     * 無効な入力値の情報を設定します
     *  
     */
    protected ValidateResult setValidateResult(String id, String message) {

    	ValidateResult validateResult = new ValidateResult();
        validateResult.setValid(false);
        validateResult.setID(id);
        validateResult.setMessage(message);
        
        return validateResult;
    }
}
