/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performanceMGR.bean;

import java.lang.Double;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.ejb.EJBException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.performanceMGR.util.CalculationMethod;
import com.clustercontrol.performanceMGR.util.JNDIConnectionManager;
import com.clustercontrol.performanceMGR.util.Repository;
import com.clustercontrol.sharedtable.bean.ValueObject;
import com.clustercontrol.snmppoller.SnmpSharedTable;

/**
 * Facilityのノード実装クラス
 * 種別がノードであるファシリティの性能値を保持します。
 * ScopeTreeクラスの内部で使用することを目的としたクラスです。
 * 
 * @version 1.0
 * @since 1.0
 */
public class Node extends Facility {
	//	ログ出力
	protected static Log m_log = LogFactory.getLog( Node.class );
	
	private HashMap<String, HashMap<Integer, long[]>> m_mibValueSet; // 性能値の計算に必要なMIB値の集合
//	private HashMap<String, long[][]> m_mibValueSet; // 性能値の計算に必要なMIB値の集合
	
	private long lastCollectTime;   // 最終更新時刻
		
	/**
	 *コンストラクター 
	 * @param facilityID
	 * @param faclityName
	 * @param ipAddress
	 */
	Node(final String facilityID, final String faclityName){
		super(facilityID, faclityName, Facility.NODE);
		
		m_mibValueSet = new HashMap<String, HashMap<Integer, long[]>>();
	}
	
	/**
	 * 性能値を戻します。
	 * 
	 * @param  itemCode 収集項目コード
	 * @param  deviceIndex デバイス番号(現在は未使用)
	 * @param  deviceName デバイス名
	 * @return 性能値
	 */
	public double calcValue(final String itemCode, final int deviceIndex, String deviceName){
		// deviceIndexは使用しない
		
		// デバイス番号の初期化
		int devIndex = 0;
		
		// デバイス名からデバイス番号を逆引きする
		if (deviceName != null) {
			Repository repository = new Repository();
			DeviceData device = null;
			
			List<DeviceData> deviceList = repository.getDeviceList(m_facilityID);
			Iterator itr = deviceList.iterator();
			
			while(itr.hasNext()) {
				device = (DeviceData)itr.next();
				
				if (deviceName.compareTo(device.getDeviceName()) == 0 ){
					devIndex = device.getDeviceIndex();
					break;
				}
			}
		}
		
		double value = CalculationMethod.getPerformance(itemCode, devIndex, m_mibValueSet);
		
		m_log.debug("calcValue() : " + m_facilityID + " " + 
				itemCode + " " +deviceName + " " + devIndex + "  " + value);
		
		// 性能値が算出できた場合はバッファに保存する
		if(!Double.isNaN(value)){
			setCalcValueBuffer(new CollectorItemPK(itemCode, devIndex), value);
		}
		
		return value;
	}

	
	/**
	 * ポーラーから収集値を取得します。
	 * 
	 * @param oids 収集対象のOIDの配列
	 * @param interval 収集間隔
	 * @return 収集時刻（全てのノードで一致させるために最後に収集された値の収集時刻とする）
	 */
	public long fetchMibValue(final OIDInfo[] oids, int interval) {
		m_log.debug("fetchMibValue() start :");
		
		// 一度値を書き込んだOIDの値を2度書き込まないように書き込んだOIDを保持する
		HashSet<String> checkOid = new HashSet<String>();  
		
		// SNMP収集値の共有テーブルをルックアップします。
		SnmpSharedTable sst = null;
		try {
			InitialContext iniCtx = JNDIConnectionManager.getInitialContext();
			
			Object obj = iniCtx.lookup(SnmpPollerConstant.JNDI_NAME);

			// 取得したオブジェクトをキャストします。
			sst = (SnmpSharedTable)PortableRemoteObject.narrow(obj, SnmpSharedTable.class);
		} catch (NamingException e) {
			// エラー処理
			throw new EJBException(e.getMessage());
		}
		
		// 性能値の算出に必要なMIB値を取得
		for(int i=0; i<oids.length; i++){
			// デバイス全てを収集対象とするか否かで条件分岐
			if(!oids[i].isAllIndex()){
				
				String oid = oids[i].getOID();
				ValueObject value = sst.getValue(getFacilityID(), interval, oid);
				
				if(value == null){
					// エラー処理
					m_log.debug("Node : " + getFacilityID() + " can't get value " + oids[i].getOID());
					
					// 対象のOIDの値をまだセットしていない場合のみ処理を行う
					if(!checkOid.contains(oid)){	
						// 値取得に失敗している場合でも、取得失敗時の適用値が設定されている場合は、
						// その値をセットする
						Long defaultValue = oids[i].getDefaultValue();
						if(defaultValue != null){
							// 値をセットする
							setMibValueSet(oids[i].getBaseOid(), oids[i].getOidIndex(), defaultValue);

							// デバック出力
							if(m_log.isTraceEnabled()){
								m_log.trace("fetchMibValue() : " + getFacilityID()
										+ " set default value "	+ oid + " = " + defaultValue);
							}
						} else {
							// 値取得に失敗し、取得失敗時の適用値が設定されていない場合は、nullとする
							// CalculationMethodクラスでnullチェックし、nullの場合は、性能値をNaNとする
							m_mibValueSet.put(oids[i].getBaseOid(), null);
							
							// デバック出力
							if(m_log.isTraceEnabled()){
								m_log.trace("fetchMibValue() : " + getFacilityID()
										+ " set default value "	+ oid + " = " + defaultValue);
							}
						}
						checkOid.add(oid);  // 書き込み済みのOIDを設定する
					}
				} else {
					String mibValueString = (String)value.getValue();
					
					// デバッグ
					m_log.debug("Node : " + getFacilityID() + " " + oids[i].getOID() + " " + value);
					
					// 対象のOIDの値をまだセットしていない場合のみ処理を行う
					if(!checkOid.contains(oid)){	
						// 値をセットする
						setMibValueSet(oids[i].getBaseOid(), oids[i].getOidIndex(), Long.parseLong(mibValueString));
						checkOid.add(oid);  // 書き込み済みのOIDを設定する
						lastCollectTime = Math.max(lastCollectTime, value.getDate());
					}
				}
				
			} else {
				// デバイスインデックスが連番でないデバイスの合計値を計算し、
				// oids[i].getBaseOid()のインデックス0の領域に格納する。
				// 例） 下記のような構成の場合、合計値は(841113847 + 0 + 3519835615)
				// IF-MIB::ifInOctets.1 = Counter32: 841113847
				// IF-MIB::ifInOctets.65539 = Counter32: 0
				// IF-MIB::ifInOctets.65540 = Counter32: 3519835615
				Set set = sst.getValueSet(getFacilityID(), interval, oids[i].getBaseOid());
				
				// 指定のOID情報がSNMPポーリング結果のテーブルにない場合は、breakする。
				if(set == null){
					break;
				}
				
				Iterator itr = set.iterator();

				if (set.size() != 0) {
					long total = 0;
					int index = 0;
					long value = 0;
					while (itr.hasNext()) {
						ValueObject vo = (ValueObject) itr.next();
						try {
							String oidStr = vo.getKey();
							index = Integer.parseInt(oidStr.substring(oidStr.lastIndexOf(".") + 1));
							value = Long.parseLong((String)vo.getValue());

							// 対象のOIDの値をまだセットしていない場合のみ処理を行う
							if (!checkOid.contains(oidStr)) {
								setMibValueSet(oids[i].getBaseOid(), index,	value);
								checkOid.add(oidStr); // 書き込み済みのOIDを設定する
							}
						} catch (Exception e) {
							m_log.error(e.getMessage(), e);
						}

						total = total + value;
					}
					String oidStr = oids[i].getBaseOid() + ".0";
					// 対象のOIDの値をまだセットしていない場合のみ処理を行う
					if (!checkOid.contains(oidStr)) {
						// インデックスが0の領域に、合計値を格納する。
						setMibValueSet(oids[i].getBaseOid(), 0, total);
						checkOid.add(oidStr); // 書き込み済みのOIDを設定する
					}
				} else {
					// エラー処理
					m_log.debug("Node : " + getFacilityID()
									+ " can't get value "
									+ oids[i].getBaseOid() + ".*");

					// CalculationMethodクラスでnullチェックし、nullの場合は、性能値をNaNとする
					m_mibValueSet.put(oids[i].getBaseOid(), null);
				}
			}
		}
		
		m_log.debug("fetchMibValue() end :");
		return lastCollectTime;
	}
	
	/**
	 * 
	 * 最新のMIB値を配列に設定します。
	 *  
	 * @param baseOid 収集対象OIDの最後の"."より前を表す文字列
	 * @param tableIndex　収集対象OIDの最後の"."以下の数値
	 * @param mibValue　収集値
	 */
	private void setMibValueSet(final String baseOid, final Integer tableIndex, final long mibValue) {
		m_log.trace("setMibValueSet() :" + baseOid + " " + tableIndex + "  " + mibValue);
		HashMap<Integer, long[]> mibValueSetList = m_mibValueSet.get(baseOid);
		
		// 値を格納する領域が確保されていない場合
		if (mibValueSetList == null){
			mibValueSetList = new HashMap<Integer, long[]>();
			
			// テーブルに追加
			m_mibValueSet.put(baseOid, mibValueSetList);
		}
		
		// 指定のインデックスの領域が確保されていない場合
		if (mibValueSetList.get(tableIndex) == null){	
			// 新規に領域を確保	
			long[] buffer = new long[2];
			mibValueSetList.put(tableIndex, buffer);
					
			// テーブルに追加
			m_mibValueSet.put(baseOid, mibValueSetList);
		}
		
		// 今まで保持していた値を前回収集の値として設定。
		mibValueSetList.get(tableIndex)[0] = mibValueSetList.get(tableIndex)[1];
		
		// 新規の値を設定。
		mibValueSetList.get(tableIndex)[1] = mibValue;
	}
	
	/**
	 * 自分自身を返す。
	 * @return HashSet 
	 */
	public HashSet getNode(HashSet nodeSet){
		nodeSet.add(this);
		return nodeSet;
	}
	
	/**
	 * 自分自身をカウントして返す。
	 */
	public int getNodeCount(){
		return 1;
	}
	
//	/**
//	 * 現在保持しているMIB値をDBに出力する
//	 *
//	 */
//	public void storeMibValue(String collectorId){
//		m_log.debug("storeRowMibValue() :" + collectorId);
//
//		// DBへ格納する日時情報
//		Date date = new Date();  // 現在の時刻とする
//		
//		// MIB値を格納するためのテーブルにアクセス可能なオブジェクトを生成
//		RecordDataDAO dao = new RecordDataDAO();
//		
//		// m_mibValueSet のキーはOID(最後のインデックスを除く)
//		Iterator itr = m_mibValueSet.keySet().iterator();  
//
//		while(itr.hasNext()){
//			String baseOid = (String)itr.next();
//
//			long[][] mibValueSetList = (long[][])m_mibValueSet.get(baseOid);
//
//			for(int i=0; i<mibValueSetList.length; i++){
//				String fullOid = baseOid + "." + i;
//				
//				dao.insertRecordData(
//						collectorId, 
//						fullOid,
//						date,
//						getFacilityID(),
//						mibValueSetList[i][0]);
//			}
//		}
//	}
}
