/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/


package com.clustercontrol.performanceMGR.bean;

import java.io.Serializable;
import java.util.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;

import javax.ejb.EJBException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityInfo;
import com.clustercontrol.bean.FacilityTreeItem;

/**
 * 収集対象スコープツリーの各スコープ，ノードの関係を保持し、
 * 指定のファシリティ（スコープもしくはノード）の性能値を算出して返す機能を提供するクラス
 * 
 * @version 1.0
 * @since 1.0
 */
public class ScopeTree implements Serializable {
	//	ログ出力
	protected static Log m_log = LogFactory.getLog(ScopeTree.class);
	
//	//** ルートとなるファシリティのID */
//	private static String rootScopeID = "root";
//	//**ルートとなるファシリティの名称　*/
//	private static String rootScopeName = "root";
	
	private Hashtable m_scopeTable;  // スコープ情報を保持するテーブル
	private Hashtable m_nodeTable;   // ノード情報を保持するテーブル
	
	private long m_lastCollectTime;			// 最終収集日時を保持
	
	// MIB情報を取得した際に正しく情報が更新されているかのフラグ
	// ポーリングに失敗し情報が更新されていない場合はfalseとなる
	private boolean updateFlag;

	/**
	 * コンストラクター
	 * 
	 * @param treeItem FacilityTreeItemの階層オブジェクト
	 */
	public ScopeTree(FacilityTreeItem treeItem){
		m_log.debug("ScopeTree コンストラクタ");
		
		m_scopeTable = new Hashtable();
		m_nodeTable = new Hashtable();
		
		this.setScopeTree(treeItem);
	}
	
	/**
	 * ファシリティツリーにスコープを追加します。
	 * 
	 * @param facilityID
	 * @param facilityName
	 * @param parents
	 */
	public void addScope(final String facilityID, final String facilityName, final String parent) {
		Scope scope = new Scope(facilityID, facilityName);
		
		// 親ファシリティを取得
		Scope parentScope = (Scope)m_scopeTable.get(parent);
			
		if (parentScope == null) {
		} else {
			// 新規追加ファシリティに親を登録する。
			scope.addParents(parentScope);
				
			// 親ファシリティに登録する。
			parentScope.addChildren(scope);		
		}
		
		// ファシリティテーブルにファシリティを追加
		m_scopeTable.put(facilityID, scope);
	}
	
	/**
	 * ファシリティツリーにノードを追加します。
	 * 
	 * @param facilityID
	 * @param facilityName
	 * @param parents
	 */
	public void addNode(final String facilityID, final String facilityName, final String parent) {
		// 既に同じIDのノードが登録されているかどうかを調べる
		Node node = (Node)m_nodeTable.get(facilityID);
		if(node == null){
			node = new Node(facilityID, facilityName);
		}
		
		// 親ファシリティを取得
		Scope parentScope = (Scope)m_scopeTable.get(parent);	
		if (parentScope == null) {
			// エラー処理
		} else {
			// 新規追加ファシリティに親を登録する。
			node.addParents(parentScope);
			
			// 親ファシリティにノードを登録する。
			parentScope.addChildren(node);		
		}
		
		// ノードテーブルにノードを追加
		m_nodeTable.put(facilityID, node);
	}
	
	/**
	 * 指定のファシリティIDのファシリティもしくはノードが存在するか否かを判定します。
	 * 
	 * @param facilityID
	 * @return　true 存在する　false 存在しない
	 */
	public boolean contains(String facilityID){
		Scope scope = (Scope)m_scopeTable.get(facilityID);	
		Node node = (Node)m_nodeTable.get(facilityID);
		if(scope == null && node == null){
			return false;
		} else {
			return true;
		}
	}
	
	/**
	 * 指定ファシリティに含まれるノードのファシリティIDのリストを返します。
	 * 
	 * @param facilityID ファシリティID
	 * @return ノードのファシリティIDのリスト
	 */
	public String[] getNodeIDList(String facilityID){
		Scope scope = (Scope)m_scopeTable.get(facilityID);	
		
		String[] nodeListFaciliyID = null;
		
		HashSet nodeListSet = new HashSet();
		if (scope != null){
			// 指定ファシリティがスコープであった場合
			scope.getNode(nodeListSet);
			Node[] nodeList = new Node[nodeListSet.size()];
			nodeListSet.toArray(nodeList);
			
			nodeListFaciliyID = new String[nodeList.length];
			for(int i=0; i<nodeList.length; i++){
				nodeListFaciliyID[i] = nodeList[i].getFacilityID();
			}
		} else {
			// 指定ファシリティがノードであった場合
			Node node = ((Node)m_nodeTable.get(facilityID));
			
			if(node != null){
				nodeListFaciliyID = new String[1];
				nodeListFaciliyID[0] = node.getFacilityID();
			} else {
				// エラー処理
				String message= "Facility not found : " + facilityID;
				throw new EJBException(message);
			}
		}
		
		return nodeListFaciliyID;
	}
	
	/**
	 * 指定ファシリティのサブファシリティのファシリティIDのリストを返します。
	 * 
	 * @param facilityID
	 * @return サブファシリティのファシリティIDのリスト
	 */
	public String[] getSubScopeIDList(String facilityID){
		Scope scope = (Scope)m_scopeTable.get(facilityID);
		
		if(scope == null){
			// エラー処理
			String message= "Facility not found : " + facilityID;
			throw new EJBException(message);
		}	
		
		return scope.getChildrenID();
	}
	
	/**
	 * 指定ファシリティ以下に含まれるノードにMIB値を設定する。
	 * 
	 * @param facilityId ファシリティID
	 * @param oids 収集対象のOIDの配列
	 * @param interval 収集間隔
	 */
	synchronized public void fetchMibValue(final String facilityId, final OIDInfo[] oids, int interval){
		// 指定ファシリティの下に含まれる全てのノードのファシリティIDを取得
		String[] nodeFid = getNodeIDList(facilityId);

		for(int i=0; i<nodeFid.length; i++){
			Node node = ((Node)m_nodeTable.get(nodeFid[i]));
			
			if(node != null){
				// MIBの値を取得し収集日時を設定する
				long lastCollectTime = node.fetchMibValue(oids, interval);

				// 前回収集時刻を保持
				long collectTimeBuffer = m_lastCollectTime;
				
				// 全てのノードで時刻を一致させるために最後に収集されたノードの値の収集時刻とする
				m_lastCollectTime = Math.max(m_lastCollectTime, lastCollectTime);
				
				// 前回収集時刻と今回の収集時刻が異なればフラグをtrueとする
				updateFlag = m_lastCollectTime != collectTimeBuffer;
			} else {
				// エラー処理
				String message= "Facility not found : " + nodeFid[i];
				throw new EJBException(message);
			}
		}
	}
	
	/**
	 * 対象ファシリティの性能値を算出して返す。
	 * 
	 * @param facilityID ファシリティID
	 * @param itemCode　収集項目コード 　
	 * @param deviceIndex　デバイスのインデックス 　
	 * @return 計算された性能データ
	 */
	synchronized public CollectedDataInfo getValue(
			final String facilityID, 
			final String itemCode, 
			final int deviceIndex,
			final String deviceName){
		double value = Double.NaN;
		
		// スコープの場合
		Facility facility = (Facility)m_scopeTable.get(facilityID);
		
		// ノードの場合
		if(facility == null){
			facility = (Facility)m_nodeTable.get(facilityID);
		}
		
		if(facility == null){
			// エラー処理
			// 登録されていないファシリティを参照しようとした
			String message= "Facility not found : " + facilityID;
			throw new EJBException(message);
		} else {
			value = facility.calcValue(itemCode, deviceIndex, deviceName);
		}		
		
		CollectedDataInfo ret = new CollectedDataInfo(facilityID, itemCode, deviceIndex, deviceName);
		
		// ポーリングの結果が正しく取得できているか否かで取得日時を変える
		// 前回と同じ取得時刻であるとduplicate keyが発生しDBに格納できない
		if(updateFlag){
			ret.setDate(new Date(m_lastCollectTime));  // 収集日時を設定
		} else {
			ret.setDate(new Date());  // 収集日時を設定
		}

		ret.setValue(value);  // 性能値を設定
		
		return ret;
	}

	/**
	 * リアルタイム収集時に値を算出できなかった場合に、前回の計算時の計算値を返す。
	 * 
	 * @param facilityID ファシリティID
	 * @param itemCode　収集項目コード 　
	 * @param deviceIndex　デバイスのインデックス 　
	 * @return 計算された性能データ
	 */
	synchronized public CollectedDataInfo getTempValue(
			final String facilityID, 
			final String itemCode, 
			final int deviceIndex,
			final String deviceName){
		double value = Double.NaN;
		
		Facility facility = (Facility)m_scopeTable.get(facilityID);
		
		if(facility == null){
			facility = (Facility)m_nodeTable.get(facilityID);
		}
		
		if (facility == null){
			// エラー処理
			// 登録されていないファシリティを参照しようとした
			String message= "Facility not found : " + facilityID;
			throw new EJBException(message);
		} else {
			value = facility.getCalcValueBuffer(new CollectorItemPK(itemCode, deviceIndex));
		}
		
		CollectedDataInfo ret = new CollectedDataInfo(facilityID, itemCode, deviceIndex, deviceName);
		if(m_lastCollectTime == 0){
			// 今まで一度も収集されていない場合(前回の収集)
			ret.setDate(new Date());  // 現時刻を設定
			ret.setValue(Double.NaN); // 性能値としてNaNを設定
		} else {
			ret.setDate(new Date(m_lastCollectTime));  // 収集日時を設定
			ret.setValue(value);  // 性能値を設定
		}
		
		return ret;
	}

	/**
	 * 指定のファシリティを返します。
     *
	 * @param facilityID ファシリティID(スコープもしくはノード)
	 * @return
	 */
	public Facility getFacility(final String facilityID){
		Scope scope = (Scope)m_scopeTable.get(facilityID);	
		Node node = (Node)m_nodeTable.get(facilityID);
		
		if (scope != null && node == null){
			return scope;		
		} else if (scope == null && node != null){
			return node;
		} else {
			// エラー処理
			String message= "Facility not found : " + facilityID;
			throw new EJBException(message);
		}
	}
	
	/**
	 * ファシリティツリーの内容を設定します。
	 * 
	 * @param treeItem FacilityTreeItemの階層オブジェクト
	 */
	private void setScopeTree(FacilityTreeItem treeItem) {
		FacilityInfo info = treeItem.getData();
		
		// ファシリティがスコープの場合
		if(info.getType() == FacilityConstant.TYPE_SCOPE){
			this.addScope(info.getFacilityId(), 
					info.getFacilityName(), 
					treeItem.getParent().getData().getFacilityId()
					);
		} else 			
		// ファシリティがノードの場合
		if(info.getType() == FacilityConstant.TYPE_NODE){			
			this.addNode(
					info.getFacilityId(),
					info.getFacilityName(),
					treeItem.getParent().getData().getFacilityId()
					);
		}
		
		// 子ファシリティを求める
		FacilityTreeItem[] tmpItem = treeItem.getChildren();
		
		if(tmpItem.length != 0){  // 子ファシリティに対して再帰的に設定
			for(int i = 0; i < tmpItem.length; i++){
				setScopeTree(tmpItem[i]);
			}
		}
	}

	/**
	 * 指定のファシリティIDのオブジェクトがスコープか否かを判定します
	 * @param facilityID ファシリティID
	 * @return 指定のファシリティIDのオブジェクトがスコープの場合は true を返します
	 */
	public boolean isScope(final String facilityID){
		return m_scopeTable.get(facilityID) != null;
	}
	
	/**
	 * 指定のファシリティIDのオブジェクトがノードか否かを判定します
	 * @param facilityID ファシリティID
	 * @return 指定のファシリティIDのオブジェクトがノードの場合は true を返します
	 */
	public boolean isNode(final String facilityID){
		return m_nodeTable.get(facilityID) != null;
	}
	
	/**
	 * 登録されている全てのスコープを取得します。
	 * @return スコープのファシリティIDの配列
	 */
	public String[] getAllFacilityIdList(){
		// 返却値を格納する配列
		String[] ret = new String[m_nodeTable.size() + m_scopeTable.size()];

		int i=0;
		
		// スコープのファシリティIDを取得
		Iterator itr = m_scopeTable.values().iterator();
		while(itr.hasNext()){
			Scope scope = (Scope)itr.next();
			ret[i] = scope.getFacilityID();
			i++;
		}

		// ノードのファシリティIDを取得
		itr = m_nodeTable.values().iterator();
		while(itr.hasNext()){
			Node node = (Node)itr.next();
			ret[i] = node.getFacilityID();
			i++;
		}
		
		return ret;
	}

	/**
	 *  現在ポーラからこのオブジェクトに取り込んだMIB値の中で
	 *  最後に収集したものの時刻を取得します。
	 *  
	 * @return 最終収集時刻
	 */
	public long getLastCollectTime() {
		return m_lastCollectTime;
	}
}
