/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.dao;

import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.CommunicationException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.util.LdapConnectionManager;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.ejb.entity.FacilityTreeBean;
import com.clustercontrol.repository.ejb.entity.FacilityTreePK;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * FacilityTree(BMP Entity Bean)のDAO実装クラス
 *
 * @version 2.3.0
 * @since 1.0.0
 */
public class FacilityTreeDAOImpl implements FacilityTreeDAO {
	protected static Log m_log = LogFactory.getLog( FacilityTreeDAOImpl.class );
	
	protected static final String OBJECTCLASS = "objectClass";
	public static final String BASE_DN = "ou=facilitytree";
	protected static final String OBJECT_CLASS_FACILITY_TREE = "ccFacilityTree";
	/** LDAP用コンテキスト取得時のリトライ上限 */
	private static int MAX_RETRY = 3;
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#init()
	 */
	public void init() {

	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#load(com.clustercontrol.repository.ejb.FacilityTreePK, com.clustercontrol.repository.ejb.FacilityTreeBean)
	 */
	public void load(FacilityTreePK pk, FacilityTreeBean ejb) throws EJBException {
		m_log.trace("load() : start " + pk.getDn());
		javax.naming.directory.DirContext ctx = null;
		
		int retryCount=0;
		while(MAX_RETRY > retryCount){

			try {
				clearBean(ejb);
				
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				//属性を取得
				javax.naming.directory.Attributes attrs = ctx.getAttributes(pk.getDn());
	
				//dn取得
				ejb.setDn(pk.getDn());
				//ccFacilityId取得
				ejb.setFacilityId((String)attrs.get(FacilityTreeAttributeConstant.FACILITYID).get(0));
				//ccTreeType取得
				ejb.setTreeType(new Integer((String)attrs.get(FacilityTreeAttributeConstant.TREETYPE).get(0)));
				ejb.setBuiltIn((new Boolean((String)attrs.get(FacilityTreeAttributeConstant.BUILTIN).get(0))));
				ejb.setSortValue(new Integer((String)attrs.get(FacilityTreeAttributeConstant.SORT_VALUE).get(0)));
	
				m_log.debug("load() : Dn " + pk.getDn() + " FacilityId " + ejb.getFacilityId() + " TreeType " + ejb.getTreeType());
					
			} catch ( javax.naming.NamingException ex ) {
			    if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
					
				// NamingExceptionが発生した場合は、catchしリトライする
				m_log.debug("load(): catch NamingException" + ex.getMessage());
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} catch (NullPointerException e) {
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("load() : LDAP query result is null");
				//whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
					
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("load() context close failed : " + e.getMessage());
				}
			}
			m_log.trace("load() : end " + pk.getDn());
			return;
		}
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#store(com.clustercontrol.repository.ejb.FacilityTreeBean)
	 */
	public void store(FacilityTreeBean ejb) throws EJBException {

	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#remove(com.clustercontrol.repository.ejb.FacilityTreePK)
	 */
	public void remove(FacilityTreePK pk) throws RemoveException, EJBException {
		m_log.trace("remove() : start " + pk.getDn());
		javax.naming.directory.DirContext ctx = null;
		int retryCount=0;
		while(MAX_RETRY > retryCount){
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				//エントリを削除
				ctx.destroySubcontext(pk.getDn());
				
			} catch ( javax.naming.NamingException ex ) {
			    if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
				// NamingExceptionが発生した場合は、catchしリトライする
				m_log.debug("remove(): catch NamingException" + ex.getMessage());
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
		            AplLogger apllog = new AplLogger("REP", "rep");
		            String[] args = {pk.getDn()};
		            apllog.put("SYS", "008", args);
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} catch (NullPointerException e) {
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("remove() : LDAP query result is null");
	//				whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("remove() context close failed : " + e.getMessage());
				}
			}
			m_log.trace("remove() : end " + pk.getDn());
			return;
		}
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#create(com.clustercontrol.repository.ejb.FacilityTreeBean)
	 */
	public FacilityTreePK create(FacilityTreeBean ejb) throws CreateException, EJBException {
		m_log.trace("createe() : start " + ejb.getDn());
		FacilityTreePK pk = null;
		javax.naming.directory.DirContext ctx = null;
		int retryCount=0;
		while(MAX_RETRY > retryCount){	
			
			try {
				//親のdnを取得
				String baseDn = BASE_DN;
				if(ejb.getParentFacilityId() != null && ejb.getParentFacilityId().compareTo("") != 0){
					try {
						FacilityTreePK parentPk = findScopeByFacilityId(ejb.getParentFacilityId());
						baseDn = parentPk.getDn();
					} catch (FinderException e) {
							m_log.warn("create() : " + e.getMessage());
						throw new EJBException("Facility ID = " + ejb.getParentFacilityId() + " not created.");
					}
				}
	
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				// Attributes for new entry
				javax.naming.directory.Attributes attrs =
					new javax.naming.directory.BasicAttributes();
				
				attrs.put(OBJECTCLASS, OBJECT_CLASS_FACILITY_TREE);
				attrs.put(FacilityTreeAttributeConstant.FACILITYID, ejb.getFacilityId());
				attrs.put(FacilityTreeAttributeConstant.TREETYPE, ejb.getTreeType().toString());
				attrs.put(FacilityTreeAttributeConstant.SORT_VALUE, ejb.getSortValue().toString());
				attrs.put(FacilityTreeAttributeConstant.BUILTIN,"FALSE");
				
				//↑FALSEは小文字falseはNG
				
				
				// dn文字列作成
				StringBuffer dn = new StringBuffer();
				dn.append(FacilityTreeAttributeConstant.FACILITYID);
				dn.append("=");
				dn.append(ejb.getFacilityId());
				dn.append(",");
				dn.append(baseDn);
				
				//dn設定
				ejb.setDn(dn.toString());
	
				//エントリの作成
				ctx.createSubcontext(ejb.getDn(), attrs);
	
				pk = new FacilityTreePK(ejb.getDn());
				
			} catch ( javax.naming.NamingException ex ) {
				// NamingExceptionが発生した場合は、catchしリトライする
				m_log.debug("create(): catch NamingException" + ex.getMessage());
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
		            AplLogger apllog = new AplLogger("REP", "rep");
		            String[] args = {ejb.getDn()};
		            apllog.put("SYS", "007", args);
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
	//		    if(ex instanceof CommunicationException){
	//		        LdapConnectionManager.getConnectionManager().setDirContext(null);
	//		    }
	//		    else if(ex instanceof NameAlreadyBoundException){
	//		    	throw new DuplicateKeyException(ex.getMessage());
	//          }
	//			throw new CreateException(ex.getMessage());
				
				
			} catch (NullPointerException e) {
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("remove() : LDAP query result is null");
	//				whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("create() context close failed : " + e.getMessage());
				}
			}
			return pk;
		}
		return pk;
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		return findAll(null, null);
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findAll(java.lang.String)
	 */
	public Collection findAll(String facilityId) throws FinderException {
		return findAll(facilityId, null);
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findAllNode(java.lang.String)
	 */
	public Collection findAllNode(String facilityId) throws FinderException {
		return findAll(facilityId, new Integer(FacilityConstant.TYPE_NODE));
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findAllScope(java.lang.String)
	 */
	public Collection findAllScope(String facilityId) throws FinderException {
		return findAll(facilityId, new Integer(FacilityConstant.TYPE_SCOPE));
	}

	/**
	 * ファシリティツリー配下検索
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param type 検索対象（ノードまたはスコープ）
	 * @return
	 * @throws FinderException
	 */
	public Collection findAll(String facilityId, Integer type) throws FinderException {
		ArrayList ret = new ArrayList();

		m_log.debug("findAll() : FacilityId = " + facilityId + ", Type = " + type);
		
		// dn文字列作成
		StringBuffer baseDn = new StringBuffer();
		if(facilityId == null || facilityId.compareTo("") == 0){
			baseDn.append(BASE_DN);
		}
		else{
			try {
				FacilityTreePK pk = findScopeByFacilityId(facilityId);
				baseDn.append(pk.getDn());
			} catch (FinderException e) {
				
				m_log.debug("findAllScope() : No Search Result : FacilityId = " + facilityId + ", Type = " + type);
				m_log.debug("findAllScope() :" + e.getMessage());
				return ret;
			}
		}
		
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// BaseDN以下にぶら下がるすべてのレベルを検索の対象
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_FACILITY_TREE);
		if(type != null && type.intValue() == FacilityConstant.TYPE_NODE){
			search.append(")(");
			search.append(FacilityTreeAttributeConstant.TREETYPE);
			search.append("=");
			search.append(FacilityConstant.TYPE_NODE);
		}
		else if(type != null && type.intValue() == FacilityConstant.TYPE_SCOPE){
			search.append(")(");
			search.append(FacilityTreeAttributeConstant.TREETYPE);
			search.append("=");
			search.append(FacilityConstant.TYPE_SCOPE);
		}
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		
		int retryCount=0;	
		while(MAX_RETRY > retryCount){
			
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				results = ctx.search(baseDn.toString(), search.toString(), constraints);
				
				if (results == null ) {
					m_log.debug("findAll() : No Search Result : FacilityId = " + facilityId + ", Type = " + type);
					return ret;
				}
	
				//検索結果を取得
				while ( results.hasMoreElements() ) {
					javax.naming.directory.SearchResult aResult = 
						(javax.naming.directory.SearchResult)results.nextElement();
					FacilityTreePK pk = null;
					if(aResult.getName().compareTo("") == 0){
					    pk = new FacilityTreePK(baseDn.toString());
					}
					else{
					    pk = new FacilityTreePK(aResult.getName() + "," + baseDn.toString());
					}
					ret.add(pk);
				}
			} catch (NamingException ex) {
				if(ex instanceof CommunicationException){
					LdapConnectionManager.getConnectionManager().setDirContext(null);
				}
				m_log.debug("findAllScope() : " + ex.getMessage());
				
	//				中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} catch (NullPointerException e){
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findAllScope() : LDAP query result is null");
	//				中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
			}
			finally{
				try {
					if(results != null)
						results.close();
					if(ctx != null)
						ctx.close();
				} catch (NamingException e) {
					m_log.error("findAll() context close failed : " + e.getMessage());
				}
			}
			
			return ret;
		}
		return ret;
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findByPrimaryKey(com.clustercontrol.repository.ejb.FacilityTreePK)
	 */
	public FacilityTreePK findByPrimaryKey(FacilityTreePK pk) throws FinderException {
		javax.naming.directory.DirContext ctx = null;
		
		int retryCount=0;	
		while(MAX_RETRY > retryCount){
			try {
			    m_log.debug("findByPrimaryKey() : dn = " + pk.getDn());
			    
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				//属性取得
				javax.naming.directory.Attributes attrs = ctx.getAttributes(pk.getDn());
				
				if (attrs != null) {
					return pk;
				}
				else{
					throw new FinderException("FacilityTreeDAOImpl.findByPrimaryKey() : dn=" + pk.getDn() + " not found.");
				}
				
			} catch ( javax.naming.NamingException ex ) {
			    if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
				m_log.debug("findByPrimaryKey() : " + ex.getMessage());				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
			} catch (NullPointerException ex) {
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findByPrimaryKey() : LDAP query result is null");
				
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("findByPrimaryKey() context close failed : " + e.getMessage());
				}
			}
		}
		return pk;
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findChildren(com.clustercontrol.repository.ejb.FacilityTreePK)
	 */
	public Collection findChildren(FacilityTreePK pk) throws FinderException {
		ArrayList ret = new ArrayList();
		
		m_log.debug("findChildren() : dn = " + pk.getDn());

		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// BaseDN直下の1レベルのみを検索の対象
		constraints.setSearchScope(javax.naming.directory.SearchControls.ONELEVEL_SCOPE);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_FACILITY_TREE);
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		
		int retryCount=0;
		
		while(MAX_RETRY > retryCount){
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				results = ctx.search(pk.getDn(), search.toString(), constraints);
				
				if (results == null ) {
					m_log.warn("findChildren() : No Search Result : dn = " + pk.getDn());
					return ret;
				}
	
				//検索結果を取得
				while ( results.hasMoreElements() ) {
					javax.naming.directory.SearchResult aResult = 
						(javax.naming.directory.SearchResult)results.nextElement();
					FacilityTreePK p = new FacilityTreePK(aResult.getName() + "," + pk.getDn());
					ret.add(p);
				}
			} catch (NamingException ex) {
				if(ex instanceof CommunicationException){
					LdapConnectionManager.getConnectionManager().setDirContext(null);
				}
				m_log.debug("findChildren() : " + ex.getMessage());
	//				中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
			} catch (NullPointerException ex) {
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findChildren() : LDAP query result is null");
	//				中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(results != null)
						results.close();
					if(ctx != null)
						ctx.close();
				} catch (NamingException e) {
					m_log.error("findChildren() context close failed : " + e.getMessage());
				}
			}
			return ret;
		}
		return ret;
	}
	/**
	 * ファシリティID検索（スコープのみ）
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId
	 * @return FacilityTreePK
	 */
	public FacilityTreePK findScopeByFacilityId(String facilityId) throws FinderException {
		m_log.debug("findScopeByFacilityId() : FacilityId = " + facilityId);
	
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// 名前付きオブジェクトを検索
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);
		
		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_FACILITY_TREE);
		search.append(")(");
		search.append(FacilityTreeAttributeConstant.FACILITYID);
		search.append("=");
		search.append(facilityId);
		search.append(")(");
		search.append(FacilityTreeAttributeConstant.TREETYPE);
		search.append("=");
		search.append(FacilityConstant.TYPE_SCOPE);
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		FacilityTreePK pk = null;
		
		int retryCount=0;
		
		while(MAX_RETRY > retryCount){
			
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				results = ctx.search(BASE_DN, search.toString(), constraints);
				
				if (results == null) {
					throw new FinderException("FacilityTreeDAOImpl.findScopeByFacilityId() : FacilityId=" + facilityId + " not found.");
				}
				
				javax.naming.directory.SearchResult aResult = null;
				javax.naming.directory.Attributes attrs = null;
				while ( results.hasMoreElements() ) {
					aResult = (javax.naming.directory.SearchResult)results.nextElement();
					attrs = aResult.getAttributes();
					break;
				}
				
				if (attrs == null) {
					throw new FinderException("FacilityTreeDAOImpl.findScopeByFacilityId() : FacilityId=" + facilityId + " not found.");
				}
	
				pk = new FacilityTreePK(aResult.getName() + "," + BASE_DN);
				
			} catch (NamingException ex) {
				if(ex instanceof CommunicationException){
					LdapConnectionManager.getConnectionManager().setDirContext(null);
				}
	
				m_log.debug("findScopeByFacilityId() : " + ex.getMessage());
				
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
	
	//			2.1.1改修分マージ時にコメントアウト 7/4 kozakai
	//		    m_log.debug("findScopeByFacilityId() : " + e.getMessage());
	//			throw new FinderException("FacilityTreeDAOImpl.findScopeByFacilityId() : FacilityId=" + facilityId + " not found.");
	
			} catch (NullPointerException ex) {
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findScopeByFacilityId() : LDAP query result is null");
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
	
			} finally {
				try {
					if(results != null)
						results.close();
					if(ctx != null)
						ctx.close();
				} catch (NamingException e) {
					m_log.error("findScopeByFacilityId() context close failed : " + e.getMessage());
				}
			}
	
			return pk;
		}
		return pk;
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findNodeByFacilityId(java.lang.String)
	 */
	public Collection findByFacilityId(String facilityId) throws FinderException {
		ArrayList ret = new ArrayList();
		
		m_log.debug("findByFacilityId() : FacilityId = " + facilityId);

		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// BaseDN以下にぶら下がるすべてのレベルを検索の対象
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_FACILITY_TREE);
		search.append(")(");
		search.append(FacilityTreeAttributeConstant.FACILITYID);
		search.append("=");
		search.append(facilityId);
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		
		int retryCount=0;
		
		while(MAX_RETRY > retryCount){
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				results = ctx.search(BASE_DN, search.toString(), constraints);
				
				if (results == null ) {
					m_log.warn("findByFacilityId() : No Search Result : FacilityId = " + facilityId);
					return ret;
				}
	
				//検索結果を取得
				while ( results.hasMoreElements() ) {
					javax.naming.directory.SearchResult aResult = 
						(javax.naming.directory.SearchResult)results.nextElement();
					FacilityTreePK pk = new FacilityTreePK(aResult.getName() + "," + BASE_DN);
					ret.add(pk);
				}
				
			} catch (NamingException ex) {
				if(ex instanceof CommunicationException){
					LdapConnectionManager.getConnectionManager().setDirContext(null);
				}
				m_log.debug("findByFacilityId() : " + ex.getMessage());
				//				中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
			} catch (NullPointerException ex){
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findByFacilityId() : LDAP query result is null");
//				中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
			} finally {
				try {
					if(results != null)
						results.close();
					if(ctx != null)
						ctx.close();
				} catch (NamingException e) {
					m_log.error("findByFacilityId() context close failed : " + e.getMessage());
				}
			}

			return ret;
		}
		//MAX_RETRYの回数の内にLDAPアクセスが成功しない場合
		LdapConnectionManager.getConnectionManager().setDirContext(null);
		throw new FinderException("findByFacilityId() : LDAP query result is null for many times");
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findOneLevel()
	 */
	public Collection findOneLevel() throws FinderException {
		return findOneLevel(null, null);
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findOneLevel()
	 */
	public Collection findOneLevel(String facilityId) throws FinderException {
		return findOneLevel(facilityId, null);
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findOneLevelNode(java.lang.String)
	 */
	public Collection findOneLevelNode(String facilityId) throws FinderException {
		return findOneLevel(facilityId, new Integer(FacilityConstant.TYPE_NODE));
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FacilityTreeDAO#findOneLevelScope(java.lang.String)
	 */
	public Collection findOneLevelScope(String facilityId) throws FinderException {
		return findOneLevel(facilityId, new Integer(FacilityConstant.TYPE_SCOPE));
	}
	
	/**
	 * ファシリティツリー直下検索
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param type 検索対象（ノードまたはスコープ）
	 * @return
	 * @throws FinderException
	 */
	public Collection findOneLevel(String facilityId, Integer type) throws FinderException {
		ArrayList ret = new ArrayList();
		
		m_log.debug("findOneLevel() : FacilityId = " + facilityId + ", Type = " + type);

		// dn文字列作成
		StringBuffer baseDn = new StringBuffer();
		if(facilityId == null || facilityId.compareTo("") == 0){
			baseDn.append(BASE_DN);
		}
		else{
			try {
				FacilityTreePK pk = findScopeByFacilityId(facilityId);
				baseDn.append(pk.getDn());
			} catch (FinderException e) {
				m_log.warn("findOneLevel() : No Search Result : FacilityId = " + facilityId + ", Type = " + type);
				return ret;
			}
		}
		
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// BaseDN直下の1レベルのみを検索の対象
		constraints.setSearchScope(javax.naming.directory.SearchControls.ONELEVEL_SCOPE);
		
		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_FACILITY_TREE);
		if(type != null && type.intValue() == FacilityConstant.TYPE_NODE){
			search.append(")(");
			search.append(FacilityTreeAttributeConstant.TREETYPE);
			search.append("=");
			search.append(FacilityConstant.TYPE_NODE);
		}
		else if(type != null && type.intValue() == FacilityConstant.TYPE_SCOPE){
			search.append(")(");
			search.append(FacilityTreeAttributeConstant.TREETYPE);
			search.append("=");
			search.append(FacilityConstant.TYPE_SCOPE);
		}
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		int retryCount=0;
		
		while(MAX_RETRY > retryCount){
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				results = ctx.search(baseDn.toString(), search.toString(), constraints);
				
				if (results == null ) {
					m_log.warn("findOneLevel() : No Search Result : FacilityId = " + facilityId + ", Type = " + type);
					return ret;
				}
	
				//検索結果を取得
				while ( results.hasMoreElements() ) {
					javax.naming.directory.SearchResult aResult = 
						(javax.naming.directory.SearchResult)results.nextElement();
					FacilityTreePK pk = new FacilityTreePK(aResult.getName() + "," + baseDn.toString());
					ret.add(pk);
				}
				
			} catch (NamingException ex) {
				if(ex instanceof CommunicationException){
					LdapConnectionManager.getConnectionManager().setDirContext(null);
				}
				m_log.debug("findOneLevel() : " + ex.getMessage());
	//				中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} catch (NullPointerException ex){
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findOneLevel() : LDAP query result is null");
	//				中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}	
			} finally {
				try {
					if(results != null)
						results.close();
					if(ctx != null)
						ctx.close();
				} catch (NamingException e) {
					m_log.error("findOneLevel() context close failed : " + e.getMessage());
				}
			}
	
			return ret;
		}
		return ret;
	}
	/**
	 * Beanクリア処理
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param ejb
	 */
	protected void clearBean(FacilityTreeBean ejb){
	    ejb.setDn(null);
	    ejb.setFacilityId(null);
	    ejb.setParentFacilityId(null);
	    ejb.setTreeType(null);
	}
}
