/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmp.dialog;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.snmp.action.DeleteSnmp;
import com.clustercontrol.snmp.action.GetSnmp;
import com.clustercontrol.snmp.action.ModifySnmp;
import com.clustercontrol.snmp.bean.SnmpTabelDefine;
import com.clustercontrol.snmp.composite.SnmpListComposite;
import com.clustercontrol.snmp.composite.SnmpListComposite.SelectItem;
import com.clustercontrol.util.Messages;

/**
 * SNMP監視一覧ダイアログクラスです。
 * 
 * @version 2.2.0
 * @since 2.0.0
 */
public class SnmpListDialog extends CommonDialog {

    // ----- instance フィールド ----- //

    /** SNMP一覧 */
    private SnmpListComposite snmpList = null;

    /** 追加ボタン（数値） */
    private Button buttonAddNumeric = null;
    
    /** 追加ボタン（文字列） */
    private Button buttonAddString = null;

    /** 変更ボタン */
    private Button buttonModify = null;

    /** 削除ボタン */
    private Button buttonDelete = null;

    /** 有効ボタン */
    private Button buttonValid = null;
    
    /** 無効ボタン */
    private Button buttonInvalid = null;

    // ----- コンストラクタ ----- //

    /**
     * ダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     */
    public SnmpListDialog(Shell parent) {
        super(parent);
        setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
    }

    // ----- instance メソッド ----- //

    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     */
    protected Point getInitialSize() {
        return new Point(800, 400);
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent
     *            親のインスタンス
     */
    protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.snmp.list"));

        // レイアウト
        GridLayout layout = new GridLayout(7, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 7;
        parent.setLayout(layout);

        /*
         * SNMP監視一覧
         */

        this.snmpList = new SnmpListComposite(parent, SWT.BORDER);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 6;
        this.snmpList.setLayoutData(gridData);

        /*
         * 操作ボタン
         */

        Composite composite = new Composite(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.numColumns = 1;
        composite.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        composite.setLayoutData(gridData);

        // 追加ボタン（数値）
        this.buttonAddNumeric = this.createButton(composite, Messages.getString("add.numeric"));
//        this.buttonAddNumeric = this.createButton(composite, Messages.getString("add"));
        this.buttonAddNumeric.addSelectionListener(new AddSelectionListener(MonitorTypeConstant.TYPE_NUMERIC));
        
        // 追加ボタン（文字列）
//        Label dummy = new Label(composite, SWT.NONE);
        this.buttonAddString = this.createButton(composite, Messages.getString("add.string"));
        this.buttonAddString.addSelectionListener(new AddSelectionListener(MonitorTypeConstant.TYPE_STRING));

        // 変更ボタン
        @SuppressWarnings("unused") Label dummy = new Label(composite, SWT.NONE);
        this.buttonModify = this.createButton(composite, Messages.getString("modify"));
        this.buttonModify.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	
            	String monitorId = null;
            	int monitorType = Integer.MAX_VALUE;
            	
            	Table table = snmpList.getTableViewer().getTable();
            	TableItem[] item = table.getSelection();
            	
            	if(item != null && item.length>0){
            		try{
            			monitorId = (String)((ArrayList)item[0].getData()).get(SnmpTabelDefine.MONITOR_ID);
            			monitorType = MonitorTypeConstant.stringToType((String)((ArrayList)item[0].getData()).get(SnmpTabelDefine.MONITOR_TYPE));
            		}
            		catch(Exception ex){
            		}
            	}

                if (monitorId != null) {
                	SnmpCreateDialog dialog = new SnmpCreateDialog(getParentShell(), monitorId, monitorType);
                	if (dialog.open() == IDialogConstants.OK_ID) {
                		int selectIndex = table.getSelectionIndex();
                		snmpList.update();
                		table.setSelection(selectIndex);
                    }
                }
                else{
                    MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.monitor.1"));
                }
            }
        });

        // 削除ボタン
        dummy = new Label(composite, SWT.NONE);
        this.buttonDelete = this.createButton(composite, Messages
                .getString("delete"));
        this.buttonDelete.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	
            	String monitorId = null;
            	int monitorType = Integer.MAX_VALUE;
            	
            	Table table = snmpList.getTableViewer().getTable();
            	TableItem[] item = table.getSelection();
            	
            	if(item != null && item.length>0){
            		try{
            			monitorId = (String)((ArrayList)item[0].getData()).get(SnmpTabelDefine.MONITOR_ID);
            			monitorType = MonitorTypeConstant.stringToType((String)((ArrayList)item[0].getData()).get(SnmpTabelDefine.MONITOR_TYPE));
            		}
            		catch(Exception ex){
            		}
            	}

            	if (monitorId != null) {
            		
            		String[] args = { monitorId };
            		if (MessageDialog.openConfirm(
            				null, 
            				Messages.getString("confirmed"), 
            				Messages.getString("message.monitor.39", args))) {
            			
            			boolean result = new DeleteSnmp().delete(monitorId, monitorType);
            			if(result){
            				MessageDialog.openInformation(
            						null, 
            						Messages.getString("successful"), 
            						Messages.getString("message.monitor.37", args));
            				
            				snmpList.update();
            			}
            			else{
            				MessageDialog.openError(
            						null, 
            						Messages.getString("failed"), 
            						Messages.getString("message.monitor.38", args));
            			}
            		}
            	}
            	else{
            		MessageDialog.openWarning(
            				null, 
            				Messages.getString("warning"), 
            				Messages.getString("message.monitor.1"));
            	}
            }
        });
        
        // 有効ボタン
        dummy = new Label(composite, SWT.NONE);
        dummy = new Label(composite, SWT.NONE);
        this.buttonValid = this.createButton(composite, Messages
                .getString("valid"));
        this.buttonValid.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	//一括で有効に変更
            	setValid(YesNoConstant.TYPE_YES);
            }
        });
        
        // 無効ボタン
        this.buttonInvalid = this.createButton(composite, Messages
                .getString("invalid"));
        this.buttonInvalid.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	//一括で無効に変更
            	setValid(YesNoConstant.TYPE_NO);
            }
        });

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 8;
        line.setLayoutData(gridData);

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    protected ValidateResult validate() {
        return super.validate();
    }

    /**
     * 共通のボタンを生成します。
     * 
     * @param parent
     *            親のコンポジット
     * @param label
     *            ボタンのラベル
     * @return 生成されたボタン
     */
    private Button createButton(Composite parent, String label) {
        Button button = new Button(parent, SWT.NONE);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);

        button.setText(label);

        return button;
    }
    
    /**
     * 閉じるボタンを作成します。
     * 
     * @param parent ボタンバーコンポジット
     */
    protected void createButtonsForButtonBar(Composite parent) {
    	
        // 閉じるボタン
        this.createButton(parent, IDialogConstants.CANCEL_ID, Messages.getString("close"), false);
    }
    
    /**
     * 追加ボタンSelectionAdapterクラス
     * 
     * @version 2.1.0
     * @since 2.1.0
     */
    class AddSelectionListener extends SelectionAdapter {
    	
    	/** 監視判定タイプ */
    	private int m_monitorType;
    	
    	/**
    	 * コンストラクタ
    	 * @param monitorType 監視判定タイプ
    	 */
    	public AddSelectionListener(int monitorType) {
    		m_monitorType = monitorType;
    	}
    	
    	/* (非 Javadoc)
    	 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
    	 */
    	@Override
    	public void widgetSelected(SelectionEvent e) {
    		
    		SnmpCreateDialog dialog = new SnmpCreateDialog(getParentShell(), m_monitorType);
    		if (dialog.open() == IDialogConstants.OK_ID) {
    			snmpList.update();
    		}
    	}
    }

    /**
     * 有効・無効変更処理
     * 
     * @param valid
     */
    public void setValid(int valid) {

		//選択された監視項目IDを取得
    	ArrayList<SelectItem> list = snmpList.getSelectionData();
		if (list != null && list.size() > 0) {

			StringBuilder monitorIds = new StringBuilder();
			for(int i = 0; i < list.size(); i++){
				if(i > 0){
					monitorIds.append(", ");
				}
				SelectItem item = list.get(i);
				monitorIds.append(item.getMonitorId());
			}

            String[] confirmArgs = { monitorIds.toString() };
            String message;
            if(valid == YesNoConstant.TYPE_YES)
            	message = Messages.getString("message.monitor.47",confirmArgs);
            else
            	message = Messages.getString("message.monitor.48",confirmArgs);
            if (!MessageDialog.openConfirm(
        			null, 
        			Messages.getString("confirmed"), 
        			message)) {
            	return;
            }
			
			for(int i = 0; i < list.size(); i++){
				SelectItem item = list.get(i);
				
				if(item != null && !item.getMonitorId().equals("")){
			        //監視情報を取得
			        MonitorInfo info = new GetSnmp().getSnmp(item.getMonitorId(), item.getMonitorType());        	
			        
			        //有効・無効を設定
			        info.setValid(valid);
			        
			        //監視情報を更新
			        boolean result = new ModifySnmp().modify(info);
                    if(!result){
                        String[] args = { item.getMonitorId() };
                    	MessageDialog.openError(
                    			null, 
                    			Messages.getString("failed"), 
                    			Messages.getString("message.monitor.36", args));
                    }
				}
			}
    			
            int selectIndex = snmpList.getTableViewer().getTable().getSelectionIndex();
            snmpList.update();
            snmpList.getTableViewer().getTable().setSelection(selectIndex);
		}
		else{
        	MessageDialog.openWarning(
        			null, 
        			Messages.getString("warning"), 
        			Messages.getString("message.monitor.1"));
        }
    }
}