/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmptrap.factory;

import java.util.Collection;
import java.util.Iterator;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.jms.JMSException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.bean.MonitorConstant;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoPK;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.monitor.run.ejb.entity.MonitorNumericValueInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorNumericValueInfoUtil;
import com.clustercontrol.snmptrap.bean.TopicConstant;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapInfoLocal;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapInfoPK;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapInfoUtil;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapOidInfoLocal;
import com.clustercontrol.snmptrap.message.UpdateSnmpTrapInfo;
import com.clustercontrol.snmptrap.util.SendTopic;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 監視情報削除クラス
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class DeleteMonitorInfo {
	
	protected static Log m_log = LogFactory.getLog( DeleteMonitorInfo.class );
	
	/**
	 * 監視情報を削除
	 * 
	 * @param info 監視情報
	 * @param user ユーザ
	 * @return 削除に成功した場合、true
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws InvalidTransactionException
	 * @throws NamingException
	 * @throws FinderException 
	 * @throws RemoveException 
	 */
	public boolean delete(String monitorId) throws NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, InvalidTransactionException, NamingException, FinderException, RemoveException {
		
    	//アクセスロック
    	AccessLock.lock(AccessLock.MONITOR_SNMPTRAP);
		
		TransactionManager tm = null;
		Transaction tx = null;
		Transaction oldTx = null;
		boolean result = false;
		
		try 
		{
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			
			// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
			if(tm.getTransaction() != null){
				oldTx = tm.suspend();
			}
			
			try{
				// トランザクション開始
				if(tm.getTransaction() == null){
					tm.begin();	
					tx = tm.getTransaction();
				}
				
				// 監視情報を削除
				result = deleteMonitorInfo(monitorId);
				if(result){
					// コミット
					tx.commit();
				}
				
				try {
					// 更新通知
					UpdateSnmpTrapInfo updateInfo = new UpdateSnmpTrapInfo(
							UpdateSnmpTrapInfo.TYPE_DELETE,
							monitorId);
					
					SendTopic sendTopic = new SendTopic(TopicConstant.TOPIC_NAME_UPDATE);
					sendTopic.put(updateInfo);
					
				} catch (NamingException e) {
					m_log.error("delete() : 送信エラー。" + e.getMessage() );
				} catch (JMSException e) {
					m_log.error("delete() : 送信エラー。" + e.getMessage() );
				}
			
			} catch(NotSupportedException e){
				outputLog(e, "delete()", monitorId);
				throw e;
			} catch (RollbackException e) {
				outputLog(e, "delete()", monitorId);
				throw e;
			} catch (HeuristicMixedException e) {
				outputLog(e, "delete()", monitorId);
				throw e;
			} catch(HeuristicRollbackException e){
				outputLog(e, "delete()", monitorId);
				throw e;
			} catch(SystemException e){
				outputLog(e, "delete()", monitorId);
				throw e;
			}
			finally{
				// トランザクション関連の例外が発生した場合は、ロールバック
				if(tx != null){
					if(!result){
						tx.rollback();
					}
					tx = null;
				}
			}
		}
		finally{
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
				
				} catch(InvalidTransactionException e){
					outputLog(e, "delete()", monitorId);
					throw e;
				} catch(IllegalStateException e){
					outputLog(e, "delete()", monitorId);
					throw e;
				} catch(SystemException e){
					outputLog(e, "delete()", monitorId);
					throw e;
				}
			}
		}
		return result;
	}

	/**
	 * 監視情報を削除
	 * 
	 * @param user ユーザ
	 * @return 削除に成功した場合、true
	 * @throws NamingException
	 * @throws FinderException 
	 * @throws RemoveException 
	 */
	public boolean deleteMonitorInfo(String monitorId) throws NamingException, FinderException, RemoveException {

		boolean result = false;
		
		try 
		{
			// 監視情報を取得
			SnmpTrapInfoLocal info = 
				SnmpTrapInfoUtil.getLocalHome().findByPrimaryKey(
					new SnmpTrapInfoPK(monitorId));
			
			// 監視OID情報を削除
			Collection ct = info.getSnmpTrapOidInfo();
		    if(ct != null && ct.size() > 0){
		        Object[] infoArray = ct.toArray();
		        for(int j = 0; j < infoArray.length; j++){
		        	SnmpTrapOidInfoLocal oidInfo = (SnmpTrapOidInfoLocal)infoArray[j];
		            if(oidInfo != null){
		            	oidInfo.remove();
		            }
		        }
		    }	
		    
		    // ジョブ連携のため追加
			// 重要度ごとに実行するジョブの情報を保持するテーブルが必要だが、
			// SNMPトラップ監視機能の設定情報を保持するテーブル cc_snmp_trap_info には、
			// ジョブ連携用の情報を保持するカラムがないため、cc_monitor_infoテーブルと
			// cc_monitor_numeric_value_infoテーブルに情報を保持しているが、これも削除する
			// cc_monitor_numeric_value_infoのレコードを削除
		    try{
		    	Collection mnvis = MonitorNumericValueInfoUtil.getLocalHome().
		    	findByMonitorTypeIdAndMonitorId(MonitorConstant.STRING_SNMPTRAP, info.getMonitorId());
		    	Iterator itr = mnvis.iterator();
		    	MonitorNumericValueInfoLocal local = null;
		    	while(itr.hasNext()){
		    		local = (MonitorNumericValueInfoLocal)itr.next();
		    		local.remove();
		    	}
		    } catch (FinderException e) {
				// ジョブ連携の定義がDBに登録されていない場合は、ジョブ連携情報を付加しない
				// ver2.2.1まではSNMPトラップ監視機能でのジョブ連携が実装されておらず、
				// SNMPトラップ監視機能で、cc_monitor_infoとcc_monitor_numeric_value_infoテーブルを
				// 利用していないため、ver2.2.1から移行されたDBでは、このルートを通る
				m_log.warn("deleteMonitorInfo():" + " MonitorNumericValueInfo not found '" + monitorId + "', " + e.getMessage());
			}

	    	// cc_monitor_infoのレコードを削除
		    try{
		    	MonitorInfoPK mipk = new MonitorInfoPK(info.getMonitorId(), MonitorConstant.STRING_SNMPTRAP);
		    	MonitorInfoLocal infoLocal = MonitorInfoUtil.getLocalHome().findByPrimaryKey(mipk);
		    	infoLocal.remove();
		    } catch (FinderException e) {
				// ジョブ連携の定義がDBに登録されていない場合は、ジョブ連携情報を付加しない
				// ver2.2.1まではSNMPトラップ監視機能でのジョブ連携が実装されておらず、
				// SNMPトラップ監視機能で、cc_monitor_infoとcc_monitor_numeric_value_infoテーブルを
				// 利用していないため、ver2.2.1から移行されたDBでは、このルートを通る
				m_log.warn("deleteMonitorInfo():" + " MonitorInfo not found '" + monitorId + "', " + e.getMessage());
			}
		    //
		    
		    // 監視情報を削除
		    info.remove();

			result = true;
			
		} catch (NamingException e) {
			outputLog(e, "deleteMonitorInfo()", monitorId);
			throw e;
		} catch (FinderException e) {
			outputLog(e, "deleteMonitorInfo()", monitorId);
			throw e;
		} catch (EJBException e) {
			outputLog(e, "deleteMonitorInfo()", monitorId);
			throw e;
		} catch (RemoveException e) {
			outputLog(e, "deleteMonitorInfo()", monitorId);
			throw e;
		}
		
		return result;
	}
	
	/**
     * アプリケーションログにログを出力
     * 
     * @param e 例外
     */
	private void outputLog(Exception e, String method, String monitorId){
		AplLogger apllog = new AplLogger("TRAP", "trap");
		String[] args = {monitorId};
        apllog.put("SYS", "003", args);
        m_log.debug(method + ":" + e.getMessage());
	}
}
