/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.syslogng.forward;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.TreeMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * ファイルの入出力を行うクラスです。<BR>
 * syslogメッセージ および ログ出力メッセージオブジェクトのファイルへの入出力を行います。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class FileUtil {
	
	/** ファイルのパス。 */
	private String m_filePath = ".";
	/** ファイルの拡張子。 */
	private String m_fileSuffix = ".txt";
	
	
	/** ログ出力のインスタンス。 */
	static Log log = LogFactory.getLog(FileUtil.class);
	
	/** ファイル名の日付部分フォーマット。 */
	private SimpleDateFormat m_dtFmt = new SimpleDateFormat("yyyyMMddHHmmss");
	/** ファイル名のインデックス部分フォーマット。 */
	private DecimalFormat m_numFmt = new DecimalFormat("-000");
	
	/** 指定のパスのディレクトリでのファイル数の上限を保持　*/
	private int m_maxFileCount = -1;
	
	/**
	 * コンストラクタ。
	 * 
	 * @param Path ファイルの親パス
	 * @param Suffix ファイルの拡張子
	 */
	public FileUtil(String Path, String Suffix, int maxFileCount) {
		
		m_filePath = Path;
		m_fileSuffix = Suffix;
		m_maxFileCount = maxFileCount;
		
	}
	
	/**
	 * メッセージをファイルに出力します。
	 * 
	 * @param msg メッセージオブジェクト
	 */
	public void write(Object msg) {
		// ファイルの上限が0以下に設定されている場合は、ファイル出力しない
		if(m_maxFileCount == 0){
			return;
		}
		
		// ファイルの上限がマイナスの値に設定されている場合は、チェックを実行しない
		if (m_maxFileCount > 0) {
			// ファイルの上限に達していないかをチェックする
			File dir = new File(m_filePath);
			File[] files = dir.listFiles();
			if (files.length >= m_maxFileCount) {
				// ファイル上限を超えている場合は出力しない
				return;
			}
		}
		
		//ファイル名決定
		File fi = getFileName();
		
		// FileOutputStreamクラスの参照型変数。
		// ここで作っておくのはfinallyでclose()メソッドを呼ぶためです。
		FileOutputStream fileOutputStream = null;
		try {
			
			// ファイル出力用のストリームです。
			fileOutputStream = new FileOutputStream(fi);
			
			// 書き込むクラスObjectOutputStreamクラスを作ります。
			ObjectOutputStream objectOutputStream = new ObjectOutputStream(
					fileOutputStream);
			
			// データを書き込みます。
			objectOutputStream.writeObject(msg);
			objectOutputStream.flush();
			
		} catch (IOException e) {
			log.error(msg, e);
		} finally {
			try {
				if (fileOutputStream != null) {
					fileOutputStream.close();
				}
			} catch (Exception e) {
			}
		}
	}
	
	/**
	 * メッセージをファイルに出力します。
	 * 
	 * @param msg 文字列メッセージ
	 */
	public void write(String msg) {
		// ファイルの上限が0に設定されている場合は、ファイル出力しない
		if(m_maxFileCount == 0){
			return;
		}

		// ファイルの上限がマイナスの値に設定されている場合は、チェックを実行しない
		if(m_maxFileCount > 0){
			//ファイルの上限に達していないかをチェックする
			File dir = new File(m_filePath);
			File[] files = dir.listFiles();
			if(files.length >= m_maxFileCount){
				// ファイル上限を超えている場合は出力しない
				return;
			}
		}
		
		//ファイル名決定
		File fi = getFileName();
		
		BufferedWriter writer = null;
		
		//ファイル出力
		try {
			
			writer = new BufferedWriter(new FileWriter(fi));
			
			writer.write(msg, 0, msg.length());
			
		} catch (IOException e) {
			log.error(msg, e);
		} finally {
			try {
				if (writer != null) {
					writer.flush();
					writer.close();
				}
			} catch (Exception e) {
			}
		}
	}
	
	/**
	 * ファイル名を返します。<BR>
	 * 「ファイルパス ＋ "yyyyMMddHHmmss-000" ＋ ファイル拡張子」のファイルインスタンスを生成します。
	 * 
	 * @return ファイル名
	 */
	private File getFileName() {
		File fi = null;
		String filenameDate = m_dtFmt.format(new Date());
		StringBuffer fileNameFull = new StringBuffer(filenameDate);
		int i = 0;
		while (true) {
			fileNameFull.append(m_numFmt.format(i++));
			fileNameFull.append(m_fileSuffix);
			fi = new File(m_filePath, fileNameFull.toString());
			if (!fi.exists())
				break;
			
			fileNameFull.delete(filenameDate.length(), fileNameFull.length());
			
		}
		return fi;
	}
	
	/**
	 * 引数で指定されたテキストファイルから、メッセージを読み込み返します。
	 * 
	 * @param file テキストファイル
	 * @return テキストメッセージ
	 */
	public String readTxtFile(File file) {
		
		//再送処理
		StringBuffer msg = new StringBuffer();
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(new FileReader(file));
			String line;
			while ((line = reader.readLine()) != null) {
				msg.append(line);
			}
		} catch (Exception e) {
			log.error(e);
			return null;
		} finally {
			try {
				if (reader != null) {
					reader.close();
				}
			} catch (Exception e) {
			}
		}
		return msg.toString();
	}
	
	/**
	 * 引数で指定されたオブジェクトファイルから、メッセージを読み込み返します。
	 * 
	 * @param file オブジェクトファイル
	 * @return オブジェクトメッセージ
	 */
	public Object readMsgFile(File file) {
		
		Object ret = null;
		FileInputStream fileInputStream = null;
		try {
			// ファイル入力用のストリームです。
			fileInputStream = new FileInputStream(file);
			
			// 読み込むクラスObjectInputStreamクラスを作ります。
			ObjectInputStream objectInputStream = new ObjectInputStream(
					fileInputStream);
			
			// データを取り出します。
			ret = objectInputStream.readObject();
			
		} catch (IOException e) {
			log.error(e);
		} catch (ClassNotFoundException e) {
			log.error(e);
		} finally {
			try {
				if (fileInputStream != null) {
					fileInputStream.close();
				}
			} catch (IOException e) {
			}
		}
		return ret;
		
	}
	
	
	/**
	 * ファイル一覧を返します。<BR>
	 * 
	 * コンストラクタで指定されたパスの配下で、指定された拡張子のファイルの一覧を返します。
	 * 一覧はファイル名でソートします。
	 * 
	 * @return ファイル一覧（{@link java.io.File}のリスト）
	 */
	public Collection getFileList() {
		
		//ファイル一覧取得
		File dir = new File(m_filePath);
		File[] files = dir.listFiles();
		
		//ファイル名のソートとフィルタリング
		TreeMap<String, File> tree = new TreeMap<String, File>();
		for (int i = 0; i < files.length; i++) {
			if (!files[i].getName().endsWith(m_fileSuffix)) {
				continue;
			}
			tree.put(files[i].getName(), files[i]);
		}
		return tree.values();
		
	}
	
}
