/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.dialog;

import java.util.Date;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityInfo;
import com.clustercontrol.bean.FacilityPath;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.calendar.composite.CalendarIdListComposite;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ScopeTreeDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.jobmanagement.common.composite.JobIdSelectionListener;
import com.clustercontrol.notify.composite.NotifyInfoComposite;
import com.clustercontrol.syslogng.action.LogManager;
import com.clustercontrol.syslogng.bean.LogFilterInfo;
import com.clustercontrol.util.Messages;

/**
 * syslog-ng[フィルタの作成・変更]ダイアログクラスです。
 * 
 * @version 2.2.0
 * @since 1.0.0
 */
public class LogCreateDialog extends CommonDialog {

    /** 入力値を保持するオブジェクト。 */
    private LogFilterInfo inputData = null;

    /** 入力値の正当性を保持するオブジェクト。 */
    private ValidateResult validateResult = null;

    /** 変更対象のフィルタID。 */
    private String logId = null;

    /** 変更対象の順序。 */
    private int order = Integer.MAX_VALUE;

    /** 選択されたスコープのファシリティID。 */
    private String facilityId = null;

    /** 説明。 */
    private Text textDescription = null;

    /** スコープ テキストボックス。 */
    private Text textScope = null;

    /** パターンマッチ表現 テキストボックス。 */
    private Text textPattern = null;

    /** 一致したら処理しない ラジオボタン。 */
    private Button radioNotProcessing = null;

    /** 一致したら処理する ラジオボタン。*/
    private Button radioProcessing = null;
    
    /** カレンダID一覧 コンポジット。 */
    private CalendarIdListComposite calendarId = null;
    
    /** 通知ID一覧 コンポジット。 */
    private NotifyInfoComposite notifyInfo = null;

    /** 重要度 ラベル。 */
    private Label labelPriority = null;
    /** 
     * 重要度 コンボボックス。
     * @see com.clustercontrol.bean.PriorityConstant
     */
    private Combo comboPriority = null;

    /** 監視項目ID ラベル。 */
    private Label labelMonitorId = null;
    /** 監視項目ID テキストボックス。 */
    private Text textMonitorId = null;

    /** メッセージID ラベル。 */
    private Label labelMessageId = null;
    /** メッセージID テキストボックス。 */
    private Text textMessageId = null;

    /** メッセージ ラベル。 */
    private Label labelLogMessage = null;
    /** メッセージ テキストボックス。 */
    private Text textLogMessage = null;
    
    /**
     * ジョブ実行 チェックボックス。
     * @see com.clustercontrol.bean.YesNoConstant
     */
    private Button m_checkJobRun = null;
    
    /** ジョブ実行のジョブID テキストボックス。 */
	private Text m_textJobId = null;
	
	/** ジョブ実行の参照 ボタン。 */
	private Button m_buttonRefer = null;
	
	/**
	 * ジョブ実行の通知抑制と連動 チェックボックス。
	 * @see com.clustercontrol.bean.YesNoConstant
	 */ 
	private Button m_checkInhibitionFlg = null;
	
	/**
	 * ジョブ実行の呼出失敗時重要度 コンボボックス。
	 * @see com.clustercontrol.bean.PriorityConstant
	 */ 
	private Combo m_comboFailurePriority = null;

    /**
     * この設定を有効にする チェックボックス。
     * @see com.clustercontrol.bean.ValidConstant
     */
    private Button confirmValid = null;
    
    /** 作成日時。 */
    private Date regDate = null;

    /** 変更フラグ */
    private boolean modify = false;

    /**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
    public LogCreateDialog(Shell parent) {
        super(parent);
    }

    /**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param logId 変更するフィルタ情報のフィルタID
	 */
    public LogCreateDialog(Shell parent, String logId) {
        super(parent);

        this.logId = logId;
    }

    /**
     * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.syslogng.action.LogManager#get(String)
	 * @see #setInputData(LogFilterInfo)
     */
    protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.syslog-ng.create.modefy"));

        // 変数として利用されるラベル
        Label label = null;
        // 変数として利用されるグリッドデータ
        GridData gridData = null;

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        parent.setLayout(layout);
        
        /*
         * 監視項目ID
         */
        // ラベル
        this.labelMonitorId = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 4;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.labelMonitorId.setLayoutData(gridData);
        this.labelMonitorId.setText(Messages.getString("monitor.id") + " : ");
        // テキスト
        this.textMonitorId = new Text(parent, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = 9;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textMonitorId.setLayoutData(gridData);
        this.textMonitorId.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_64));

        /*
         * 説明
         */
        // ラベル
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 4;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("description") + " : ");
        // テキスト
        this.textDescription = new Text(parent, SWT.BORDER | SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalSpan = 9;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textDescription.setLayoutData(gridData);
        this.textDescription.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_256));
        
        /*
         * 監視条件グループ
         */
        // ラベル
        this.labelMonitorId = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.labelMonitorId.setLayoutData(gridData);
        this.labelMonitorId.setText(Messages.getString("monitor.rule") + " : ");

        // グループ
        Group monitorRuleGroup1 = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        monitorRuleGroup1.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        monitorRuleGroup1.setLayoutData(gridData);
        
        /*
         * カレンダ
         */
		this.calendarId = new CalendarIdListComposite(monitorRuleGroup1, SWT.NONE, true);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		calendarId.setLayoutData(gridData);
		
		/*
         * チェック条件グループ
         */
		//ラベル
        label = new Label(monitorRuleGroup1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("check.rule") + " : ");
        
        // グループ
        Group groupCheckRule = new Group(monitorRuleGroup1, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 15;
        groupCheckRule.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupCheckRule.setLayoutData(gridData);

        /*
         * スコープ
         */
        // ラベル
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 5;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("scope") + " : ");
        // テキスト
        this.textScope = new Text(groupCheckRule, SWT.BORDER | SWT.LEFT | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalSpan = 8;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textScope.setLayoutData(gridData);
        // 参照ボタン
        Button button = new Button(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);
        button.setText(Messages.getString("refer"));
        button.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                ScopeTreeDialog dialog = new ScopeTreeDialog(getParentShell(),false,true);
                if (dialog.open() == IDialogConstants.OK_ID) {
                    FacilityTreeItem item = dialog.getSelectItem();
                    FacilityInfo info = item.getData();
                    facilityId = info.getFacilityId();
                    if (info.getType() == FacilityConstant.TYPE_NODE) {
                        textScope.setText(info.getFacilityName());
                    } else {
                        FacilityPath path = new FacilityPath(
                                ClusterControlPlugin.getDefault()
                                        .getSeparator());
                        textScope.setText(path.getPath(item));
                    }
                }
            }
        });

        /*
         * パターンマッチ表現
         */
        // ラベル
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 5;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("pattern.matching.expression") + " : ");
        // テキスト
        this.textPattern = new Text(groupCheckRule, SWT.BORDER | SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalSpan = 10;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textPattern.setLayoutData(gridData);
        this.textPattern.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_1024));

        /*
         * 出力の条件
         */
        /*
         * 処理しない
         */
        this.radioNotProcessing = new Button(parent, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.radioNotProcessing.setLayoutData(gridData);
        this.radioNotProcessing.setText(Messages
                .getString("don't.process.if.matched"));

        /*
         * 処理する
         */
        this.radioProcessing = new Button(parent, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.radioProcessing.setLayoutData(gridData);
        this.radioProcessing.setText(Messages.getString("process.if.matched"));
        this.radioProcessing.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                setEnabledForOutputCondition(radioProcessing.getSelection());
            }
        });

        /*
         * グループ
         */
        // 大グループ
        Group group1 = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        group1.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        group1.setLayoutData(gridData);
        
        /*
         * 通知ID
         * アプリケーションID
         */
		// 通知情報
		this.notifyInfo = new NotifyInfoComposite(group1, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		notifyInfo.setLayoutData(gridData);

        /*
         * 重要度
         */
        // ラベル
        this.labelPriority = new Label(group1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 5;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.labelPriority.setLayoutData(gridData);
        this.labelPriority.setText(Messages.getString("priority") + " : ");
        // コンボボックス
        this.comboPriority = new Combo(group1, SWT.DROP_DOWN | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalSpan = 3;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.comboPriority.setLayoutData(gridData);
        this.comboPriority.add(PriorityConstant.STRING_CRITICAL);
        this.comboPriority.add(PriorityConstant.STRING_WARNING);
        this.comboPriority.add(PriorityConstant.STRING_INFO);
        this.comboPriority.add(PriorityConstant.STRING_UNKNOWN);
        this.comboPriority.setText(PriorityConstant.STRING_CRITICAL);
        // 空白
        label = new Label(group1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 7;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);

        /*
         * メッセージID
         */
        // ラベル
        this.labelMessageId = new Label(group1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 5;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.labelMessageId.setLayoutData(gridData);
        this.labelMessageId.setText(Messages.getString("message.id") + " : ");
        // テキスト
        this.textMessageId = new Text(group1, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = 10;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textMessageId.setLayoutData(gridData);
        this.textMessageId.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_64));

        /*
         * ログメッセージ
         */
        // ラベル
        this.labelLogMessage = new Label(group1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 5;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.labelLogMessage.setLayoutData(gridData);
        this.labelLogMessage.setText(Messages.getString("message") + " : ");
        // テキスト
        this.textLogMessage = new Text(group1, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = 10;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textLogMessage.setLayoutData(gridData);
        this.textLogMessage.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_256));
        
        // ジョブ実行
        this.m_checkJobRun = new Button(group1, SWT.CHECK);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        m_checkJobRun.setLayoutData(gridData);
        m_checkJobRun.setText(Messages.getString("run") + " : ");
        m_checkJobRun.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	m_textJobId.setEnabled(m_checkJobRun.getSelection());
            	m_buttonRefer.setEnabled(m_checkJobRun.getSelection());
            	m_checkInhibitionFlg.setEnabled(m_checkJobRun.getSelection());
            	m_comboFailurePriority.setEnabled(m_checkJobRun.getSelection());
            }
        });
		
		// ジョブ実行グループ
        Group groupRunJob = new Group(group1, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 15;
        groupRunJob.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupRunJob.setLayoutData(gridData);
        
        // ジョブID
		label = new Label(groupRunJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 6;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("job.id"));
		
		// 空白
        label = new Label(groupRunJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 1;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
		
		// 通知抑制と連動
		label = new Label(groupRunJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("couple.with.notify.inhibition"));
		
		// 空白
        label = new Label(groupRunJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 1;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
		
		// 呼出失敗時
		label = new Label(groupRunJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("failure.call.value"));
		
		// 空白
        label = new Label(groupRunJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 1;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
		
		/*
         * 実行ジョブ
         */
        this.m_textJobId = new Text(groupRunJob, SWT.BORDER | SWT.LEFT | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 4;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textJobId.setLayoutData(gridData);
		
		this.m_buttonRefer = new Button(groupRunJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_buttonRefer.setLayoutData(gridData);
		this.m_buttonRefer.setText(Messages.getString("refer"));
		this.m_buttonRefer.addSelectionListener(new JobIdSelectionListener(m_textJobId));
		
		// 空白
        label = new Label(groupRunJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 1;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
		
        // 通知抑制と連動
        this.m_checkInhibitionFlg = new Button(groupRunJob, SWT.CHECK);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.CENTER;
		gridData.grabExcessHorizontalSpace = true;
		this.m_checkInhibitionFlg.setLayoutData(gridData);
		
		// 空白
        label = new Label(groupRunJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 1;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        // 呼出失敗時
        this.m_comboFailurePriority = new Combo(groupRunJob, SWT.DROP_DOWN | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_comboFailurePriority.setLayoutData(gridData);
		this.m_comboFailurePriority.add(PriorityConstant.STRING_CRITICAL);
		this.m_comboFailurePriority.add(PriorityConstant.STRING_WARNING);
		this.m_comboFailurePriority.add(PriorityConstant.STRING_INFO);
		this.m_comboFailurePriority.add(PriorityConstant.STRING_UNKNOWN);
		this.m_comboFailurePriority.setText(PriorityConstant.STRING_UNKNOWN);
		
		// 空白
        label = new Label(groupRunJob, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        /*
         * 有効／無効
         */
        this.confirmValid = new Button(parent, SWT.CHECK);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.confirmValid.setLayoutData(gridData);
        this.confirmValid
                .setText(Messages.getString("setting.valid.confirmed"));

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 15;
        line.setLayoutData(gridData);

        // サイズを最適化
        // グリッドレイアウトを用いた場合、こうしないと横幅が画面いっぱいになります。
        shell.pack();
        shell.setSize(new Point(550, shell.getSize().y));

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);

        // ログIDが指定されている場合、その情報を初期表示する。
        LogFilterInfo log = LogManager.getInstance().get(this.logId);
        if (log == null) {
            log = new LogFilterInfo();
            modify = false;
        }
        else{
        	modify = true;
        }
        this.setInputData(log);
    }

    /**
	 * 入力値を保持したフィルタ情報を返します。
	 * 
	 * @return フィルタ情報
	 */
    public LogFilterInfo getInputData() {
        return this.inputData;
    }

    /**
	 * 引数で指定されたフィルタ情報の値を、各項目に設定します。
	 * 
	 * @param log 設定値として用いるフィルタ情報
	 * 
     * @since 1.0.0
	 * 
	 * @see #setEnabledForOutputCondition(boolean)
     */
    protected void setInputData(LogFilterInfo log) {

        this.inputData = log;

        // 各項目に反映
        this.logId = log.getLogId();
        this.order = log.getOrderNo();
        this.facilityId = log.getFacilityId();

        if (log.getMonitorId() != null) {
            this.textMonitorId.setText(log.getMonitorId());
        }
        if (log.getDescription() != null) {
            this.textDescription.setText(log.getDescription());
        }
        if (log.getCalendarId() != null) {
            this.calendarId.setText(log.getCalendarId());
        }
        if (log.getScope() != null) {
            this.textScope.setText(log.getScope());
        }
        if (log.getPattern() != null) {
            this.textPattern.setText(log.getPattern());
        }
        if (log.getProcessType() == ProcessConstant.TYPE_YES) {
            this.radioProcessing.setSelection(true);
        } else {
            this.radioNotProcessing.setSelection(true);
        }
        if (log.getNotifyId() != null) {
            this.notifyInfo.setText(log.getNotifyId());
        }
        this.comboPriority.setText(PriorityConstant.typeToString(log.getPriority()));
        if (log.getApplication() != null) {
            this.notifyInfo.setApplication(log.getApplication());
        }
        if (log.getMessageId() != null) {
            this.textMessageId.setText(log.getMessageId());
        }
        if (log.getMessage() != null) {
            this.textLogMessage.setText(log.getMessage());
        }
        
        // ジョブ実行
        this.m_checkJobRun.setSelection(YesNoConstant.typeToBoolean(log.getJobRun()));
        if(log.getJobId() != null){
			this.m_textJobId.setText(log.getJobId());
		}
		this.m_checkInhibitionFlg.setSelection(YesNoConstant.typeToBoolean(log.getJobInhibitionFlg()));
		this.m_comboFailurePriority.setText(PriorityConstant.typeToString(log.getJobFailurePriority()));

        if (log.isValidFlg() == ValidConstant.BOOLEAN_VALID) {
            this.confirmValid.setSelection(true);
        }
        
        this.regDate = log.getRegDate();

        // 入力制御
        this.setEnabledForOutputCondition(this.radioProcessing.getSelection());
    }

    /**
	 * 入力値を設定したフィルタ情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return フィルタ情報
	 * 
	 * @see #setValidateResult(String, String)
	 */
    protected LogFilterInfo createInputData() {
        LogFilterInfo log = new LogFilterInfo();

        log.setLogId(this.logId);
        log.setOrderNo(this.order);
        log.setFacilityId(this.facilityId);

        if (this.textMonitorId.getText() != null
                && !"".equals((this.textMonitorId.getText()).trim())) {
            log.setMonitorId(this.textMonitorId.getText());
        } else {
            if (this.radioProcessing.getSelection()) {
                this.setValidateResult(Messages.getString("message.hinemos.1"),
                        Messages.getString("message.monitor.1"));
                return null;
            }
        }
        if (this.textDescription.getText() != null
                && !"".equals((this.textDescription.getText()).trim())) {
            log.setDescription(this.textDescription.getText());
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),
                    Messages.getString("message.monitor.26"));
            return null;
        }
        
        // カレンダID
        String calendarId = this.calendarId.getText();
        if (calendarId != null && !"".equals(calendarId.trim())) {
        	log.setCalendarId(calendarId);
        }

        if (this.textScope.getText() != null
                && !"".equals((this.textScope.getText()).trim())) {
            log.setScope(this.textScope.getText());
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),
                    Messages.getString("message.hinemos.3"));
            return null;
        }

        if (this.textPattern.getText() != null
                && !"".equals((this.textPattern.getText()).trim())) {
        	
        	try{
    			Pattern.compile(this.textPattern.getText());
    			log.setPattern(this.textPattern.getText());
    		}
    		catch(PatternSyntaxException e){
    			this.setValidateResult(Messages.getString("message.hinemos.1"),
                        Messages.getString("message.monitor.27"));
                return null;
    		}
            
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),
                    Messages.getString("message.monitor.27"));
            return null;
        }

        if (this.radioProcessing.getSelection()) {
            log.setProcessType(ProcessConstant.TYPE_YES);
        } else {
            log.setProcessType(ProcessConstant.TYPE_NO);
        }
        String priorityText = this.comboPriority.getText();
        log.setPriority(PriorityConstant.stringToType(priorityText));
        
        // 通知ID
        String notifyId = this.notifyInfo.getText();
        if (notifyId != null && !"".equals(notifyId.trim())) {
        	log.setNotifyId(notifyId);
        }
        
        if (this.notifyInfo.getApplication() != null
                && !"".equals((this.notifyInfo.getApplication()).trim())) {
            log.setApplication(this.notifyInfo.getApplication());
        } else {
            if (this.radioProcessing.getSelection()) {
            	if (notifyId != null && !"".equals(notifyId.trim())) {
            		this.setValidateResult(Messages.getString("message.hinemos.1"),
                            Messages.getString("message.monitor.10"));
                    return null;
            	}
            }
        }

        if (this.textMessageId.getText() != null
                && !"".equals((this.textMessageId.getText()).trim())) {
            log.setMessageId(this.textMessageId.getText());
        } else {
            if (this.radioProcessing.getSelection()) {
            	if (notifyId != null && !"".equals(notifyId.trim())) {
            		this.setValidateResult(Messages.getString("message.hinemos.1"),
                            Messages.getString("message.monitor.28"));
                    return null;	
            	}
            }
        }
        if (this.textLogMessage.getText() != null
                && !"".equals((this.textLogMessage.getText()).trim())) {
            log.setMessage(this.textLogMessage.getText());
        } else {
            if (this.radioProcessing.getSelection()) {
            	if (notifyId != null && !"".equals(notifyId.trim())) {
            		this.setValidateResult(Messages.getString("message.hinemos.1"),
                            Messages.getString("message.monitor.29"));
                    return null;	
            	}
            }
        }
        
        // ジョブ実行
		log.setJobRun(Integer.valueOf(YesNoConstant.booleanToType(m_checkJobRun.getSelection())));
		// ジョブID
		if (m_textJobId.getText() != null && !"".equals(m_textJobId.getText().trim())) {
			log.setJobId(m_textJobId.getText());	
		}
		else{
			if(log.getJobRun() == YesNoConstant.TYPE_YES){
				String[] args = {""};
				this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.25", args));
				return null;
			}
		}
		// 通知抑制と連動
		log.setJobInhibitionFlg(Integer.valueOf(YesNoConstant.booleanToType(m_checkInhibitionFlg.getSelection())));
		// 呼出失敗時
		log.setJobFailurePriority(PriorityConstant.stringToType(m_comboFailurePriority.getText()));
        
        // 有効
        if (this.confirmValid.getSelection()) {
            log.setValidFlg(ValidConstant.BOOLEAN_VALID);
        } else {
            log.setValidFlg(ValidConstant.BOOLEAN_INVALID);
        }
        
        // 登録日時
        log.setRegDate(this.regDate);
        
        return log;
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    protected ValidateResult validate() {
        // 入力値生成
        this.inputData = this.createInputData();

        if (this.inputData != null) {
            return super.validate();
        } else {
            return validateResult;
        }
    }
    
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    protected boolean action() {
    	boolean result = false;
    	
    	LogFilterInfo info = this.inputData;
    	if(info != null){
            if(!this.modify){
            	// 作成の場合
            	result = LogManager.getInstance().add(info);
            }
            else{
            	// 変更の場合
            	result = LogManager.getInstance().modify(info);
            }
    	}
    	
    	return result;
    }

    /**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    protected String getOkButtonText() {
        return Messages.getString("ok");
    }

    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }

    /**
	 * 処理情報のコンポジットの選択可/不可を設定します。
	 *
	 * @param enable 選択可の場合、<code> true </code>
	 */
    private void setEnabledForOutputCondition(boolean enable) {

    	/*
        this.labelPriority.setEnabled(enable);
        this.labelMessageId.setEnabled(enable);
        this.labelLogMessage.setEnabled(enable);
        */
        this.comboPriority.setEnabled(enable);
        this.notifyInfo.setEnabled(enable);
        this.textMessageId.setEnabled(enable);
        this.textLogMessage.setEnabled(enable);
        this.m_checkJobRun.setEnabled(enable);
        this.m_textJobId.setEnabled(enable);
        this.m_buttonRefer.setEnabled(enable);
        this.m_checkInhibitionFlg.setEnabled(enable);
        this.m_comboFailurePriority.setEnabled(enable);
    }
    
    /**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 */
    private void setValidateResult(String id, String message) {

        this.validateResult = new ValidateResult();
        this.validateResult.setValid(false);
        this.validateResult.setID(id);
        this.validateResult.setMessage(message);
    }
}