/*
                                                                                                
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.failover.monitor.jmx;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Properties;
import java.util.Vector;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.naming.CompositeName;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NamingException;

import org.jboss.ha.framework.interfaces.HAPartition;
import org.jboss.jmx.adaptor.rmi.RMIAdaptor;
import org.jboss.naming.NonSerializableFactory;
import org.jboss.system.ServiceMBeanSupport;

/**
 * @jmx.mbean
 *     name="user:service=FailoverCheckService,name=FailoverCheckService"
 *     description="FailoverCheckService MBean"
 *     extends="org.jboss.system.ServiceMBean"
 * 
 * @jboss.service servicefile="FailoverCheck"
 */
public class FailoverCheckService extends ServiceMBeanSupport 
    implements FailoverCheckServiceMBean {
	
	private String jndiName = "FailoverCheckService";  // JNDIに登録する際の名前
	
	private static final String LOOKUP_NAME = "jmx/invoker/RMIAdaptor";
	private static final String OBJECT_NAME = "jboss:service=";
	private static final String GET_NODE_NAME = "getNodeName";
	private static final String GET_CURRENT_VIEW = "getCurrentView";
	
	private String partitionName = "HinemosPartition";
	private HAPartition partition;
	private ArrayList<String> members = new ArrayList<String>();

	// フェイルオーバ後にHinemosの各機能を実行するか否かのフラグ
	// true になるまで起動は待機される
	boolean failoverExecCheck = false;
	
	public FailoverCheckService(){
		failoverExecCheck = false;
	}
	
	/**
	 * @jmx.managed-attribute
	 * 
	 * @see com.clustercontrol.failover.monitor.jmx.FailoverCheckServiceMBean#getPartitionName()
	 */
	public String getPartitionName() {
		return partitionName;
	}


	/**
	 * @jmx.managed-attribute
	 * 
	 * @see com.clustercontrol.failover.monitor.jmx.FailoverCheckServiceMBean#setPartitionName(java.lang.String)
	 */
	public void setPartitionName(String name) {
		partitionName = name;
	}
	
    /**
     * @jmx.managed-attribute
     * 
     * @see com.clustercontrol.failover.monitor.jmx.FailoverCheckServiceMBean#getJndiName()
     */
    public String getJndiName() {
        return jndiName;
    }
    
    /**
     * @jmx.managed-attribute
     * 
     * @see com.clustercontrol.failover.monitor.jmx.FailoverCheckServiceMBean#setJndiName(java.lang.String)
     * 
     * @param jndiName
     * @throws NamingException 
     */
    public void setJndiName(String jndiName) throws NamingException {
        log.info("setJndiName() : jndiName = " + jndiName);
    	
        String oldName = this.jndiName;
        this.jndiName = jndiName;
        
        if (super.getState() == STARTED) {
        	try {
        		unbind(oldName);
        	} catch (NamingException e){
        		NamingException ne = new NamingException("Failed to unbind");
        		ne.setRootCause(e);
        		log.error(ne);
        	}

        	try{
        		rebind();
        	} catch (NamingException e){
        		NamingException ne = new NamingException("Failed to rebind");
        		ne.setRootCause(e);
        		log.error(ne);
        	}
        }
    }
	
    /**
     * @jmx.managed-attribute
     * 
     * @see org.jboss.system.ServiceMBean#getName()
     */
    public String getName() {
        return "FailoverCheckService(" + jndiName + ")";
    }
    
    /* (non-Javadoc)
     * @see org.jboss.system.ServiceMBeanSupport#startService()
     */
    public void startService() throws NamingException {
        log.info("Start FailoverCheckService(" + jndiName + ")");
    	
        InitialContext ctx = new InitialContext();
        String partitionJndiName = "/HAPartition/" + partitionName;
        partition = (HAPartition) ctx.lookup(partitionJndiName);
        
        // 自身のIPアドレスを取得する
        String bindAddress = getNodeName();
        
		// 自身のIPアドレスをチェック
//		String bindAddress = System.getProperty("jboss.bind.address");
//	
//		try {
//			if(bindAddress == null || "0.0.0.0".equals(bindAddress)){
//				bindAddress = InetAddress.getLocalHost().getHostAddress();
//			}
//		} catch (UnknownHostException e) {
//			log.warn("Failed to lookup local hostname", e);
//			log.error("hostname : <unknown>");
//			throw new NamingException();	
//		}
        
        //全メンバリスト取得
        getAllMembers();
        
        //　プロパティファイルから設定を読み込む
	    Properties properties = new Properties();
		String homedir = System.getProperty("jboss.server.home.dir");
		String propertyFile = homedir + File.separator + "conf" + File.separator + "trouble-detection.properties";
		
		Iterator<String> itr = members.iterator();
		
		String oldMember = null;
		while(itr.hasNext()){
			if(oldMember == null){
				oldMember = itr.next();
			} else {
				oldMember = oldMember + ", " + itr.next();
			}
		}
		log.info("member : " + oldMember);
		
		// member の数の整合性をチェックする。
		// マスタサーバからスレーブサーバへフェイルオーバされたとき、 member の台数が２となる場合がある。
		if(members.size() < 1){
			log.error("Failed in getting member info!");
			throw new NamingException();
		} else if (members.size() > 2){
			log.error("Too match server!!");
			throw new NamingException();
		}
		
		// プロパティファイルからキーと値のリストを読み込みます
		try {
			properties.load(new FileInputStream(propertyFile));
		} catch (FileNotFoundException e) {
			log.info(e);
			throw new NamingException();	
		} catch (IOException e) {
			log.info(e);
			throw new NamingException();	
		}

		String jbossServer1 = properties.getProperty("jboss.server1");
		String jbossServer2 = properties.getProperty("jboss.server2");
		String command = properties.getProperty("jboss.failover.command");
		String backend1 = properties.getProperty("server1.backend.ip");
		String backend2 = properties.getProperty("server2.backend.ip");
		int commandRetry = Integer.parseInt(properties.getProperty("jboss.failover.command.retry"));
		
		log.info("bind address : " + bindAddress);

		// 自身がスレーブサーバの場合は、マスタサーバからスレーブサーバへのフェイルオーバと判断
		// コマンド実行対象をマスタサーバとする
		if (bindAddress.equals(jbossServer2)) {
			for(int i=0; i<commandRetry; i++){
				// 外部コマンドを実行
				String[] cmd = { command, backend1 };
				FailoverMonitor monitor = new FailoverMonitor();
				failoverExecCheck = monitor.callExternalCommand(cmd);
			}
		} else
		// 自身がマスタサーバの場合は、スレーブサーバからマスタサーバへのフェイルバックと判断
		// コマンド実行対象をスレーブサーバとする
		if (bindAddress.equals(jbossServer1)) {
			for(int i=0; i<commandRetry; i++){
				// 外部コマンドを実行
				String[] cmd = { command, backend2 };
				FailoverMonitor monitor = new FailoverMonitor();
				failoverExecCheck = monitor.callExternalCommand(cmd);
			}
		} else {
			log.error("This node(" + bindAddress
					+ ") is not MsaterServer(" + jbossServer1
					+ ") or SlaveServer(" + jbossServer2
					+ ") in " + propertyFile);
			throw new NamingException();
		}

		rebind();
		
		if (!failoverExecCheck) {
			log.error("Stopping Hinemos Startup.");
		}

		// 外部コマンドの実行の結果戻り値が0でない場合はサービスの起動を待機する
		while (!failoverExecCheck) {
			try {
				Thread.sleep(1000);
			} catch (InterruptedException e) {
				throw new NamingException();
			}
		}
    	
        log.info("Started FailoverCheckService(" + jndiName + ")");
    }
    
    /**
     * @jmx.managed-operation
     */
    public void continueStartup(){
   		failoverExecCheck = true;
    }
    
    /* (non-Javadoc)
     * @see org.jboss.system.ServiceMBeanSupport#stopService()
     */
    public void stopService() throws NamingException {
        log.info("Stop FailoverCheckService(" + jndiName + ")");

    	unbind(jndiName);

        log.info("Stoped FailoverCheckService(" + jndiName + ")");
    }

    private static Context createContext(Context rootCtx, Name name) throws NamingException {
    	Context subctx = rootCtx;
    	
    	for (int n = 0; n < name.size(); n++) {
    		String atom = name.get(n);
    		
    		try {
    			Object obj = subctx.lookup(atom);
    			subctx = (Context) obj;
    		} catch (NamingException e) {
    			// 存在しない場合は、サブコンテキストを生成
    			subctx = subctx.createSubcontext(atom);
    		}
    	}
    	
    	return subctx;
    }
    
    private void rebind() throws NamingException {
    	InitialContext rootCtx = new InitialContext();
    	
    	Name fullName = rootCtx.getNameParser("").parse(jndiName);
    	Name parentName = fullName;
    	if(fullName.size() > 1){
    		parentName = fullName.getPrefix(fullName.size()-1);
    	} else {
    		parentName = new CompositeName();
    	}
    	
    	Context parentCtx = createContext(rootCtx, parentName);
    	Name atomName = fullName.getSuffix(fullName.size()-1);
    	String atomStirng = atomName.get(0);
    	
    	NonSerializableFactory.rebind(parentCtx, atomStirng, this);
    }
    
    private void unbind(String jndiName) throws NamingException {
    	InitialContext rootCtx = null;
    	
    	try {
    		rootCtx = new InitialContext();
    		
    		Name fullName = rootCtx.getNameParser("").parse(jndiName);
    		Name atomName = fullName.getSuffix(fullName.size() - 1);
    		String atom = atomName.get(0);
    		
    		rootCtx.unbind(jndiName);
    		NonSerializableFactory.unbind(atom);
    	} finally {
    		if(rootCtx != null) { 
    			rootCtx.close(); 
    		}
    	}
    }
	
	/**
	 * 自身のノード名を取得する
	 * 
	 */
	private String getNodeName() {
		log.debug("getAllMembers() start :");
		
		String nodeName = null;
		try{
	        InitialContext ic = new InitialContext();
	        
	        //RMIAdaptorを取得
	        RMIAdaptor server = (RMIAdaptor) ic.lookup(LOOKUP_NAME);

	        //ObjectNameを設定
	        ObjectName name = new ObjectName(OBJECT_NAME + partitionName);
	        
	        //ObjectNameのOperationNameのメソッドを実行
	        Object returnObject = server.invoke(name, GET_NODE_NAME, null, null);
	        
	        //結果からJBossのパーティションへの参加をチェックする
	        if(returnObject instanceof String){
	        	nodeName = ((String)returnObject).split(":")[0];
	        }
		} catch (NamingException e) {
			log.error("getAllMembers() : " + e.getMessage());
		} catch (MalformedObjectNameException e) {
			log.error("getAllMembers() : " + e.getMessage());
		} catch (NullPointerException e) {
			log.error("getAllMembers() : " + e.getMessage());
		} catch (InstanceNotFoundException e) {
			log.error("getAllMembers() : " + e.getMessage());
		} catch (MBeanException e) {
			log.error("getAllMembers() : " + e.getMessage());
		} catch (ReflectionException e) {
			log.error("getAllMembers() : " + e.getMessage());
		} catch (IOException e) {
			log.error("getAllMembers() : " + e.getMessage());
		}
        
		log.debug("getAllMembers() end :");
		
		return nodeName;
	}
    
	/**
	 * 全メンバリスト取得
	 * 
	 */
	private void getAllMembers() {
		log.debug("getAllMembers() start :");
		
		try{
			//全メンバリストをクリア
			members.clear();
			
	        InitialContext ic = new InitialContext();
	        
	        //RMIAdaptorを取得
	        RMIAdaptor server = (RMIAdaptor) ic.lookup(LOOKUP_NAME);

	        //ObjectNameを設定
	        ObjectName name = new ObjectName(OBJECT_NAME + partitionName);
	        
	        //ObjectNameのOperationNameのメソッドを実行
	        Object returnObject = server.invoke(name, GET_CURRENT_VIEW, null, null);
	        
	        //結果からJBossのパーティションへの参加をチェックする
	        if(returnObject instanceof Vector){
	        	Vector returnVector = (Vector)returnObject;
	        	
	        	for(int i = 0; i < returnVector.size(); i++){
	        		//host:portを取得
	        		String work = (String)returnVector.get(i);
	        		String node[] = work.split(":");

	        		members.add(node[0]);
	        	}
	        }
		} catch (NamingException e) {
			log.error("getAllMembers() : " + e.getMessage());
		} catch (MalformedObjectNameException e) {
			log.error("getAllMembers() : " + e.getMessage());
		} catch (NullPointerException e) {
			log.error("getAllMembers() : " + e.getMessage());
		} catch (InstanceNotFoundException e) {
			log.error("getAllMembers() : " + e.getMessage());
		} catch (MBeanException e) {
			log.error("getAllMembers() : " + e.getMessage());
		} catch (ReflectionException e) {
			log.error("getAllMembers() : " + e.getMessage());
		} catch (IOException e) {
			log.error("getAllMembers() : " + e.getMessage());
		}
        
		log.debug("getAllMembers() end :");
	}
}
